<?php
// admin/academic_sessions.php - Academic Sessions Management (FIXED FOR HEADER ERROR)

// --- CRITICAL SETUP: All necessary includes/sessions/DB connections must be here ---
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
// We need settings and connection for actions
include_once('../config/db.php');

$conn = getDBConnection();

$settings = [];
$result = $conn->query("SELECT setting_key, setting_value FROM settings");
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $settings[$row['setting_key']] = $row['setting_value'];
    }
}

// Minimal user role check
$user_role_id = $_SESSION['role_id'] ?? 1;

// Authorization Check: Only Admin can access this page
if ($user_role_id != 1) {
    // Redirection before any output
    header('Location: dashboard.php?err=Access Denied');
    exit;
}

$conn->close(); // Connection closed after fetching settings

$message = '';
$error = '';

// --- PHP ACTIONS WITH REDIRECTION (MUST BE AT THE TOP) ---

// 1. ADD / EDIT Session
if ($_SERVER['REQUEST_METHOD'] == 'POST' && (isset($_POST['add_session']) || isset($_POST['edit_session']))) {
    $conn = getDBConnection(); // Re-open connection for action
    $name = trim($_POST['name']);
    $start_date = $_POST['start_date'];
    $end_date = $_POST['end_date'];
    $is_edit = isset($_POST['edit_session']);
    $session_id = $is_edit ? (int)$_POST['session_id'] : 0;

    // Check session limit for new sessions only (not for edits)
    if (!$is_edit) {
        $count_result = $conn->query("SELECT COUNT(id) as total_sessions FROM academic_sessions");
        $total_sessions = $count_result->fetch_assoc()['total_sessions'];
        
        if ($total_sessions >= 5) {
            $error = "Error: Aap maximum 5 sessions hi add kar sakte hain. Apna plan upgrade karein.";
            $conn->close();
            header("Location: academic_sessions.php?msg=" . urlencode($message) . "&err=" . urlencode($error));
            exit;
        }
        
        // Check if any active session exists (ONLY sessions with future end date)
        $active_check = $conn->query("SELECT COUNT(id) as active_count FROM academic_sessions WHERE end_date >= CURDATE()");
        $active_sessions = $active_check->fetch_assoc()['active_count'];
        
        if ($active_sessions > 0) {
            $error = "Error: Ek active session already chal raha hai. Naya session tabhi add kar sakte hain jab current session complete ho jaye.";
            $conn->close();
            header("Location: academic_sessions.php?msg=" . urlencode($message) . "&err=" . urlencode($error));
            exit;
        }
    }

    if (empty($name) || empty($start_date) || empty($end_date)) {
        $error = "Error: Saari fields bharna zaroori hai.";
    } elseif ($is_edit) {
        // UPDATE
        $stmt = $conn->prepare("UPDATE academic_sessions SET name=?, start_date=?, end_date=? WHERE id=?");
        $stmt->bind_param("sssi", $name, $start_date, $end_date, $session_id);
        if ($stmt->execute()) {
            $message = "Academic Session update ho gaya hai.";
        } else {
            $error = "Error updating session: " . $conn->error;
        }
        $stmt->close();
    } else {
        // INSERT - Naya session add karo
        $stmt = $conn->prepare("INSERT INTO academic_sessions (name, start_date, end_date) VALUES (?, ?, ?)");
        $stmt->bind_param("sss", $name, $start_date, $end_date);
        if ($stmt->execute()) {
            $message = "Naya Academic Session add ho gaya hai.";
        } else {
            $error = "Error adding session (Duplicate name ya koi aur masla): " . $conn->error;
        }
        $stmt->close();
    }
    $conn->close();
    header("Location: academic_sessions.php?msg=" . urlencode($message) . "&err=" . urlencode($error));
    exit;
}

// 2. DELETE Session
if (isset($_GET['action']) && $_GET['action'] == 'delete' && isset($_GET['id'])) {
    $conn = getDBConnection(); // Re-open connection for action
    $session_id = (int)$_GET['id'];

    // Check if session is active (end date baaki hai) - FIXED: Use >= CURDATE() instead of date comparison
    $check_active = $conn->prepare("SELECT end_date FROM academic_sessions WHERE id = ?");
    $check_active->bind_param("i", $session_id);
    $check_active->execute();
    $session_data = $check_active->get_result()->fetch_assoc();
    $check_active->close();
    
    // FIXED: Session should be considered active if end_date is today or in future
    $is_active = strtotime($session_data['end_date']) >= strtotime(date('Y-m-d'));
    
    if ($is_active) {
        $error = "Error: Active session ko delete nahi kiya ja sakta.";
    } else {
        $check_stmt = $conn->prepare("SELECT COUNT(id) AS count FROM student_enrollments WHERE session_id = ?");
        $check_stmt->bind_param("i", $session_id);
        $check_stmt->execute();
        $student_count = $check_stmt->get_result()->fetch_assoc()['count'];
        $check_stmt->close();
        
        if ($student_count > 0) {
            $error = "Error: Is session se {$student_count} students enrolled hain. Pehle unko migrate karein.";
        } else {
            $stmt = $conn->prepare("DELETE FROM academic_sessions WHERE id = ?");
            $stmt->bind_param("i", $session_id);
            if ($stmt->execute()) {
                $message = "Academic Session delete ho gaya hai.";
            } else {
                $error = "Error deleting session: " . $conn->error;
            }
            $stmt->close();
        }
    }
    $conn->close();
    header("Location: academic_sessions.php?msg=" . urlencode($message) . "&err=" . urlencode($error));
    exit;
}

// --- PHP Setup for DISPLAY (Now that all actions/redirects are handled) ---

// Header aur Sidebar files include karein. (Ab yeh safely include ho sakte hain)
include_once('includes/header.php'); 

// 3. EDIT Load (No redirection, so it runs after header include)
$edit_session = null;
if (isset($_GET['action']) && $_GET['action'] == 'edit' && isset($_GET['id'])) {
    $conn = getDBConnection();
    $session_id = (int)$_GET['id'];
    $stmt = $conn->prepare("SELECT id, name, start_date, end_date FROM academic_sessions WHERE id = ?");
    $stmt->bind_param("i", $session_id);
    $stmt->execute();
    $edit_session = $stmt->get_result()->fetch_assoc();
    $stmt->close();
    $conn->close();
}

// --- Data Fetching (Final List for Display) ---
$conn = getDBConnection();

$sessions_result = $conn->query("SELECT id, name, start_date, end_date FROM academic_sessions ORDER BY start_date DESC");
$sessions = $sessions_result->fetch_all(MYSQLI_ASSOC);

// Get total sessions count for display
$count_result = $conn->query("SELECT COUNT(id) as total_sessions FROM academic_sessions");
$total_sessions = $count_result->fetch_assoc()['total_sessions'];

// Check if any active session exists (ONLY sessions with FUTURE end date) - FIXED: Use >= CURDATE()
$active_check = $conn->query("SELECT COUNT(id) as active_count FROM academic_sessions WHERE end_date >= CURDATE()");
$active_sessions = $active_check->fetch_assoc()['active_count'];
$has_active_session = $active_sessions > 0;

$conn->close();
?>

<div class="container-fluid">
    <div class="row">
        <div class="col-lg-12">
            <h2 class="mb-4"><i class="fas fa-calendar-alt"></i> Academic Sessions Management</h2>
            <hr>
            
            <?php 
            // Messages are read from URL query strings (after successful action/redirect)
            $message = $_GET['msg'] ?? $message;
            $error = $_GET['err'] ?? $error;
            
            if ($message): ?>
                <div class="alert alert-success"><?php echo htmlspecialchars($message); ?></div>
            <?php endif; ?>
            <?php if ($error): ?>
                <div class="alert alert-danger"><?php echo htmlspecialchars($error); ?></div>
            <?php endif; ?>
            
            <!-- Session Limit Info -->
            <div class="alert alert-info">
                <strong>Session Limit:</strong> Aapke paas <?php echo $total_sessions; ?> / 5 sessions hain. 
                <?php if ($total_sessions >= 5): ?>
                    <strong class="text-danger">Maximum limit reached! Apna plan upgrade karein.</strong>
                <?php else: ?>
                    <?php echo (5 - $total_sessions); ?> sessions aur add kar sakte hain.
                <?php endif; ?>
                
                <?php if ($has_active_session): ?>
                    <div class="mt-2">
                        <strong class="text-warning">⚠️ Ek active session chal raha hai. Naya session tabhi add kar sakte hain jab current session complete ho jaye.</strong>
                    </div>
                <?php else: ?>
                    <div class="mt-2">
                        <strong class="text-success">✅ Koi active session nahi hai. Aap naya session add kar sakte hain.</strong>
                    </div>
                <?php endif; ?>
            </div>
            
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">
                        <?php echo $edit_session ? 'Edit Existing Session' : 'Add New Academic Session'; ?>
                    </h6>
                </div>
                <div class="card-body">
                    <form method="POST" action="academic_sessions.php">
                        <?php if ($edit_session): ?>
                            <input type="hidden" name="session_id" value="<?php echo htmlspecialchars($edit_session['id']); ?>">
                            <input type="hidden" name="edit_session" value="1">
                        <?php else: ?>
                            <input type="hidden" name="add_session" value="1">
                        <?php endif; ?>

                        <div class="form-row">
                            <div class="form-group col-md-4">
                                <label for="name">Session Name (e.g., 2024-2025)</label>
                                <input type="text" name="name" class="form-control" value="<?php echo htmlspecialchars($edit_session['name'] ?? ''); ?>" required>
                            </div>
                            <div class="form-group col-md-4">
                                <label for="start_date">Start Date</label>
                                <input type="date" name="start_date" class="form-control" value="<?php echo htmlspecialchars($edit_session['start_date'] ?? date('Y-m-d')); ?>" required>
                            </div>
                            <div class="form-group col-md-4">
                                <label for="end_date">End Date</label>
                                <input type="date" name="end_date" class="form-control" value="<?php echo htmlspecialchars($edit_session['end_date'] ?? date('Y-m-d', strtotime('+1 year -1 day'))); ?>" required>
                            </div>
                        </div>

                        <button type="submit" class="btn btn-<?php echo $edit_session ? 'warning' : 'primary'; ?> mt-3" 
                            <?php echo (!$edit_session && ($total_sessions >= 5 || $has_active_session)) ? 'disabled' : ''; ?>>
                            <i class="fas fa-<?php echo $edit_session ? 'pencil-alt' : 'plus'; ?>"></i> 
                            <?php echo $edit_session ? 'Update Session' : 'Add Session'; ?>
                        </button>
                        <?php if ($edit_session): ?>
                            <a href="academic_sessions.php" class="btn btn-secondary mt-3">Cancel Edit</a>
                        <?php endif; ?>
                    </form>
                </div>
            </div>
            
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">Existing Sessions (Total: <?php echo $total_sessions; ?>)</h6>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                    <table class="table table-bordered table-striped" id="dataTable" width="100%" cellspacing="0">
                        <thead class="thead-dark">
                            <tr>
                                <th>ID</th>
                                <th>Session Name</th>
                                <th>Start Date</th>
                                <th>End Date</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($sessions as $session): ?>
                                <?php 
                                    // FIXED: Session should be active if end_date is today or in future
                                    // Ab session end date ke poore din tak active rahega (11:59 PM tak)
                                    $is_active = strtotime($session['end_date']) >= strtotime(date('Y-m-d'));
                                    
                                    if ($is_active) {
                                        $status_badge = '<span class="badge badge-success">ACTIVE</span>';
                                    } else {
                                        $status_badge = '<span class="badge badge-info">COMPLETED</span>';
                                    }
                                ?>
                            <tr>
                                <td><?php echo htmlspecialchars($session['id']); ?></td>
                                <td><?php echo htmlspecialchars($session['name']); ?></td>
                                <td><?php echo htmlspecialchars($session['start_date']); ?></td>
                                <td><?php echo htmlspecialchars($session['end_date']); ?></td>
                                <td><?php echo $status_badge; ?></td>
                                <td>
                                    <!-- EDIT BUTTON - Available for ALL sessions -->
                                    <a href="academic_sessions.php?action=edit&id=<?php echo $session['id']; ?>" 
                                        class="btn btn-info btn-sm" title="Edit Session">
                                        <i class="fas fa-edit"></i>
                                    </a>
                                    
                                    <!-- DELETE BUTTON - Only for COMPLETED sessions -->
                                    <?php if (!$is_active): ?>
                                        <a href="academic_sessions.php?action=delete&id=<?php echo $session['id']; ?>" 
                                            class="btn btn-danger btn-sm" title="Delete Session"
                                            onclick="return confirm('Warning: Kya aap is session ko delete karna chahte hain? Students data ka masla ho sakta hai.');">
                                            <i class="fas fa-trash"></i>
                                        </a>
                                    <?php endif; ?>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                            <?php if (empty($sessions)): ?>
                                <tr><td colspan="6" class="text-center">Koi Academic Session nahi mila.</td></tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                    </div>
                </div>
            </div>

        </div>
    </div>
</div>

<?php
include_once('includes/footer.php'); 
?>