<?php
// FILE: admin/add_expense.php

session_start();
ob_start();
include_once('../config/db.php');

$conn = getDBConnection();

$expense_types = ['Salaries', 'Rent', 'Utilities', 'Maintenance', 'Supplies', 'Others'];
$message = '';

// Delete expense functionality
if (isset($_GET['delete_id'])) {
    $delete_id = intval($_GET['delete_id']);
    $stmt = $conn->prepare("DELETE FROM other_expenses WHERE id = ?");
    if ($stmt) {
        $stmt->bind_param("i", $delete_id);
        if ($stmt->execute()) {
            $message = '<div class="alert alert-success">Expense deleted successfully!</div>';
        } else {
            $message = '<div class="alert alert-danger">Error deleting expense: ' . $stmt->error . '</div>';
        }
        $stmt->close();
    }
}

// Edit expense functionality
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Check if it's an edit request
    if (isset($_POST['edit_id'])) {
        // Update existing expense
        $edit_id = intval($_POST['edit_id']);
        $date = $_POST['date'] ?? date('Y-m-d');
        $expense_type = $_POST['expense_type'] ?? '';
        $description = $_POST['description'] ?? '';
        $amount = $_POST['amount'] ?? 0;

        // Validation
        if (empty($expense_type) || $amount <= 0 || empty($date)) {
            $message = '<div class="alert alert-danger">Please fill all required fields correctly.</div>';
        } else {
            $stmt = $conn->prepare("UPDATE other_expenses SET date = ?, expense_type = ?, description = ?, amount = ? WHERE id = ?");
            if ($stmt) {
                $stmt->bind_param("sssdi", $date, $expense_type, $description, $amount, $edit_id);
                if ($stmt->execute()) {
                    $message = '<div class="alert alert-success">Expense updated successfully!</div>';
                    // Redirect after success to prevent form resubmission
                    header("Location: add_expense.php?status=expense_updated");
                    exit();
                } else {
                    $message = '<div class="alert alert-danger">Error updating expense: ' . $stmt->error . '</div>';
                }
                $stmt->close();
            } else {
                $message = '<div class="alert alert-danger">Database error: ' . $conn->error . '</div>';
            }
        }
    } else {
        // Add new expense
        $date = $_POST['date'] ?? date('Y-m-d');
        $expense_type = $_POST['expense_type'] ?? '';
        $description = $_POST['description'] ?? '';
        $amount = $_POST['amount'] ?? 0;

        // Validation
        if (empty($expense_type) || $amount <= 0 || empty($date)) {
            $message = '<div class="alert alert-danger">Please fill all required fields correctly.</div>';
        } else {
            $stmt = $conn->prepare("INSERT INTO other_expenses (date, expense_type, description, amount) VALUES (?, ?, ?, ?)");
            if ($stmt) {
                $stmt->bind_param("sssd", $date, $expense_type, $description, $amount);
                if ($stmt->execute()) {
                    $message = '<div class="alert alert-success">Expense added successfully!</div>';
                    // Redirect after success to prevent form resubmission
                    header("Location: add_expense.php?status=expense_added");
                    exit();
                } else {
                    $message = '<div class="alert alert-danger">Error executing query: ' . $stmt->error . '</div>';
                }
                $stmt->close();
            } else {
                $message = '<div class="alert alert-danger">Database error: ' . $conn->error . '</div>';
            }
        }
    }
}

// Fetch expense data for editing
$edit_expense = null;
if (isset($_GET['edit_id'])) {
    $edit_id = intval($_GET['edit_id']);
    $stmt = $conn->prepare("SELECT * FROM other_expenses WHERE id = ?");
    if ($stmt) {
        $stmt->bind_param("i", $edit_id);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result->num_rows > 0) {
            $edit_expense = $result->fetch_assoc();
        }
        $stmt->close();
    }
}

// Fetch all expenses for the table
$expenses = [];
$expenses_query = "SELECT * FROM other_expenses ORDER BY date DESC, id DESC";
$expenses_result = $conn->query($expenses_query);
if ($expenses_result && $expenses_result->num_rows > 0) {
    while ($row = $expenses_result->fetch_assoc()) {
        $expenses[] = $row;
    }
}

// Includes for Page Layout
include('includes/header.php'); 
include_once('includes/sidebar.php');
?>

<div class="content-wrapper">
    <div class="content-header">
        <div class="container-fluid">
            <h1 class="m-0">
                <?php echo $edit_expense ? '✏️ Edit Expense' : '➕ Add New Expense'; ?>
            </h1>
        </div>
    </div>
    
    <section class="content">
        <div class="container-fluid">

            <div class="card <?php echo $edit_expense ? 'card-primary' : 'card-warning'; ?>">
                <div class="card-header">
                    <h3 class="card-title">
                        <?php echo $edit_expense ? 'Edit Expense Details' : 'Expense Details'; ?>
                    </h3>
                </div>
                <div class="card-body">
                    <?php echo $message; ?>
                    <form method="POST" action="add_expense.php">
                        <?php if ($edit_expense): ?>
                            <input type="hidden" name="edit_id" value="<?php echo $edit_expense['id']; ?>">
                        <?php endif; ?>
                        
                        <div class="row">
                            <div class="form-group col-md-4">
                                <label for="date">Date:</label>
                                <input type="date" name="date" id="date" class="form-control" 
                                       value="<?php echo $edit_expense ? htmlspecialchars($edit_expense['date']) : date('Y-m-d'); ?>" 
                                       required>
                            </div>
                            
                            <div class="form-group col-md-4">
                                <label for="expense_type">Expense Head:</label>
                                <select name="expense_type" id="expense_type" class="form-control" required>
                                    <option value="">-- Select Type --</option>
                                    <?php foreach ($expense_types as $type): ?>
                                        <option value="<?php echo $type; ?>" 
                                            <?php echo ($edit_expense && $edit_expense['expense_type'] == $type) ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($type); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            
                            <div class="form-group col-md-4">
                                <label for="amount">Amount (PKR):</label>
                                <input type="number" name="amount" id="amount" class="form-control" 
                                       value="<?php echo $edit_expense ? number_format($edit_expense['amount'], 2, '.', '') : ''; ?>" 
                                       step="0.01" min="0" required>
                            </div>
                        </div>

                        <div class="form-group">
                            <label for="description">Description (Optional):</label>
                            <textarea name="description" id="description" class="form-control" rows="3"><?php echo $edit_expense ? htmlspecialchars($edit_expense['description'] ?? '') : ''; ?></textarea>
                        </div>
                        
                        <div class="form-group">
                            <?php if ($edit_expense): ?>
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-save"></i> Update Expense
                                </button>
                                <a href="add_expense.php" class="btn btn-secondary">
                                    <i class="fas fa-times"></i> Cancel Edit
                                </a>
                            <?php else: ?>
                                <button type="submit" class="btn btn-success">
                                    <i class="fas fa-save"></i> Save Expense
                                </button>
                            <?php endif; ?>
                            <a href="financial_summary_report.php" class="btn btn-info">
                                <i class="fas fa-chart-bar"></i> Back to Report
                            </a>
                        </div>
                    </form>
                </div>
            </div>

            <!-- Expenses List Table -->
            <div class="card card-primary mt-4">
                <div class="card-header">
                    <h3 class="card-title">📋 All Expenses</h3>
                    <div class="card-tools">
                        <span class="badge badge-light">Total: <?php echo count($expenses); ?> expenses</span>
                    </div>
                </div>
                <div class="card-body">
                    <?php if (empty($expenses)): ?>
                        <div class="alert alert-info">
                            <i class="fas fa-info-circle"></i> No expenses found. Add your first expense above.
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-bordered table-striped table-hover">
                                <thead class="thead-dark">
                                    <tr>
                                        <th>#</th>
                                        <th>Date</th>
                                        <th>Expense Head</th>
                                        <th>Description</th>
                                        <th>Amount (PKR)</th>
                                        <th>Created At</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php 
                                    $total_amount = 0;
                                    foreach ($expenses as $index => $expense): 
                                        $total_amount += $expense['amount'];
                                    ?>
                                    <tr>
                                        <td><?php echo $index + 1; ?></td>
                                        <td><?php echo htmlspecialchars($expense['date']); ?></td>
                                        <td>
                                            <span class="badge badge-info"><?php echo htmlspecialchars($expense['expense_type']); ?></span>
                                        </td>
                                        <td><?php echo htmlspecialchars($expense['description'] ?? 'N/A'); ?></td>
                                        <td class="text-right font-weight-bold"><?php echo number_format($expense['amount'], 2); ?></td>
                                        <td><?php echo htmlspecialchars($expense['created_at']); ?></td>
                                        <td>
                                            <div class="btn-group btn-group-sm">
                                                <!-- Edit Button -->
                                                <a href="add_expense.php?edit_id=<?php echo $expense['id']; ?>" 
                                                   class="btn btn-warning" 
                                                   title="Edit Expense">
                                                    <i class="fas fa-edit"></i> Edit
                                                </a>
                                                <!-- Delete Button -->
                                                <a href="add_expense.php?delete_id=<?php echo $expense['id']; ?>" 
                                                   class="btn btn-danger" 
                                                   onclick="return confirm('Are you sure you want to delete this expense? This action cannot be undone.')"
                                                   title="Delete Expense">
                                                    <i class="fas fa-trash"></i> Delete
                                                </a>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                                <tfoot>
                                    <tr class="table-success">
                                        <th colspan="4" class="text-right">TOTAL EXPENSES:</th>
                                        <th class="text-right"><?php echo number_format($total_amount, 2); ?></th>
                                        <th colspan="2"></th>
                                    </tr>
                                </tfoot>
                            </table>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
            
        </div>
    </section>
</div>

<!-- Success/Error Message Modal -->
<div class="modal fade" id="messageModal" tabindex="-1" role="dialog" aria-hidden="true">
    <div class="modal-dialog modal-sm" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Notification</h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body" id="modalMessage">
                <!-- Message will be shown here -->
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">Close</button>
            </div>
        </div>
    </div>
</div>

<script>
// Function to show modal message
function showModalMessage(message, type = 'info') {
    $('#modalMessage').html(message);
    $('#messageModal').modal('show');
}

// Auto-hide alerts after 5 seconds
setTimeout(function() {
    $('.alert').fadeOut('slow');
}, 5000);

// Show success message from URL parameter
$(document).ready(function() {
    const urlParams = new URLSearchParams(window.location.search);
    const status = urlParams.get('status');
    
    if (status === 'expense_added') {
        showModalMessage('<div class="alert alert-success"><i class="fas fa-check-circle"></i> Expense added successfully!</div>');
    } else if (status === 'expense_updated') {
        showModalMessage('<div class="alert alert-success"><i class="fas fa-check-circle"></i> Expense updated successfully!</div>');
    }
    
    // Scroll to form when editing
    <?php if ($edit_expense): ?>
        $('html, body').animate({
            scrollTop: $(".card-primary").offset().top - 100
        }, 500);
    <?php endif; ?>
});

// Form validation
document.addEventListener('DOMContentLoaded', function() {
    const form = document.querySelector('form');
    const amountInput = document.getElementById('amount');
    
    if (form) {
        form.addEventListener('submit', function(e) {
            const amount = parseFloat(amountInput.value);
            if (amount <= 0) {
                e.preventDefault();
                showModalMessage('<div class="alert alert-danger"><i class="fas fa-exclamation-triangle"></i> Please enter a valid amount greater than 0.</div>');
                amountInput.focus();
                return false;
            }
            
            const expenseType = document.getElementById('expense_type').value;
            if (!expenseType) {
                e.preventDefault();
                showModalMessage('<div class="alert alert-danger"><i class="fas fa-exclamation-triangle"></i> Please select an expense type.</div>');
                return false;
            }
        });
    }
    
    // Format amount on blur
    if (amountInput) {
        amountInput.addEventListener('blur', function() {
            const value = parseFloat(this.value);
            if (!isNaN(value) && value >= 0) {
                this.value = value.toFixed(2);
            }
        });
    }
});

// Confirm delete with sweet alert style
function confirmDelete(link) {
    if (confirm('Are you sure you want to delete this expense?\n\nThis action cannot be undone.')) {
        return true;
    }
    return false;
}
</script>

<style>
.card-header {
    background: linear-gradient(45deg, #f8f9fa, #e9ecef);
}
.table th {
    background-color: #495057;
    color: white;
}
.btn-group-sm > .btn {
    padding: 0.25rem 0.5rem;
    font-size: 0.875rem;
}
.badge {
    font-size: 0.85em;
}
.alert {
    border-radius: 0.5rem;
    border: none;
}
</style>

<?php 
// Connection Closure 
if (isset($conn) && $conn instanceof mysqli) {
    //@$conn->close();
}

include_once('includes/footer.php'); 
ob_end_flush();
?>