<?php
// FILE: admin/class_datesheet.php - School-Wide Datesheet Composition & Management (Matrix View - Simplified)

session_start();
// Security Check (Agar zaruri ho)
// if (!isset($_SESSION['user_id'])) { header('Location: login.php'); exit; }

include_once('../config/db.php');

// Utility function to establish DB connection (assuming getDBConnection() is available)
if (!function_exists('getDBConnection')) {
    // Placeholder/Fallback: Replace with actual DB connection details
    function getDBConnection() {
        // Replace with your actual DB connection details
        return new mysqli('localhost', 'db_user', 'db_password', 'db_name'); 
    }
}

// Attempt to establish connection
try {
    $conn = getDBConnection(); 
    if ($conn->connect_error) {
        throw new Exception("Database Connection Failed: " . $conn->connect_error);
    }
} catch (Exception $e) {
    $conn = null;
    $error = "System Error: " . $e->getMessage();
}

$message = '';
if (!isset($error)) $error = '';


// --------------------------------------------------------------------------------
// 0. DELETE LOGIC (MOVED TO TOP to prevent 'headers already sent' error)
// --------------------------------------------------------------------------------
if (isset($_GET['action']) && $_GET['action'] === 'delete_datesheet' && isset($_GET['exam_id'])) {
    $exam_id = (int)$_GET['exam_id'];
    
    if ($conn) {
        // Get exam name for message before deleting
        $exam_q = @$conn->query("SELECT name FROM exams WHERE id = " . $exam_id);
        $exam_name_for_msg = $exam_q ? htmlspecialchars($exam_q->fetch_assoc()['name']) : "ID: $exam_id";

        // Delete from exam_datesheet table
        $delete_q = @$conn->prepare("DELETE FROM exam_datesheet WHERE exam_id = ?");
        if ($delete_q) {
            $delete_q->bind_param("i", $exam_id);
            if ($delete_q->execute()) {
                $message = "Complete Datesheet for Exam: **$exam_name_for_msg** successfully deleted.";
            } else {
                $error = "Error deleting datesheet: " . $conn->error;
            }
            $delete_q->close();
        } else {
             $error = "Database error: Could not prepare delete statement.";
        }
        
    } else {
         $error = "Cannot connect to database to perform deletion.";
    }
    // Redirect must be the last thing before exit
    header("Location: class_datesheet.php?msg=" . urlencode($message) . "&err=" . urlencode($error));
    exit;
}
// --------------------------------------------------------------------------------

// --------------------------------------------------------------------------------
// 1. AJAX Endpoint for Print Preview Data Fetch (MOVED TO TOP)
// --------------------------------------------------------------------------------
if (isset($_GET['action']) && $_GET['action'] === 'fetch_print_data') {
    // Ye line output hai, isliye iske baad koi header() call nahi ho sakta.
    header('Content-Type: application/json');
    
    $exam_id = (int)($_GET['exam_id'] ?? 0);
    // Default response for safety if tables are missing
    $response = ['datesheet_data' => [], 'exam_info' => ['name' => 'Unknown Exam', 'start_date' => date('Y-m-d'), 'end_date' => date('Y-m-d')], 'error' => null];

    if ($exam_id && $conn) {
        
        // A. Fetch Exam Information (Name, Date Range)
        $exam_info_q = @$conn->query("
            SELECT e.name, MIN(ed.exam_date) AS start_date, MAX(ed.exam_date) AS end_date
            FROM exams e
            JOIN exam_datesheet ed ON e.id = ed.exam_id
            WHERE e.id = {$exam_id} 
            GROUP BY e.id, e.name
            LIMIT 1
        ");
        if ($exam_info_q && $exam_info_q->num_rows > 0) {
            $response['exam_info'] = $exam_info_q->fetch_assoc();
        } else {
             $response['error'] = ($response['error'] ?? '') . "Warning: Could not fetch exam date range. Tables 'exams' or 'exam_datesheet' missing or empty.";
        }


        // B. Fetch Datesheet Data 
        // Note: Time and Marks are fetched but will be ignored for the matrix print view
        $datesheet_q = @$conn->query("
            SELECT 
                ed.exam_date,
                ed.start_time,
                ed.end_time,
                ed.total_marks AS max_marks, 
                ed.passing_marks AS pass_marks,
                ac.name AS class_name, 
                s.name AS subject_name,
                ed.subject_id
            FROM exam_datesheet ed
            LEFT JOIN academic_classes ac ON ed.class_id = ac.id
            LEFT JOIN subjects s ON ed.subject_id = s.id
            WHERE ed.exam_id = {$exam_id}
            ORDER BY ed.exam_date ASC, ac.order_no ASC
        ");

        if ($datesheet_q) {
             while ($row = $datesheet_q->fetch_assoc()) {
                 
                 // Handle HOLIDAY case where subject_id is 0
                 if ($row['subject_id'] == 0) {
                    $row['subject_name'] = 'HOLIDAY';
                 }
                 
                 $exam_date = $row['exam_date'];
                 
                 if (!isset($response['datesheet_data'][$exam_date])) {
                     $response['datesheet_data'][$exam_date] = [];
                 }
                 
                 // Store all necessary fields for the matrix lookup in JS
                 $response['datesheet_data'][$exam_date][] = [
                     'class_name' => $row['class_name'],
                     'subject_name' => htmlspecialchars($row['subject_name']),
                 ];
             }
        } else {
            $response['error'] = ($response['error'] ?? '') . "Error fetching datesheet data. 'exam_datesheet' table may be missing.";
        }
        
    } else {
        $response['error'] = "Missing required exam ID or Database connection failed.";
    }

    echo json_encode($response);
    exit; // EXIT here is CRITICAL to prevent header output.
}
// --------------------------------------------------------------------------------


// --- CRITICAL FIX: INCLUDE HEADER HERE ---
// Header/Sidebar files output HTML, which is why the previous header() calls failed.
// Ab humne saari redirects aur output calls upar kar di hain.
include_once('includes/header.php'); 
// -----------------------------------------


// 🚨 FIX FOR FATAL ERROR: Re-establish connection because header.php/sidebar.php likely closed the previous one.
// Agar $conn band ho chuka hai (jo ki line 70 par error ka sabab bana), to dobara open karein.
$conn = getDBConnection();


// --- START: Settings & Global Data Fetch ---
$settings_data = [];
if ($conn) {
    // Assuming a 'settings' table exists
    $settings_query = @$conn->query("SELECT setting_key, setting_value FROM settings"); 
    if ($settings_query) {
        while ($row = $settings_query->fetch_assoc()) {
            $settings_data[trim($row['setting_key'])] = trim($row['setting_value']);
        }
    }
}
$institute_name = $settings_data['institute.name'] ?? "SCHOOL NAME (UPDATE SETTINGS)";
$institute_address = $settings_data['institute.address'] ?? "Address Not Set";
$institute_phone = $settings_data['institute.phone'] ?? "Phone Not Set";

// LOGO PATH FIX APPLIED HERE
$raw_logo_path = $settings_data['institute.logo_path'] ?? '';
$logoPathParts = explode('|', $raw_logo_path);
$logo_filename = end($logoPathParts); 
// Path: ../ (go up to root) /uploads/ [filename]
$logo_path = $logo_filename ? '../uploads/' . basename($logo_filename) : ''; 
// --- END: Settings Fetch ---


// Default values for form
$selected_exam_id = $_POST['exam_id'] ?? null;
$start_date = $_POST['start_date'] ?? null;
$end_date = $_POST['end_date'] ?? null;
$selected_exam_name = '';

// --- Fetch Exams and All Classes ---
$exams = [];
$classes = [];
if ($conn) {
    // Class names fetch kar rahe hain 'name' as 'class_name'
    $class_q = @$conn->query("SELECT id, name AS class_name, order_no FROM academic_classes ORDER BY order_no ASC"); 
    if ($class_q) {
        while ($row = $class_q->fetch_assoc()) {
            $classes[] = $row;
        }
    }
    $exam_q = @$conn->query("SELECT id, name AS exam_name FROM exams ORDER BY id DESC"); 
    if ($exam_q) {
        while ($row = $exam_q->fetch_assoc()) {
            $exams[] = $row;
        }
    }
}


// --------------------------------------------------------------------------------
// 2. DATA PROCESSING (Form Submission: SAVE) - PHP Subject Repetition Check Added
// --------------------------------------------------------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_datesheet']) && $conn) {
    if (!$selected_exam_id) {
        $error = "Please select an Exam.";
    } else {
        
        // **NEW: PHP-side Repetition Check** (Client-side JS check ke bawajood, server check zaruri hai)
        $class_subject_tracker = []; // Stores subject_id for each class to check for repetition
        $has_repetition_error = false;
        $repetition_error_message = "Error: Following classes have the same subject selected multiple times: ";
        
        // Phase 1: Check for repetition before database transaction
        foreach ($_POST['datesheet_data'] as $class_id => $date_data) {
            foreach ($date_data as $date_string => $data) {
                $subject_id = $data['subject_id'] ?? 'NONE';
                
                if ($subject_id != 'NONE' && $subject_id != 'HOLIDAY') {
                    if (!isset($class_subject_tracker[$class_id])) {
                        $class_subject_tracker[$class_id] = [];
                    }
                    
                    $subject_key = (int)$subject_id;
                    
                    if (in_array($subject_key, $class_subject_tracker[$class_id])) {
                         // Repetition found for this class!
                         $has_repetition_error = true;
                         // Get Class Name from $classes array (if needed for better message)
                         $className = 'Unknown Class';
                         foreach ($classes as $class) {
                             if ($class['id'] == $class_id) {
                                 $className = $class['class_name'];
                                 break;
                             }
                         }
                         $repetition_error_message .= " [{$className} / Subject ID: {$subject_key}]";
                         break 2; // Exit inner loops as error is found
                    }
                    $class_subject_tracker[$class_id][] = $subject_key;
                }
            }
        }
        
        if ($has_repetition_error) {
            $error = $repetition_error_message . ". Please ensure each subject is taught only once in a class.";
        } else {
            // No repetition error, proceed with DB saving
            
            @$conn->begin_transaction();
            
            try {
                // A. Existing data delete karein
                $delete_stmt = @$conn->prepare("DELETE FROM exam_datesheet WHERE exam_id = ?");
                if ($delete_stmt) {
                    $delete_stmt->bind_param("i", $selected_exam_id);
                    $delete_stmt->execute();
                    $delete_stmt->close();
                }
                
                $saved_count = 0;

                // B. Prepare the INSERT statement (Simplified - removed time/marks, using defaults)
                // Note: Hardcoding '00:00:00' and '0' for total/passing marks
                $insert_stmt = @$conn->prepare("
                    INSERT INTO exam_datesheet 
                    (exam_id, class_id, subject_id, exam_date, start_time, end_time, total_marks, passing_marks) 
                    VALUES (?, ?, ?, ?, '00:00:00', '00:00:00', 0, 0)
                ");
                
                // C. Loop through POST data
                if ($insert_stmt) {
                    foreach ($_POST['datesheet_data'] as $class_id => $date_data) {
                        foreach ($date_data as $date_string => $data) {
                            
                            if (empty($date_string) || $date_string === '0000-00-00') {
                                continue;
                            }
                            
                            $subject_id = $data['subject_id'] ?? 'NONE';
                            
                            // Skip agar koi subject select nahi kiya
                            if ($subject_id == 'NONE') {
                                continue; 
                            }

                            $exam_date = $date_string;
                            
                            // Holiday ke liye subject_id ko 0 set karein
                            $insert_subject_id = $subject_id == 'HOLIDAY' ? 0 : (int)$subject_id;
                            
                            // Bind parameters and execute
                            $insert_stmt->bind_param(
                                "iiis", 
                                $selected_exam_id, 
                                $class_id, 
                                $insert_subject_id, 
                                $exam_date
                            );
                            
                            if (!$insert_stmt->execute()) {
                                 // Check for duplicate key error (1062)
                                 if ($conn->errno == 1062) {
                                      throw new Exception("Duplicate entry error (Code 1062). Please ensure the PRIMARY KEY on 'exam_datesheet' table includes (exam_id, class_id, exam_date) and not subject_id.");
                                 } else {
                                      throw new Exception("SQL execution failed: " . $insert_stmt->error);
                                 }
                            }
                            $saved_count++;
                        }
                    }
                    
                    $insert_stmt->close();
                }
                
                @$conn->commit();
                $message = "Datesheet successfully saved. Total entries: " . $saved_count;
                
            } catch (Exception $e) {
                @$conn->rollback();
                $error = "Error saving datesheet: Please ensure all required tables exist. Technical Error: " . $e->getMessage();
            }
        }
    }
    
    // Redirect after POST
    header("Location: class_datesheet.php?exam_id=" . $selected_exam_id . "&msg=" . urlencode($message) . "&err=" . urlencode($error));
    exit;
}


// --------------------------------------------------------------------------------
// 3. DATA FETCHING & DATE GENERATION (For Composition Table Display)
// --------------------------------------------------------------------------------

// A. Fetch ALL subjects, mapped by class_id
$class_subjects_map = []; 
if ($conn) {
    $subject_q_text = "
        SELECT s.id AS subject_id, s.name AS subject_name, ca.class_id
        FROM subjects s
        JOIN course_allocation ca ON s.id = ca.subject_id
        GROUP BY s.id, s.name, ca.class_id
        ORDER BY s.name ASC
    ";
    $subject_result = @$conn->query($subject_q_text);
    if ($subject_result) {
        while ($row = $subject_result->fetch_assoc()) {
            $class_subjects_map[$row['class_id']][] = [
                'id' => $row['subject_id'],
                'name' => htmlspecialchars($row['subject_name'])
            ];
        }
    }
}


// B. Fetch Existing Datesheet Data
$existing_datesheet_data = []; // [date => [class_id => {subject_id}]]
$all_exam_dates = []; // [unique dates]

if ($selected_exam_id && $conn) {
    $exam_q = @$conn->query("SELECT name FROM exams WHERE id = " . (int)$selected_exam_id);
    if ($exam_q && $exam_q->num_rows > 0) {
        $selected_exam_name = htmlspecialchars($exam_q->fetch_assoc()['name']);
    }

    $datesheet_q_text = "
        SELECT class_id, subject_id, exam_date
        FROM exam_datesheet 
        WHERE exam_id = " . (int)$selected_exam_id . "
        ORDER BY exam_date ASC, class_id ASC
    ";
    
    $datesheet_result = @$conn->query($datesheet_q_text);
    
    if ($datesheet_result) {
        while ($row = $datesheet_result->fetch_assoc()) {
            $date = $row['exam_date'];
            $class_id = $row['class_id'];
            
            if (!isset($existing_datesheet_data[$date])) {
                $existing_datesheet_data[$date] = [];
            }
            
            // Agar subject_id = 0 hai, toh isko 'HOLIDAY' mark karein
            $subject_value = $row['subject_id'] == 0 ? 'HOLIDAY' : $row['subject_id'];
            
            $existing_datesheet_data[$date][$class_id] = [
                'subject_id' => $subject_value,
            ];
            
            if (!in_array($date, $all_exam_dates)) {
                $all_exam_dates[] = $date;
            }
        }
        sort($all_exam_dates); 
    }
}

// C. Date Range Generation and Sunday Auto-Holiday Logic
$dates_to_display = [];

if ($start_date && $end_date) {
    try {
        $begin = new DateTime($start_date);
        $end = new DateTime($end_date);
        $end->modify('+1 day'); 

        $interval = DateInterval::createFromDateString('1 day');
        $period = new DatePeriod($begin, $interval, $end);

        foreach ($period as $dt) {
            $date = $dt->format("Y-m-d");
            $dates_to_display[] = $date;

            // Auto-fill Sunday Holiday Logic (0 is Sunday)
            if ($dt->format('w') == 0) { 
                if (!isset($existing_datesheet_data[$date])) {
                    $existing_datesheet_data[$date] = [];
                }
                foreach ($classes as $class) {
                    $class_id = $class['id'];
                    // Only pre-fill if the database doesn't already have an entry for this class/date
                    if (!isset($existing_datesheet_data[$date][$class_id])) {
                         $existing_datesheet_data[$date][$class_id] = [
                            'subject_id' => 'HOLIDAY',
                        ];
                    }
                }
            }
        }
    } catch (Exception $e) {
        $error .= " Invalid date range selected. ";
        $dates_to_display = $all_exam_dates; // Fallback to existing dates
    }
} else {
    // Fallback: If no range selected, display existing dates
    $dates_to_display = $all_exam_dates;
}

// Add 5 empty rows if current dates are less than 5
$dates_to_display = array_unique($dates_to_display);
sort($dates_to_display);

if (count($dates_to_display) < 5) {
    for ($i = 0; $i < (5 - count($dates_to_display)); $i++) {
        $dates_to_display[] = ''; // Empty slots for new entries
    }
}

// PHP Helper function to get FULL Day name (l)
function get_day_name($date_string) {
    if (empty($date_string) || $date_string === '0000-00-00') {
        return '-';
    }
    try {
        $date_obj = new DateTime($date_string);
        return $date_obj->format('l'); // 'l' for full day name: Monday, Tuesday, etc.
    } catch (Exception $e) {
        return '?';
    }
}


// --------------------------------------------------------------------------------
// 4. FETCH DISTINCT SAVED DATESHEETS FOR LISTING
// --------------------------------------------------------------------------------
$distinct_datesheets = [];
if ($conn) {
    // Query to get Exam Name, ID, Min/Max Dates
    $saved_datesheet_q = @$conn->query(" 
        SELECT ed.exam_id, e.name AS exam_name, MIN(ed.exam_date) AS start_date, MAX(ed.exam_date) AS end_date, COUNT(DISTINCT ed.class_id) as class_count 
        FROM exam_datesheet ed 
        JOIN exams e ON ed.exam_id = e.id 
        GROUP BY ed.exam_id, e.name 
        ORDER BY start_date DESC 
        LIMIT 200 
    ");
    if ($saved_datesheet_q) {
        while ($row = $saved_datesheet_q->fetch_assoc()) {
            $distinct_datesheets[] = $row;
        }
    }
}
// --------------------------------------------------------------------------------
?> 
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <h1 class="h3 mb-4 text-gray-800">🗓️ School Datesheet Composer (Matrix)</h1>
        </div>
    </div>

    <?php if ($message): ?>
        <div class="alert alert-success"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>
    <?php if ($error): ?>
        <div class="alert alert-danger"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <div class="card shadow mb-4">
        <div class="card-header py-3">
            <h6 class="m-0 font-weight-bold text-primary">Select Exam & Date Range</h6>
        </div>
        <div class="card-body">
            <form id="edit_trigger_form" method="POST" action="class_datesheet.php" style="display: none;">
                <input type="hidden" name="exam_id" id="edit_exam_id">
                <input type="hidden" name="start_date" id="edit_start_date">
                <input type="hidden" name="end_date" id="edit_end_date">
            </form>
            
            <form method="POST" action="class_datesheet.php">
                <div class="form-row">
                    <div class="form-group col-md-4">
                        <label for="exam_id">Select Exam Name</label>
                        <select id="exam_id" name="exam_id" class="form-control" required>
                            <option value="">-- Select Exam --</option>
                            <?php foreach ($exams as $exam): ?>
                                <option 
                                    value="<?php echo $exam['id']; ?>"
                                    <?php echo ($exam['id'] == $selected_exam_id) ? 'selected' : ''; ?>
                                >
                                    <?php echo htmlspecialchars($exam['exam_name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="form-group col-md-3">
                        <label for="start_date">Start Date</label>
                        <input type="date" id="start_date" name="start_date" class="form-control" value="<?php echo htmlspecialchars($start_date); ?>" required>
                    </div>
                    <div class="form-group col-md-3">
                        <label for="end_date">End Date</label>
                        <input type="date" id="end_date" name="end_date" class="form-control" value="<?php echo htmlspecialchars($end_date); ?>" required>
                    </div>
                    <div class="form-group col-md-2 d-flex align-items-end">
                        <button type="submit" class="btn btn-info btn-block" name="load_datesheet">Load Datesheet</button>
                    </div>
                </div>
            </form>
        </div>
    </div>
    
    <?php if ($selected_exam_id): ?>
    <div class="card shadow mb-4">
        <div class="card-header py-3">
            <h6 class="m-0 font-weight-bold text-success">
                Compose Datesheet for: **<?php echo $selected_exam_name; ?>** (<?php echo htmlspecialchars($start_date); ?> to <?php echo htmlspecialchars($end_date); ?>)
            </h6>
        </div>
        <div class="card-body">
            <form method="POST" action="class_datesheet.php" id="datesheet_form">
                <input type="hidden" name="exam_id" value="<?php echo htmlspecialchars($selected_exam_id); ?>">
                <input type="hidden" name="start_date" value="<?php echo htmlspecialchars($start_date); ?>">
                <input type="hidden" name="end_date" value="<?php echo htmlspecialchars($end_date); ?>">
                <div class="table-responsive">
                    <table class="table table-bordered table-striped" id="datesheetComposerTable" width="100%" cellspacing="0">
                        <thead>
                            <tr class="bg-dark text-white text-center">
                                <th style="min-width: 150px; width: 15%;">Date & Day</th>
                                <?php foreach ($classes as $class): ?>
                                    <th class="class-column" data-class-id="<?php echo $class['id']; ?>" style="min-width: 120px; width: auto;"><?php echo htmlspecialchars($class['class_name']); ?></th>
                                <?php endforeach; ?>
                            </tr>
                        </thead>
                        <tbody>
                            <?php $num_rows = count($dates_to_display); // Start loop for rows (Dates)
                            for($i = 0; $i < $num_rows; $i++): 
                                $date_value = $dates_to_display[$i] ?? '';
                                $row_id = "row_" . $i; ?>
                                <tr id="<?php echo $row_id; ?>">
                                    <td class="align-middle bg-light text-center" style="padding: 5px;">
                                        <input type="date" name="exam_dates[]" class="form-control form-control-sm datesheet-date-input" value="<?php echo htmlspecialchars($date_value); ?>" onchange="updateDayName(this)">
                                        <span class="badge badge-secondary mt-1 day-name-display"><?php echo get_day_name($date_value); ?></span>
                                    </td>
                                    <?php foreach ($classes as $class): 
                                        $class_id = $class['id'];
                                        // Existing/Auto-Holiday data check
                                        $data = $existing_datesheet_data[$date_value][$class_id] ?? [];
                                        $selected_subject_id = $data['subject_id'] ?? 'NONE';
                                        
                                        $subject_options = '<option value="NONE">-- Select Subject --</option>';
                                        $subject_options .= '<option value="HOLIDAY" ' . ($selected_subject_id == 'HOLIDAY' ? 'selected' : '') . '>HOLIDAY / OFF</option>';
                                        
                                        // Only add subjects if available for this class
                                        if (isset($class_subjects_map[$class_id])) {
                                            foreach ($class_subjects_map[$class_id] as $subject) {
                                                $selected = ($subject['id'] == $selected_subject_id) ? 'selected' : '';
                                                $subject_options .= "<option value=\"{$subject['id']}\" {$selected}>{$subject['name']}</option>";
                                            }
                                        }
                                        
                                        // Name attribute is set dynamically via JS when date is selected/changed (updateDayName)
                                        $name_attr = $date_value ? "name=\"datesheet_data[{$class_id}][{$date_value}][subject_id]\"" : "name=\"\"";
                                    ?>
                                    <td class="p-1">
                                        <select 
                                            class="form-control form-control-sm subject-selector" 
                                            data-class-id="<?php echo $class_id; ?>"
                                            data-row-index="<?php echo $i; ?>"
                                            <?php echo $name_attr; ?>
                                        >
                                            <?php echo $subject_options; ?>
                                        </select>
                                    </td>
                                    <?php endforeach; ?>
                                </tr>
                            <?php endfor; ?>
                        </tbody>
                        <tfoot>
                            <tr>
                                <td colspan="<?php echo count($classes) + 1; ?>" class="text-right">
                                    <button type="button" class="btn btn-secondary btn-sm" onclick="addNewRow()">+ Add Date Row</button>
                                    <button type="submit" class="btn btn-success" name="save_datesheet">Save Datesheet</button>
                                </td>
                            </tr>
                        </tfoot>
                    </table>
                </div>
            </form>
        </div>
    </div>
    <?php endif; ?>


    <hr class="mt-5 mb-4">
    <div class="card shadow mb-4">
        <div class="card-header py-3">
            <h6 class="m-0 font-weight-bold text-info">
                📑 Existing Exam Datesheets (Total: <?php echo count($distinct_datesheets); ?>)
            </h6>
        </div>
        <div class="card-body">
            
            <?php if (!empty($distinct_datesheets)): ?>
            
            <div class="table-responsive">
                <table class="table table-bordered table-striped small" id="savedDatesheetsTable" width="100%" cellspacing="0">
                    <thead>
                        <tr>
                            <th style="width: 35%;">Exam Name</th>
                            <th style="width: 35%;">Date Range</th>
                            <th style="width: 10%;">Classes</th>
                            <th style="width: 20%;">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($distinct_datesheets as $entry): ?>
                        <tr>
                            <td class="text-left font-weight-bold"><?php echo htmlspecialchars($entry['exam_name']); ?></td>
                            <td class="text-nowrap">
                                <?php echo date('d-M-Y', strtotime($entry['start_date'])); ?> to <?php echo date('d-M-Y', strtotime($entry['end_date'])); ?>
                            </td>
                            <td class="text-center">
                                <?php echo $entry['class_count']; ?>
                            </td>
                            <td class="text-nowrap">
                                
                                <button 
                                    type="button" 
                                    class="btn btn-sm btn-success btn-print-datesheet"
                                    data-exam-id="<?php echo $entry['exam_id']; ?>"
                                    data-exam-name="<?php echo htmlspecialchars($entry['exam_name']); ?>"
                                    data-start-date="<?php echo $entry['start_date']; ?>"
                                    data-end-date="<?php echo $entry['end_date']; ?>"
                                    title="View & Print Preview of this Datesheet"
                                >
                                    <i class="fas fa-print"></i> Print
                                </button>
                                
                                <button 
                                    class="btn btn-sm btn-primary edit-datesheet-btn" 
                                    data-exam-id="<?php echo $entry['exam_id']; ?>" 
                                    data-start-date="<?php echo $entry['start_date']; ?>" 
                                    data-end-date="<?php echo $entry['end_date']; ?>" 
                                    title="Load this datesheet into the composer"
                                >
                                    <i class="fas fa-edit"></i> Edit
                                </button>
                                
                                <a 
                                    href="class_datesheet.php?action=delete_datesheet&exam_id=<?php echo $entry['exam_id']; ?>" 
                                    class="btn btn-sm btn-danger ml-1" 
                                    onclick="return confirm('Are you sure you want to permanently delete the FULL DATESHEET for Exam: <?php echo htmlspecialchars($entry['exam_name']); ?>? This cannot be undone.')" 
                                    title="Delete this entire datesheet"
                                >
                                    <i class="fas fa-trash"></i> Delete
                                </a>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            <?php else: ?>
                <div class="alert alert-info">No existing exam datesheets found.</div>
            <?php endif; ?>
        </div>
    </div>
    
</div>

<?php 
// PHP variables to pass to JS for printing settings
$js_settings = [
    'institute_name' => $institute_name,
    'institute_address' => $institute_address,
    'institute_phone' => $institute_phone,
    'logo_path' => $logo_path,
    'classes' => $classes, // Pass classes for print view
];
?>

<script src="../assets/js/jquery.min.js"></script>

<script>
const DATESHEET_SETTINGS = <?php echo json_encode($js_settings); ?>;
const CLASSES = DATESHEET_SETTINGS.classes;

document.addEventListener('DOMContentLoaded', function() {
    
    // Helper function for date formatting
    const formatPrintDate = (dateString) => {
        if (!dateString || dateString.includes('0000-00-00')) return 'N/A';
        const date = new Date(dateString);
        return date.toLocaleDateString('en-GB', { day: '2-digit', month: 'short', year: 'numeric' });
    };
    
    // --- Existing JS logic: Edit Button Handler ---
    const editTriggerForm = document.getElementById('edit_trigger_form');
    const editExamId = document.getElementById('edit_exam_id');
    const editStartDate = document.getElementById('edit_start_date');
    const editEndDate = document.getElementById('edit_end_date');
    
    if (editTriggerForm) {
        document.querySelectorAll('.edit-datesheet-btn').forEach(button => {
            button.addEventListener('click', function() {
                editExamId.value = this.getAttribute('data-exam-id');
                editStartDate.value = this.getAttribute('data-start-date');
                editEndDate.value = this.getAttribute('data-end-date');
                editTriggerForm.submit();
            });
        });
    }
    // ---------------------------------------------


    // -----------------------------------------------------------------------------------
    // 5. Core Print Function (MODIFIED FOR MATRIX VIEW)
    // -----------------------------------------------------------------------------------
    window.printDatesheet = function(datesheetData, examInfo) {
        
        const settings = DATESHEET_SETTINGS;
        const classes = settings.classes; // Array of {id, class_name, order_no}
        
        // 1. Generate Report Header
        const headerTitle = `Datesheet for ${examInfo.name}`;
        const dateRange = `${formatPrintDate(examInfo.start_date)} to ${formatPrintDate(examInfo.end_date)}`;
        
        const reportHeader = `
            <div class="report-header">
                <table class="header-table">
                    <tr>
                        <td class="logo-cell">
                            ${settings.logo_path ? `<img src="${settings.logo_path}" alt="School Logo" class="logo">` : ''}
                        </td>
                        <td class="info-cell">
                            <h2 style="margin: 0; font-size: 16pt; font-weight: bold;">${settings.institute_name}</h2>
                            <p style="margin: 0; font-size: 10pt;">${settings.institute_address} | Phone: ${settings.institute_phone}</p>
                        </td>
                        <td style="width: 20%;"></td> 
                    </tr>
                </table>
                <hr style="border: 1px solid #000; margin-top: 5px;">
                <h3 style="margin: 5px 0 5px; font-size: 14pt; text-align: center; font-weight: bold;">${headerTitle}</h3>
                <p style="margin: 0 0 10px 0; font-size: 11pt; text-align: center;">
                    **Exam Period:** ${dateRange}
                </p>
                <hr style="border: 1px dashed #999; margin-bottom: 10px;">
            </div>
        `;

        // 2. Process Datesheet Data for Matrix Table
        let datesheetHTML = '';
        const dates = Object.keys(datesheetData).sort(); 
        
        if (dates.length === 0) {
            alert('No datesheet data found to print for this exam. Check if the "exam_datesheet" table has entries.');
            return;
        }
        
        // Create a simplified lookup map: [date][class_name] = subject_name
        const datesheetMatrix = {};
        dates.forEach(date => {
            datesheetMatrix[date] = {};
            datesheetData[date].forEach(item => {
                // Use class_name as key for easy lookup
                datesheetMatrix[date][item.class_name] = item.subject_name;
            });
        });

        // Start the main matrix table
        datesheetHTML += `
            <table class="datesheet-matrix-table" width="100%" cellspacing="0">
                <thead>
                    <tr>
                        <th style="width: 15%; min-width: 150px; background-color: #343a40; color: white;">Date & Day</th>
                        ${classes.map(c => 
                            `<th style="background-color: #343a40; color: white;">${c.class_name}</th>`
                        ).join('')}
                    </tr>
                </thead>
                <tbody>
        `;
        
        // Add data rows
        dates.forEach(date => {
            const day_name = new Date(date).toLocaleDateString('en-US', { weekday: 'long' });
            const formattedDate = formatPrintDate(date);
            
            datesheetHTML += `<tr>`;
            // Date/Day Column
            datesheetHTML += `
                <td class="text-nowrap" style="font-weight: bold; background-color: #f8f9fa; border-right: 2px solid #000 !important;">
                    ${formattedDate} (${day_name})
                </td>
            `;
            
            // Class Columns (Subjects)
            classes.forEach(classItem => {
                const className = classItem.class_name;
                // Lookup subject name for this date/class
                const subjectName = datesheetMatrix[date] && datesheetMatrix[date][className] 
                                    ? datesheetMatrix[date][className] 
                                    : 'N/A'; // N/A if class doesn't have an exam on this date
                
                // Highlight HOLIDAY in print
                const cellStyle = subjectName === 'HOLIDAY' || subjectName === 'HOLIDAY / OFF'
                                  ? 'style="background-color: #fff0f0; font-weight: bold; color: #cc0000; font-style: italic;"' 
                                  : 'style="font-weight: 500;"';
                                  
                datesheetHTML += `<td ${cellStyle}>${subjectName}</td>`;
            });
            
            datesheetHTML += `</tr>`;
        });

        // End table
        datesheetHTML += `
                </tbody>
            </table>
        `;


        // 3. Open Print Window
        const printWindow = window.open('', '', 'height=600,width=800');
        printWindow.document.write('<html><head><title>Datesheet</title>');
        printWindow.document.write('<style>');
        printWindow.document.write(`
            /* --- LANDSCAPE SETTING --- */
            @page { size: A4 landscape; margin: 10mm; } 
            
            body { 
                font-family: Arial, sans-serif; 
                font-size: 9pt; 
                padding: 10px; 
            }
            
            .report-header { text-align: center; }
            .header-table { width: 100%; border-collapse: collapse; margin-bottom: 10px; }
            .header-table td { border: none; padding: 0 5px; vertical-align: middle; }
            .header-table .logo-cell { width: 10%; text-align: left; }
            .header-table .info-cell { width: 70%; text-align: center; }
            .header-table .logo { max-width: 80px; max-height: 60px; width: auto; height: auto; display: block; margin: 0 auto; }

            .datesheet-matrix-table { 
                width: 100%; 
                border-collapse: collapse; 
                page-break-inside: auto; 
                margin-top: 15px; 
                font-size: 9.5pt;
            }
            .datesheet-matrix-table th, .datesheet-matrix-table td { 
                border: 1px solid #000; 
                padding: 5px 8px; 
                text-align: center; 
                vertical-align: middle; 
                height: 35px; /* Ensures uniform height */
            }
            
            .datesheet-matrix-table th { 
                font-weight: bold; 
                font-size: 10pt; 
                color: white;
            }
        `);
        printWindow.document.write('</style>');
        printWindow.document.write('</head><body>');
        
        printWindow.document.write('<div class="print-container">');
        printWindow.document.write(reportHeader); 
        printWindow.document.write(datesheetHTML); 
        
        // SIGNATURES BLOCK REMOVED AS PER USER REQUEST

        printWindow.document.write('</div>'); 
        printWindow.document.write('</body></html>');
        printWindow.document.close();
        
        printWindow.onload = function() {
            const logoElement = printWindow.document.querySelector('.report-header img');
            if (logoElement && settings.logo_path) {
                // Logo path is already correctly set in PHP and passed via JS settings
                logoElement.src = settings.logo_path; 
            }
            printWindow.print();
            printWindow.close(); 
        };
    };


    // -----------------------------------------------------------------------------------
    // 6. Event Listener for the Print Button
    // -----------------------------------------------------------------------------------
    document.querySelectorAll('.btn-print-datesheet').forEach(button => {
        button.addEventListener('click', async function() {
            const examId = this.dataset.examId;
            const examName = this.dataset.examName;
            const startDate = this.dataset.startDate;
            const endDate = this.dataset.endDate;

            // AJAX endpoint call to fetch data
            const fetchUrl = `class_datesheet.php?action=fetch_print_data&exam_id=${examId}`;
            
            $(this).prop('disabled', true).html('<i class="fas fa-spinner fa-spin"></i> Loading...');
            
            try {
                // Fetch the detailed data
                const response = await fetch(fetchUrl);
                const data = await response.json();
                
                if (data.error && Object.keys(data.datesheet_data).length === 0) {
                    alert('Error fetching datesheet data for print: ' + data.error);
                    return;
                }
                
                // Call the printDatesheet function with the fetched data
                window.printDatesheet(data.datesheet_data, {
                    // Use the fetched info for safety, or local data if fetch failed
                    name: data.exam_info.name || examName,
                    start_date: data.exam_info.start_date || startDate, 
                    end_date: data.exam_info.end_date || endDate 
                });
                
            } catch (error) {
                console.error('Fetch error:', error);
                alert('Could not load datesheet data for print preview. Please check network and server logs.');
            } finally {
                 $(this).prop('disabled', false).html('<i class="fas fa-print"></i> Print');
            }
            
        });
    });
    
    // --- Existing JS logic: Other functions (updateDayName, addNewRow, updateFutureSubjects) ---
    
    function updateDayName(inputElement) {
        const dateValue = inputElement.value;
        const row = $(inputElement).closest('tr');
        const dayDisplay = row.find('.day-name-display');
        
        if (dateValue) {
            try {
                const date = new Date(dateValue);
                const dayName = date.toLocaleDateString('en-US', { weekday: 'long' });
                dayDisplay.text(dayName);
            } catch(e) {
                dayDisplay.text('Invalid Date');
            }
        } else {
            dayDisplay.text('-');
        }

        // Update name attributes for the selectors in this row
        row.find('.subject-selector').each(function() {
            const select = this;
            const classId = select.getAttribute('data-class-id');
            // The new name will include the date value
            const newName = dateValue ? `datesheet_data[${classId}][${dateValue}][subject_id]` : '';
            select.name = newName;
        });
        
        // Update subject availability
        updateFutureSubjects();
    }
    window.updateDayName = updateDayName; // Make it globally accessible for onchange in PHP

    function addNewRow() {
        const table = $('#datesheetComposerTable tbody');
        const numRows = table.find('tr').length;
        const newIndex = numRows;
        
        // Get the options HTML (assuming the first row has valid options for cloning)
        let subjectOptions = table.find('tr:first .subject-selector').html();
        if (!subjectOptions) {
             // Fallback if the first row is empty, must ensure options are available
             subjectOptions = '<option value="NONE">-- Select Subject --</option><option value="HOLIDAY">HOLIDAY / OFF</option>';
        }

        // Start HTML for the new row
        let newRowHTML = `<tr id="row_${newIndex}">`;
        
        // Date Column
        newRowHTML += `
            <td class="align-middle bg-light text-center" style="padding: 5px;">
                <input type="date" name="exam_dates[]" class="form-control form-control-sm datesheet-date-input" value="" onchange="updateDayName(this)">
                <span class="badge badge-secondary mt-1 day-name-display">-</span>
            </td>
        `;

        // Subject Columns (loop through classes)
        CLASSES.forEach(classItem => {
            const classId = classItem.id;
            newRowHTML += `
                <td class="p-1">
                    <select 
                        class="form-control form-control-sm subject-selector" 
                        data-row-index="${newIndex}" 
                        data-class-id="${classId}"
                        name=""
                    > 
                        ${subjectOptions} 
                    </select>
                </td>
            `; 
        });
        
        newRowHTML += `</tr>`;
        
        table.append(newRowHTML);

        // New row add hone ke baad, future subject availability update karein
        updateFutureSubjects();
    }
    window.addNewRow = addNewRow;

    // Subject repetition logic
    function updateFutureSubjects() {
        const table = $('#datesheetComposerTable tbody');
        
        table.find('tr').each(function(rowIndex) {
            const currentRow = $(this);
            currentRow.find('.subject-selector').each(function() {
                const selector = $(this);
                const classId = selector.data('class-id');
                
                // 1. Pehle sab options ko enable karein (purana disable state clear)
                selector.find('option').prop('disabled', false);

                // 2. Iterate through all previous rows to find selected subjects
                for (let prevRowIndex = 0; prevRowIndex < rowIndex; prevRowIndex++) {
                    const prevRow = table.find(`tr:eq(${prevRowIndex})`);
                    if (prevRow.length) {
                        // Find the selector in the previous row for the same class
                        const prevSelector = prevRow.find(`.subject-selector[data-class-id="${classId}"]`);
                        const prevSelectedSubjectId = prevSelector.val();
                        
                        // Disable the subject in the current row if it's selected in a previous row (for the same class)
                        if (prevSelectedSubjectId !== 'NONE' && prevSelectedSubjectId !== 'HOLIDAY' && prevSelectedSubjectId) {
                            selector.find(`option[value="${prevSelectedSubjectId}"]`).prop('disabled', true);
                        }
                    }
                }
                
                // 3. Agar current selected value ab disabled ho gayi hai, toh 'NONE' par set karein.
                if (selector.val() !== 'NONE' && selector.val() !== 'HOLIDAY' && selector.find(`option[value="${selector.val()}"]`).is(':disabled')) {
                    selector.val('NONE');
                }
            });
        });
    }
    
    // Initial load par future subject availability update karein (for existing data)
    updateFutureSubjects(); 
    
    // Attach event listener for subject change
    $('#datesheetComposerTable').on('change', '.subject-selector', function() {
        updateFutureSubjects();
    });
    
    // Attach event listener for date change 
    $('#datesheetComposerTable').on('change', '.datesheet-date-input', function(event) {
        updateDayName(event.target);
        updateFutureSubjects();
    });
    
});
</script>

<?php 
// Connection close karein
if (isset($conn) && $conn && method_exists($conn, 'close')) {
    @$conn->close();
}
// Assuming includes/footer.php exists
include_once('includes/footer.php'); 
?>