<?php
// FILE: admin/class_planner.php - Class Weekly Lesson Planner (Grid View with Weekly List Management)
// FINAL FIXES: 
// 1. Implemented AJAX endpoint for fetching weekly print data.
// 2. Updated Weekly Summary Table to have separate Print, Edit, and Delete buttons.
// 3. Print button now opens the print pop-up directly without redirecting to the grid view.
// 4. Added Bulk Delete logic for a weekly plan from the summary table.
// 5. CHANGED: Saturday holiday removed, Sunday holiday implemented

session_start();

include_once('../config/db.php'); 
$conn = getDBConnection(); 

// Initialize variables
$message = $_GET['msg'] ?? '';
$error = $_GET['err'] ?? '';

// Default values for form
$selected_class_id = (int)($_POST['class_id'] ?? $_GET['class_id'] ?? 0);
$selected_section_id = (int)($_POST['section_id'] ?? $_GET['section_id'] ?? 0);
$start_date_input = $_POST['start_date'] ?? $_GET['start_date'] ?? date('Y-m-d', strtotime('this week monday'));
$end_date_input = $_POST['end_date'] ?? $_GET['end_date'] ?? date('Y-m-d', strtotime('this week saturday'));

// Helper function to find class/section name for display
function find_name_by_id($array, $id, $type) {
    $name_key = ($type == 'class') ? 'class_name' : 'section_name';
    foreach ($array as $item) {
        if ($item['id'] == $id) {
            return $item[$name_key] ?? 'N/A';
        }
    }
    return 'N/A';
}

// --------------------------------------------------------------------------------
// 0. NEW: AJAX Endpoint for Print Preview Data Fetch
// --------------------------------------------------------------------------------
if (isset($_GET['action']) && $_GET['action'] === 'fetch_print_data') {
    header('Content-Type: application/json');
    
    $class_id = (int)($_GET['class_id'] ?? 0);
    $section_id = (int)($_GET['section_id'] ?? 0);
    $start_date = $_GET['start_date'] ?? '';
    $end_date = $_GET['end_date'] ?? '';
    
    $response = ['subjects' => [], 'planner_data' => [], 'error' => null];

    if ($class_id && $start_date && $end_date) {
        
        // A. Fetch Subjects assigned to the class
        $subject_q = $conn->query("
            SELECT s.id, s.name AS subject_name 
            FROM subjects s
            JOIN course_allocation ca ON s.id = ca.subject_id
            WHERE ca.class_id = {$class_id}
            GROUP BY s.id, s.name
            ORDER BY s.name ASC
        ");
        if ($subject_q) {
            while ($row = $subject_q->fetch_assoc()) {
                $response['subjects'][] = $row;
            }
        } else {
            $response['error'] = "Error fetching subjects: " . $conn->error;
            echo json_encode($response);
            exit;
        }

        // B. Fetch Planner Data for the week
        $section_where_clause = ($section_id == 0) ? 
            "AND (section_id IS NULL OR section_id = 0)" : 
            "AND section_id = " . $section_id;
            
        $planner_q = $conn->query("
            SELECT subject_id, planner_date, cw_content, hw_content 
            FROM daily_planner 
            WHERE class_id = {$class_id} 
            {$section_where_clause}
            AND planner_date BETWEEN '{$start_date}' AND '{$end_date}'
        ");
        if ($planner_q) {
             while ($row = $planner_q->fetch_assoc()) {
                 $key = $row['planner_date'] . '_' . $row['subject_id'];
                 $response['planner_data'][$key] = [
                     'cw' => $row['cw_content'],
                     'hw' => $row['hw_content']
                 ];
             }
        } else {
            $response['error'] = "Error fetching planner data: " . $conn->error;
        }
        
    } else {
        $response['error'] = "Missing required parameters (class_id, start_date, end_date).";
    }

    echo json_encode($response);
    exit;
}

// --------------------------------------------------------------------------------
// 1. DELETE LOGIC (Executed first to allow redirects)
// --------------------------------------------------------------------------------

// 1A. Single entry delete (Kept for integrity, though not used in new summary table)
if (isset($_GET['action']) && $_GET['action'] === 'delete' && isset($_GET['planner_id'])) {
    $planner_id = (int)$_GET['planner_id'];
    
    // Default redirect parameters to maintain current view
    $redirect_class_id = (int)($_GET['class_id'] ?? $selected_class_id);
    $redirect_section_id = (int)($_GET['section_id'] ?? $selected_section_id);
    $redirect_start_date = $_GET['start_date'] ?? $start_date_input;
    $redirect_end_date = $_GET['end_date'] ?? $end_date_input;
    
    $delete_q = $conn->prepare("DELETE FROM daily_planner WHERE id = ?");
    if ($delete_q === false) {
         $error = "Prepare failed: (" . $conn->errno . ") " . $conn->error;
    } else {
        $delete_q->bind_param("i", $planner_id);
        if ($delete_q->execute()) {
            $message = "Planner entry (ID: $planner_id) successfully deleted.";
        } else {
            $error = "Error deleting planner entry: " . $delete_q->error;
        }
        $delete_q->close();
    }
    
    // Redirect back to the weekly grid's filtered view
    header("Location: class_planner.php?class_id={$redirect_class_id}&section_id={$redirect_section_id}&start_date={$redirect_start_date}&end_date={$redirect_end_date}&msg=" . urlencode($message) . "&err=" . urlencode($error));
    exit;
}

// 1B. NEW: BULK DELETE LOGIC (Deletes ALL entries for a Weekly Plan)
if (isset($_GET['action']) && $_GET['action'] === 'delete_week') {
    $del_class_id = (int)($_GET['class_id'] ?? 0);
    $del_section_id = (int)($_GET['section_id'] ?? 0);
    $del_start_date = $_GET['start_date'] ?? '';
    $del_end_date = $_GET['end_date'] ?? '';
    
    if ($del_class_id && $del_start_date && $del_end_date) {
        
        $section_where_clause = ($del_section_id == 0) ? 
            "AND (section_id IS NULL OR section_id = 0)" : 
            "AND section_id = " . $del_section_id;

        $delete_sql = "
            DELETE FROM daily_planner 
            WHERE class_id = ? 
            {$section_where_clause}
            AND planner_date BETWEEN ? AND ?
        ";
        
        $delete_stmt = $conn->prepare($delete_sql);
        if ($delete_stmt) {
             // Bind parameters: integer, string, string (for class_id, start_date, end_date)
             $delete_stmt->bind_param("iss", $del_class_id, $del_start_date, $del_end_date);

            if ($delete_stmt->execute()) {
                $affected_rows = $delete_stmt->affected_rows;
                $message = "Successfully deleted **$affected_rows** planner entries for the week: $del_start_date to $del_end_date.";
            } else {
                $error = "Error deleting weekly plan: " . $delete_stmt->error;
            }
            $delete_stmt->close();
        } else {
            $error = "Delete prepare failed: " . $conn->error;
        }
    } else {
        $error = "Invalid parameters for weekly plan deletion.";
    }
    
    // Redirect back to the summary list
    header("Location: class_planner.php?msg=" . urlencode($message) . "&err=" . urlencode($error));
    exit;
}
// --------------------------------------------------------------------------------

// --------------------------------------------------------------------------------
// 2. DATA PROCESSING (Weekly Grid Form Submission: SAVE/UPDATE/DELETE)
// --------------------------------------------------------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_planner'])) {
    
    // 1. Extract common data (Logic is kept identical to previous working version)
    $class_id = (int)($_POST['class_id'] ?? 0);
    $section_id = (int)($_POST['section_id'] ?? 0);
    $cw_data = $_POST['cw_planner_data'] ?? [];
    $hw_data = $_POST['hw_planner_data'] ?? [];
    
    $count = 0;
    $error_occurred = false;

    // 2. Prepare the combined data for processing
    $all_keys = array_unique(array_merge(array_keys($cw_data), array_keys($hw_data)));
    
    // 3. Loop through all date_subject keys (Save/Update/Delete logic)
    foreach ($all_keys as $key) {
        list($planner_date, $subject_id) = explode('_', $key);
        $subject_id = (int)$subject_id;
        
        $cw_content = trim($cw_data[$key] ?? '');
        $hw_content = trim($hw_data[$key] ?? '');
        
        $db_section_id = $section_id > 0 ? $section_id : 0;
        // Null or empty string for the database (using NULL if field is empty)
        $db_cw_content = $cw_content ?: null; 
        $db_hw_content = $hw_content ?: null; 

        if ($class_id > 0 && $subject_id > 0 && $planner_date) {
            
            // --- Check for existing entry ---
            $check_where_section = ($db_section_id === 0) ? "AND (section_id IS NULL OR section_id = 0)" : "AND section_id = ?";

            $check_sql = "
                SELECT id FROM daily_planner 
                WHERE planner_date = ? AND class_id = ? AND subject_id = ? 
                {$check_where_section}
            ";

            $check_stmt = $conn->prepare($check_sql);
            
            if ($check_stmt) {
                if ($db_section_id === 0) {
                    $check_stmt->bind_param("sii", $planner_date, $class_id, $subject_id);
                } else {
                    $check_stmt->bind_param("siii", $planner_date, $class_id, $subject_id, $db_section_id);
                }
                
                $check_stmt->execute();
                $check_result = $check_stmt->get_result();
                $existing_entry = $check_result->fetch_assoc();
                $check_stmt->close();
            } else { 
                 $error .= " (Check Prepare Failed: " . $conn->error . ")";
                 $error_occurred = true;
                 continue;
            }


            if (!empty($cw_content) || !empty($hw_content)) {
                if ($existing_entry) {
                    // UPDATE Logic
                    $update_sql = "UPDATE daily_planner SET cw_content = ?, hw_content = ? WHERE id = ?";
                    $update_stmt = $conn->prepare($update_sql);
                    if ($update_stmt) {
                        $update_stmt->bind_param("ssi", $db_cw_content, $db_hw_content, $existing_entry['id']);
                        $update_stmt->execute();
                        $update_stmt->close();
                        $count++;
                    } else { 
                        $error .= " (Update Prepare Failed: " . $conn->error . ")";
                        $error_occurred = true;
                    }
                } else {
                    // INSERT Logic
                    $insert_sql = "INSERT INTO daily_planner (planner_date, class_id, subject_id, section_id, cw_content, hw_content) VALUES (?, ?, ?, ?, ?, ?)";
                    $insert_stmt = $conn->prepare($insert_sql);
                    if ($insert_stmt) {
                        $insert_stmt->bind_param("siiiss", $planner_date, $class_id, $subject_id, $db_section_id, $db_cw_content, $db_hw_content);
                        $insert_stmt->execute();
                        $insert_stmt->close();
                        $count++;
                    } else { 
                        $error .= " (Insert Prepare Failed: " . $conn->error . ")";
                        $error_occurred = true;
                    }
                }
            } else if ($existing_entry) {
                 // DELETE if both fields are made empty
                $delete_sql = "DELETE FROM daily_planner WHERE id = ?";
                $delete_stmt = $conn->prepare($delete_sql);
                if ($delete_stmt) {
                    $delete_stmt->bind_param("i", $existing_entry['id']);
                    $delete_stmt->execute();
                    $delete_stmt->close();
                    $count++;
                } else { 
                    $error .= " (Delete Prepare Failed: " . $conn->error . ")"; 
                    $error_occurred = true;
                }
            }
        }
    }
    
    $message = "$count planner entries successfully processed (saved/updated/deleted)." . ($error_occurred ? " However, some errors occurred." : ""); 
    
    // Redirect back to the weekly grid's filtered view
    header("Location: class_planner.php?class_id={$class_id}&section_id={$section_id}&start_date={$_POST['start_date']}&end_date={$_POST['end_date']}&msg=" . urlencode($message) . "&err=" . urlencode($error));
    exit;
} 
// --------------------------------------------------------------------------------

// --- START: Settings Fetch Logic (FOR PRINTING) ---
$settings_data = [];
$settings_query = $conn->query("SELECT setting_key, setting_value FROM settings");
if ($settings_query) {
    while ($row = $settings_query->fetch_assoc()) {
        $settings_data[trim($row['setting_key'])] = trim($row['setting_value']);
    }
}
$institute_name = $settings_data['institute.name'] ?? "SCHOOL NAME (UPDATE SETTINGS)";
$institute_address = $settings_data['institute.address'] ?? "Address Not Set";
$institute_phone = $settings_data['institute.phone'] ?? "Phone Not Set";

$raw_logo_path = $settings_data['institute.logo_path'] ?? '';
$logoPathParts = explode('|', $raw_logo_path);
$logo_filename = end($logoPathParts); 
$logo_path = $logo_filename ? '../uploads/' . basename($logo_filename) : ''; 
// --- END: Settings Fetch Logic ---


// --- Fetch all Classes and Sections for the dropdowns ---
$classes = [];
$sections = [];

// 1. Fetch Classes (from academic_classes)
$class_q = $conn->query("SELECT id, name AS class_name FROM academic_classes ORDER BY order_no ASC");
if ($class_q) {
    while ($row = $class_q->fetch_assoc()) {
        $classes[] = $row;
    }
}

// 2. Fetch Sections (from academic_sections)
$section_q = $conn->query("SELECT id, name AS section_name FROM academic_sections WHERE status='active' ORDER BY name ASC");
if ($section_q) {
    while ($row = $section_q->fetch_assoc()) {
        $sections[] = $row;
    }
}

// --- Fetch Existing Planner Data for Weekly Summary List ---
$list_weekly_plans = [];

$list_sql = "
    SELECT 
        dp.class_id, 
        dp.section_id, 
        ac.name AS class_name, 
        asec.name AS section_name, 
        MIN(dp.planner_date) AS start_date, 
        MAX(dp.planner_date) AS end_date,
        COUNT(dp.id) AS total_entries
    FROM daily_planner dp
    JOIN academic_classes ac ON dp.class_id = ac.id
    LEFT JOIN academic_sections asec ON dp.section_id = asec.id
    GROUP BY dp.class_id, dp.section_id, ac.name, asec.name
    ORDER BY end_date DESC, class_name ASC
    LIMIT 100 
";

if ($conn) {
    $list_result = $conn->query($list_sql);
    if ($list_result) {
        $list_weekly_plans = $list_result->fetch_all(MYSQLI_ASSOC);
    } else {
        $error .= " Failed to fetch recent planner list: " . $conn->error;
    }
}
// -----------------------------------------------------------


// --- Fetch ALL Subject Names (Needed for Print Pop-up Headers) ---
$all_subjects = [];
$all_subjects_q = $conn->query("SELECT id, name AS subject_name FROM subjects ORDER BY name ASC");
if ($all_subjects_q) {
    while ($row = $all_subjects_q->fetch_assoc()) {
        // Store subjects as a list for JS to use (for print pop-up)
        $all_subjects[] = ['id' => $row['id'], 'subject_name' => $row['subject_name']];
    }
}

// --------------------------------------------------------------------------------
// 3. DATA FETCHING (For Weekly Grid Table Display)
// --------------------------------------------------------------------------------
$subjects = [];
$existing_planner_data = [];
$dates = [];

if ($selected_class_id && $selected_section_id) {
    
    // A. Fetch Subjects (Only those assigned to the selected class)
    $subject_q_text = "
        SELECT s.id, s.name AS subject_name 
        FROM subjects s
        -- NOTE: 'course_allocation' is used as the linking table name. 
        JOIN course_allocation ca ON s.id = ca.subject_id
        WHERE ca.class_id = " . (int)$selected_class_id . "
        GROUP BY s.id, s.name
        ORDER BY s.name ASC
    ";
    
    $subject_result = $conn->query($subject_q_text);
    if ($subject_result) {
        while ($row = $subject_result->fetch_assoc()) {
            $subjects[] = $row;
        }
    } else {
        $error .= " Error fetching subjects (Please check 'course_allocation' table). SQL Error: " . $conn->error;
    }

    // B. Generate Dates Array
    $current_date = new DateTime($start_date_input);
    $end_date = new DateTime($end_date_input);
    $end_date->modify('+1 day'); 
    while ($current_date < $end_date) {
        $dates[] = $current_date->format('Y-m-d');
        $current_date->modify('+1 day');
    }

    // C. Fetch Existing Planning Data for the Grid
    if (!empty($dates)) {
        // Section ID handling: 0 or NULL sections are treated the same if section_id is 0
        $section_where_clause = ($selected_section_id == 0) ? 
            "AND (section_id IS NULL OR section_id = 0)" : 
            "AND section_id = " . (int)$selected_section_id;
            
        $planner_q = $conn->query("
            SELECT subject_id, planner_date, cw_content, hw_content 
            FROM daily_planner 
            WHERE class_id = " . (int)$selected_class_id . " 
            {$section_where_clause}
            AND planner_date BETWEEN '{$start_date_input}' AND '{$end_date_input}'
        ");
        if ($planner_q) {
             while ($row = $planner_q->fetch_assoc()) {
                 $key = $row['planner_date'] . '_' . $row['subject_id'];
                 $existing_planner_data[$key] = [
                     'cw' => $row['cw_content'],
                     'hw' => $row['hw_content']
                 ];
             }
        }
    }
}


$class_name_display = find_name_by_id($classes, $selected_class_id, 'class');
$section_name_display = find_name_by_id($sections, $selected_section_id, 'section');

// --- INCLUDE HEADER NOW ---
include_once('includes/header.php'); 

?>

<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <h1 class="h3 mb-4 text-gray-800">✍️ Class Weekly Lesson Planner (Grid View)</h1>
        </div>
    </div>

    <?php if ($message): ?>
        <div class="alert alert-success"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>
    <?php if ($error): ?>
        <div class="alert alert-danger"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>
    
    <div class="card shadow mb-4">
        <div class="card-header py-3">
            <h6 class="m-0 font-weight-bold text-primary">Select Class and Date Range</h6>
        </div>
        <div class="card-body">
            <form id="planner_selection_form" method="POST" action="class_planner.php">
                <div class="form-row">
                    <div class="col-md-3 mb-3">
                        <label for="class_id">Class</label>
                        <select name="class_id" id="class_id" class="form-control" required>
                            <option value="">Select Class</option>
                            <?php foreach ($classes as $class): ?>
                                <option value="<?php echo $class['id']; ?>" <?php echo $selected_class_id == $class['id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($class['class_name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="col-md-2 mb-3">
                        <label for="section_id">Section</label>
                        <select name="section_id" id="section_id" class="form-control" required>
                            <option value="">Select Section</option>
                            <?php foreach ($sections as $section): ?>
                                <option value="<?php echo $section['id']; ?>" <?php echo $selected_section_id == $section['id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($section['section_name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="col-md-3 mb-3">
                        <label for="date_range_preset">Month/Week Preset</label>
                        <select id="date_range_preset" class="form-control">
                            <option value="">Custom Range</option>
                            <option value="current_week">Current Week (Mon - Sat)</option>
                            <option value="next_week">Next Week (Mon - Sat)</option>
                        </select>
                    </div>

                    <div class="col-md-2 mb-3">
                        <label for="start_date">Start Date</label>
                        <input type="date" name="start_date" id="start_date" class="form-control" value="<?php echo htmlspecialchars($start_date_input); ?>" required>
                    </div>
                    
                    <div class="col-md-2 mb-3">
                        <label for="end_date">End Date</label>
                        <input type="date" name="end_date" id="end_date" class="form-control" value="<?php echo htmlspecialchars($end_date_input); ?>" required>
                    </div>
                </div>

                <button type="submit" class="btn btn-primary">Generate Planner Table</button>
            </form>
        </div>
    </div>

    <?php if ($selected_class_id && $selected_section_id && !empty($subjects) && !empty($dates)): ?>
        <div class="card shadow mb-4">
            <div class="card-header py-3 d-flex justify-content-between align-items-center">
                <h6 class="m-0 font-weight-bold text-success">
                    Planner for **Class <?php echo htmlspecialchars($class_name_display); ?> (<?php echo htmlspecialchars($section_name_display); ?>)**
                </h6>
                <button type="button" class="btn btn-info btn-sm" onclick="printCurrentGrid()">
                    <i class="fas fa-print"></i> **Print Weekly Study Plan**
                </button>
            </div>
            <div class="card-body">
                <form method="POST" action="class_planner.php" id="planner_form">
                    <input type="hidden" name="class_id" value="<?php echo $selected_class_id; ?>">
                    <input type="hidden" name="section_id" value="<?php echo $selected_section_id; ?>">
                    <input type="hidden" name="start_date" value="<?php echo $start_date_input; ?>">
                    <input type="hidden" name="end_date" value="<?php echo $end_date_input; ?>">
                    
                    <div class="table-responsive">
                        <table class="table table-bordered table-striped" id="plannerTable" width="100%" cellspacing="0">
                            <thead>
                                <tr>
                                    <th style="width: 120px; min-width: 120px;">Date / Day</th>
                                    <?php foreach ($subjects as $subject): ?>
                                        <th style="min-width: 250px;"><?php echo htmlspecialchars($subject['subject_name']); ?></th>
                                    <?php endforeach; ?>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($dates as $date): 
                                    $day_name = date('l', strtotime($date)); 
                                    // CHANGED: Saturday holiday removed, Sunday holiday implemented
                                    $is_weekend = in_array(date('D', strtotime($date)), ['Sun']);
                                    $row_class = $is_weekend ? 'table-warning' : '';
                                ?>
                                <tr class="<?php echo $row_class; ?>">
                                    <td class="small" style="text-align: center;">
                                        <div style="font-weight: bold;"><?php echo date('d-M-Y', strtotime($date)); ?></div>
                                        <div class="text-primary" style="font-size: 0.9em;"><?php echo $day_name; ?></div> 
                                    </td>
                                    <?php foreach ($subjects as $subject): 
                                        $key = $date . '_' . $subject['id'];
                                        $cw_content = htmlspecialchars($existing_planner_data[$key]['cw'] ?? '');
                                        $hw_content = htmlspecialchars($existing_planner_data[$key]['hw'] ?? '');

                                        $cw_input_name = "cw_planner_data[{$date}_{$subject['id']}]";
                                        $hw_input_name = "hw_planner_data[{$date}_{$subject['id']}]";
                                    ?>
                                        <td style="text-align: center;"> 
                                            <?php if (!$is_weekend): ?>
                                                <div class="mb-2">
                                                    <label class="small font-weight-bold text-success">C.W.</label>
                                                    <textarea 
                                                        name="<?php echo $cw_input_name; ?>" 
                                                        class="form-control form-control-sm planner-input" 
                                                        rows="2" 
                                                        placeholder="Class Work/Topic"
                                                        style="text-align: center;" 
                                                    ><?php echo $cw_content; ?></textarea>
                                                </div>
                                                <div>
                                                    <label class="small font-weight-bold text-danger">H.W.</label>
                                                    <textarea 
                                                        name="<?php echo $hw_input_name; ?>" 
                                                        class="form-control form-control-sm planner-input" 
                                                        rows="2" 
                                                        placeholder="Home Work/Assignment"
                                                        style="text-align: center;"
                                                    ><?php echo $hw_content; ?></textarea>
                                                </div>
                                            <?php else: ?>
                                                <span class="text-muted small">Weekend / No Planning</span>
                                            <?php endif; ?>
                                        </td>
                                    <?php endforeach; ?>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>

                    <button type="submit" name="save_planner" class="btn btn-success mt-3"><i class="fas fa-save"></i> Save Daily Planner</button>
                </form>
            </div>
        </div>
    <?php elseif ($selected_class_id && empty($subjects)): ?>
        <div class="alert alert-danger">
            No subjects are assigned to Class: **<?php echo htmlspecialchars($class_name_display); ?>**. 
            Please assign subjects using your **Subject Allocation** interface.
        </div>
    <?php endif; ?>
    
    
    <hr class="mt-5 mb-4">
    
    <div class="card shadow mb-4">
        <div class="card-header py-3">
            <h6 class="m-0 font-weight-bold text-info">
                🗓️ Recent Weekly Plans Summary (Total: <?php echo count($list_weekly_plans); ?>)
            </h6>
        </div>
        <div class="card-body">
            
            <?php if (!empty($list_weekly_plans)): ?>
            
            <div class="table-responsive">
                <table class="table table-bordered table-striped small" id="weeklyPlanSummaryTable" width="100%" cellspacing="0">
                    <thead>
                        <tr>
                            <th style="width: 25%;">Planner Date Range</th>
                            <th style="width: 15%;">Class</th>
                            <th style="width: 15%;">Section</th>
                            <th style="width: 10%;">Entries</th>
                            <th style="width: 35%;">Action</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($list_weekly_plans as $plan): 
                            $date_range = date('d-M-Y', strtotime($plan['start_date'])) . ' - ' . date('d-M-Y', strtotime($plan['end_date']));
                            
                            // Edit URL: Redirects to the weekly grid's filtered view
                            $edit_url = "class_planner.php?class_id={$plan['class_id']}&section_id={$plan['section_id']}&start_date={$plan['start_date']}&end_date={$plan['end_date']}";
                            
                            // Delete URL: Uses the new 'delete_week' action
                            $delete_url = "class_planner.php?action=delete_week&class_id={$plan['class_id']}&section_id={$plan['section_id']}&start_date={$plan['start_date']}&end_date={$plan['end_date']}";
                        ?>
                        <tr>
                            <td>**<?php echo htmlspecialchars($date_range); ?>**</td>
                            <td><?php echo htmlspecialchars($plan['class_name']); ?></td>
                            <td><?php echo htmlspecialchars($plan['section_name'] ?? 'Default'); ?></td>
                            <td><?php echo $plan['total_entries']; ?></td>
                            <td class="text-nowrap">
                                
                                <button 
                                    type="button" 
                                    class="btn btn-sm btn-success btn-print-weekly"
                                    data-class-id="<?php echo $plan['class_id']; ?>"
                                    data-section-id="<?php echo $plan['section_id']; ?>"
                                    data-start-date="<?php echo $plan['start_date']; ?>"
                                    data-end-date="<?php echo $plan['end_date']; ?>"
                                    data-class-name="<?php echo htmlspecialchars($plan['class_name']); ?>"
                                    data-section-name="<?php echo htmlspecialchars($plan['section_name'] ?? 'Default'); ?>"
                                    title="View & Print Preview of this Weekly Plan"
                                >
                                    <i class="fas fa-print"></i> **Print Preview**
                                </button>
                                
                                <a 
                                    href="<?php echo $edit_url; ?>" 
                                    class="btn btn-sm btn-primary"
                                    title="Load this weekly plan into the grid above for editing"
                                >
                                    <i class="fas fa-edit"></i> **Edit**
                                </a>
                                
                                <a 
                                    href="<?php echo $delete_url; ?>" 
                                    class="btn btn-sm btn-danger btn-delete-week"
                                    onclick="return confirm('Are you sure you want to delete ALL planner entries for the week: <?php echo date('d-M-Y', strtotime($plan['start_date'])); ?> to <?php echo date('d-M-Y', strtotime($plan['end_date'])); ?>? This action cannot be undone.');"
                                    title="Delete all entries for this specific week"
                                >
                                    <i class="fas fa-trash"></i> **Delete**
                                </a>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            <?php else: ?>
                <div class="alert alert-warning">No weekly planner entries found.</div>
            <?php endif; ?>
        </div>
    </div>
    
</div>

<?php 
// PHP variables to pass to JS (Only printPlanner function needs this)
$js_settings = [
    'institute_name' => $institute_name,
    'institute_address' => $institute_address,
    'institute_phone' => $institute_phone,
    'logo_path' => $logo_path,
    // Current grid's selected info (for the top print button)
    'class_name' => $class_name_display, 
    'section_name' => $section_name_display,
    'start_date' => $start_date_input, // YYYY-MM-DD
    'end_date' => $end_date_input,   // YYYY-MM-DD
    'subjects_list' => $all_subjects // Full list of all subjects for print pop-up headers
];

// Current grid data (for the top print button)
$current_grid_data = [
    'subjects' => $subjects, // Subjects relevant to the current grid's class
    'planner_data' => $existing_planner_data // Data for the current grid's date range
];
?>

<script>
// Required for jQuery selection/dom manipulation
const PLANNER_SETTINGS = <?php echo json_encode($js_settings); ?>;
const CURRENT_GRID_DATA = <?php echo json_encode($current_grid_data); ?>;

document.addEventListener('DOMContentLoaded', function() {
    const presetSelect = document.getElementById('date_range_preset');
    const startDateInput = document.getElementById('start_date');
    const endDateInput = document.getElementById('end_date');
    const plannerForm = document.getElementById('planner_selection_form');

    // Function to format date as YYYY-MM-DD
    function formatDate(date) {
        return date.toISOString().split('T')[0];
    }
    
    // Function to format date as d-M-Y for printing display
    const formatPrintDate = (dateString) => {
        if (!dateString) return 'N/A';
        const date = new Date(dateString);
        return date.toLocaleDateString('en-GB', { day: '2-digit', month: 'short', year: 'numeric' });
    };

    // Function to calculate start/end dates based on preset
    function updateDateRange(preset) {
        let startDate, endDate;
        const today = new Date();
        const day = today.getDay(); // 0 is Sunday, 1 is Monday

        if (preset === 'current_week') {
            let diff = today.getDate() - day + (day === 0 ? -6 : 1);
            startDate = new Date(today.setDate(diff));
            endDate = new Date(startDate);
            endDate.setDate(startDate.getDate() + 5); // Saturday
        } else if (preset === 'next_week') {
            let diff = today.getDate() - day + 8;
            startDate = new Date(today.setDate(diff));
            endDate = new Date(startDate);
            endDate.setDate(startDate.getDate() + 5); // Saturday
        }
        
        if (startDate && endDate) {
            startDateInput.value = formatDate(startDate);
            endDateInput.value = formatDate(endDate);
            
            const classId = document.getElementById('class_id').value;
            const sectionId = document.getElementById('section_id').value;
            if (classId && sectionId) {
                 // Auto submit only if Class and Section are already selected
                 plannerForm.submit();
            }
        }
    }

    // Event Listener for Preset changes
    presetSelect.addEventListener('change', function() {
        if (this.value) {
            updateDateRange(this.value);
        }
    });


    // -----------------------------------------------------------------------------------
    // 1. Core Print Function (Modified to accept data)
    // -----------------------------------------------------------------------------------
    window.printPlanner = function(subjects, plannerData, overrides) {
        
        // Use global settings as defaults
        const settings = PLANNER_SETTINGS;
        
        // Apply overrides for the specific weekly plan
        const class_name = overrides.class_name || settings.class_name;
        const section_name = overrides.section_name || settings.section_name;
        
        // Date formatting
        const start_date_display = formatPrintDate(overrides.start_date || settings.start_date);
        const end_date_display = formatPrintDate(overrides.end_date || settings.end_date);
        
        
        // 1. Generate Print Dates Array
        const dates = [];
        let currentDate = new Date(overrides.start_date);
        const endDate = new Date(overrides.end_date);
        endDate.setDate(endDate.getDate() + 1); // For loop condition

        while (currentDate < endDate) {
            dates.push(formatDate(currentDate)); // formatDate is the YYYY-MM-DD helper
            currentDate.setDate(currentDate.getDate() + 1);
        }
        
        if(subjects.length === 0) {
             alert(`Cannot print: No subjects are assigned to Class ${class_name}.`);
             return;
        }

        // 2. Generate Report Header
        const reportHeader = `
            <div class="report-header">
                <table class="header-table">
                    <tr>
                        <td class="logo-cell">
                            ${settings.logo_path ? `<img src="${settings.logo_path}" alt="School Logo" class="logo">` : ''}
                        </td>
                        <td class="info-cell">
                            <h2 style="margin: 0; font-size: 16pt; font-weight: bold;">${settings.institute_name}</h2>
                            <p style="margin: 0; font-size: 10pt;">${settings.institute_address} | Phone: ${settings.institute_phone}</p>
                        </td>
                        <td style="width: 20%;"></td> 
                    </tr>
                </table>
                <hr style="border: 1px solid #000; margin-top: 5px;">
                <h3 style="margin: 5px 0 5px; font-size: 14pt; text-align: center;">Weekly Study Plan</h3>
                <p style="margin: 0; font-size: 11pt; font-weight: bold; text-align: center;">
                    Class: ${class_name} (${section_name})
                    (Date: ${start_date_display} to ${end_date_display})
                    </p>
                <hr style="border: 1px dashed #999; margin-bottom: 10px;">
            </div>
        `;

        // 3. Generate Table Structure
        const date_column_print_width = 20; 
        const subject_columns_combined_width = 100 - date_column_print_width;
        const subject_column_print_width = subject_columns_combined_width / subjects.length;
        
        const headerRowHTML = `
            <tr>
                <th style="width: ${date_column_print_width}%;">Date / Day</th> 
                ${subjects.map(subject => `<th style="width: ${subject_column_print_width}%;">${subject.subject_name}</th>`).join('')}
            </tr>
        `;

        let cwTableBodyHTML = '';
        let hwTableBodyHTML = '';

        dates.forEach(date => {
            const dateObj = new Date(date);
            const day_name = dateObj.toLocaleDateString('en-US', { weekday: 'long' });
            // CHANGED: Saturday holiday removed, Sunday holiday implemented
            const is_weekend = (dateObj.getDay() === 0); // 0=Sun only
            const rowClass = is_weekend ? 'table-warning' : '';

            const dateDayCellHTML = `
                <div style="font-weight: bold;">${formatPrintDate(date)}</div>
                <div class="text-primary" style="font-size: 0.9em;">${day_name}</div>
            `;
            
            let cwRowContent = '';
            let hwRowContent = '';

            if (is_weekend) {
                const weekendHtml = `<td colspan="${subjects.length}" style="text-align: center;"><span class="text-muted small">Weekend / No Planning</span></td>`;
                cwRowContent = weekendHtml;
                hwRowContent = weekendHtml;
            } else {
                subjects.forEach(subject => {
                    const key = `${date}_${subject.id}`;
                    // plannerData format: { "YYYY-MM-DD_SubjectID": { cw: "...", hw: "..." } }
                    const entry = plannerData[key] || {};
                    const cwContent = (entry.cw || '').trim().replace(/\n/g, '<br>');
                    const hwContent = (entry.hw || '').trim().replace(/\n/g, '<br>');
                    
                    cwRowContent += `<td><span class="planner-content-print">${cwContent}</span></td>`;
                    hwRowContent += `<td><span class="planner-content-print">${hwContent}</span></td>`;
                });
            }
            
            cwTableBodyHTML += `<tr class="${rowClass}"><td>${dateDayCellHTML}</td>${cwRowContent}</tr>`;
            hwTableBodyHTML += `<tr class="${rowClass}"><td>${dateDayCellHTML}</td>${hwRowContent}</tr>`;
        });
        
        const cwTableHTML = `
            <h4 style="margin: 20px 0 10px; font-size: 13pt; text-align: left; color: green;">I. C.W. (Class Work) Plan</h4>
            <table class="report-table" width="100%" cellspacing="0">
                <thead>${headerRowHTML}</thead>
                <tbody>${cwTableBodyHTML}</tbody>
            </table>
        `;

        const hwTableHTML = `
            <h4 style="margin: 30px 0 10px; font-size: 13pt; text-align: left; color: red;">II. H.W. (Home Work/Assignment) Plan</h4>
            <table class="report-table" width="100%" cellspacing="0">
                <thead>${headerRowHTML}</thead>
                <tbody>${hwTableBodyHTML}</tbody>
            </table>
        `;
        
        // 4. Open Print Window
        const printWindow = window.open('', '', 'height=600,width=800');
        printWindow.document.write('<html><head><title>Weekly Study Plan</title>');
        printWindow.document.write('<style>');
        printWindow.document.write(`
            @page { size: A4 portrait; margin: 5mm; } 
            
            body { font-family: Arial, sans-serif; font-size: 9pt; padding: 10px; }
            
            .header-table { width: 100%; border-collapse: collapse; margin-bottom: 10px; }
            .header-table td { border: none; padding: 0 5px; vertical-align: middle; }
            .header-table .logo-cell { width: 20%; text-align: left; }
            .header-table .info-cell { width: 60%; text-align: center; }
            .header-table .logo { max-width: 100px; max-height: 80px; width: auto; height: auto; display: block; }

            table { width: 100%; border-collapse: collapse; page-break-inside: auto; margin-top: 10px; }
            th, td { border: 1px solid #000; padding: 5px; text-align: center; vertical-align: top; }
            
            th { 
                background-color: #f2f2f2 !important; 
                font-weight: bold; 
                font-size: 8pt; 
            }

            .report-table tbody tr td:first-child > div:first-child {
                font-size: 8pt !important; 
                line-height: 1.1;
                font-weight: bold;
            }
            .report-table tbody tr td:first-child > div:last-child {
                font-size: 8pt !important; 
                line-height: 1.1;
            }
            
            .planner-content-print {
                white-space: pre-wrap;
                word-wrap: break-word;
                display: block;
                min-height: 20px; 
                font-size: 8pt; 
                line-height: 1.2;
                text-align: center; 
            }
        `);
        printWindow.document.write('</style>');
        printWindow.document.write('</head><body>');
        
        printWindow.document.write('<div class="print-container">');
        printWindow.document.write(reportHeader); 
        
        printWindow.document.write(cwTableHTML); 
        printWindow.document.write(hwTableHTML); 

        printWindow.document.write('<div style="margin-top: 50px; display: flex; justify-content: space-around; width: 90%; margin: 50px auto 0; page-break-before: avoid;">');

        printWindow.document.write('</div>');
        
        printWindow.document.write('</div>'); 
        printWindow.document.write('</body></html>');
        printWindow.document.close();
        
        printWindow.onload = function() {
            const logoElement = printWindow.document.querySelector('.report-header img');
            if (logoElement && settings.logo_path) {
                logoElement.src = settings.logo_path; 
            }
            printWindow.print();
            printWindow.close(); 
        };
    };

    // -----------------------------------------------------------------------------------
    // 2. Function to print the currently loaded grid (Top Button)
    // -----------------------------------------------------------------------------------
    window.printCurrentGrid = function() {
        if (!CURRENT_GRID_DATA.subjects || CURRENT_GRID_DATA.subjects.length === 0) {
            alert("No planner data generated to print. Please select a class and generate the table first.");
            return;
        }
        
        window.printPlanner(
            CURRENT_GRID_DATA.subjects, 
            CURRENT_GRID_DATA.planner_data, 
            {
                class_name: PLANNER_SETTINGS.class_name,
                section_name: PLANNER_SETTINGS.section_name,
                start_date: PLANNER_SETTINGS.start_date,
                end_date: PLANNER_SETTINGS.end_date
            }
        );
    };


    // -----------------------------------------------------------------------------------
    // 3. Event Listener for Weekly Summary Table Print Button (Bottom Button)
    // -----------------------------------------------------------------------------------
    document.querySelectorAll('.btn-print-weekly').forEach(button => {
        button.addEventListener('click', async function() {
            const classId = this.dataset.classId;
            const sectionId = this.dataset.sectionId;
            const startDate = this.dataset.startDate;
            const endDate = this.dataset.endDate;
            const className = this.dataset.className;
            const sectionName = this.dataset.sectionName;

            const fetchUrl = `class_planner.php?action=fetch_print_data&class_id=${classId}&section_id=${sectionId}&start_date=${startDate}&end_date=${endDate}`;
            
            try {
                // Fetch the detailed data using the AJAX endpoint
                const response = await fetch(fetchUrl);
                const data = await response.json();
                
                if (data.error) {
                    alert('Error fetching data for print preview: ' + data.error);
                    return;
                }
                
                // Call the printPlanner function with the fetched data
                window.printPlanner(data.subjects, data.planner_data, {
                    class_name: className,
                    section_name: sectionName,
                    start_date: startDate, 
                    end_date: endDate 
                });
                
            } catch (error) {
                console.error('Fetch error:', error);
                alert('Could not load weekly data for print preview. Please try the Edit button instead.');
            }
            
        });
    });
    
});
</script>

<?php 
// Connection close karein
include_once('includes/footer.php'); 
?>