<?php
// FILE: admin/class_result_sheet_generate.php - FINAL WORKING CODE (Rank 1 & Tie FIX)

include_once('../config/db.php'); 

if (!function_exists('getDBConnection')) {
    die("Error: getDBConnection() function not found. Please define it in your ../config/db.php file.");
}

$conn = getDBConnection();

$class_id = (int)($_GET['class_id'] ?? 0);
$section_id = (int)($_GET['section_id'] ?? 0);
$exam_id = (int)($_GET['exam_id'] ?? 0); 

if ($class_id === 0 || $section_id === 0 || $exam_id === 0) {
    die("<h1>Error:</h1> Please select Class, Section, and Exam."); 
}

$report_data = [];
$subject_details = []; 
$class_details = [];
$settings = [];

try {
    // --- 3. Fetch School Settings (Same) ---
    $settings_data = [];
    $settings_query = $conn->query("SELECT setting_key, setting_value FROM settings");
    while ($row = $settings_query->fetch_assoc()) {
        $settings_data[trim($row['setting_key'])] = trim($row['setting_value']);
    }
    $settings['institute_name'] = $settings_data['institute.name'] ?? $settings_data['institute name'] ?? $settings_data['institute_name'] ?? "Institute Name Not Set";
    $settings['institute_address'] = $settings_data['institute.address'] ?? $settings_data['institute address'] ?? $settings_data['institute_address'] ?? "Address Not Set";
    $settings['institute_phone'] = $settings_data['institute.phone'] ?? $settings_data['institute phone'] ?? $settings_data['institute_phone'] ?? "Phone Not Set";
    $settings['institute_logo_path'] = $settings_data['institute.logo_path'] ?? $settings_data['institute logo path'] ?? '';
    
    // --- 4. Fetch Class, Section, and Exam Details (Same) ---
    $details_query = $conn->prepare("
        SELECT 
            c.name as class_name, 
            s.name as section_name, 
            e.name as exam_name        
        FROM academic_classes c        
        JOIN academic_sections s ON s.id = ?  
        JOIN exams e ON e.id = ?            
        WHERE c.id = ?
    ");
    $details_query->bind_param("iii", $section_id, $exam_id, $class_id);
    $details_query->execute();
    $class_details = $details_query->get_result()->fetch_assoc();
    $details_query->close();
    $termName = $class_details['exam_name'] ?? 'N/A';

    // --- 5. Fetch Subjects and Exam Structure (Same) ---
    $subjects_query = $conn->prepare("
        SELECT 
            t2.id AS subject_id, 
            t2.name AS subject_name,
            ces.total_marks,
            ces.passing_marks
        FROM course_allocation t1  
        JOIN subjects t2 ON t1.subject_id = t2.id  
        LEFT JOIN class_exam_structure ces 
            ON ces.subject_id = t2.id AND ces.class_id = ? AND ces.exam_id = ?
        WHERE t1.class_id = ?
        ORDER BY t2.id ASC
    ");
    $subjects_query->bind_param("iii", $class_id, $exam_id, $class_id); 
    $subjects_query->execute();
    $subjects_result = $subjects_query->get_result()->fetch_all(MYSQLI_ASSOC);
    $subjects_query->close();
    
    foreach ($subjects_result as $subject) {
        $subject_details[$subject['subject_id']] = [
            'name' => $subject['subject_name'],
            'total_marks' => (int)$subject['total_marks'], 
            'passing_marks' => (int)$subject['passing_marks']
        ];
    }
    
    // --- 6. Fetch All Students (Same) ---
    $students_query = $conn->prepare("
        SELECT 
            t1.id as student_id, 
            t1.first_name as student_name, 
            t1.father_name,
            t2.roll_no  
        FROM students t1                             
        JOIN student_enrollments t2 ON t1.id = t2.student_id        
        WHERE t2.class_id = ? AND t2.section_id = ? AND t1.status = 'active'
        ORDER BY t2.roll_no ASC, t1.first_name ASC
    ");
    $students_query->bind_param("ii", $class_id, $section_id);
    $students_query->execute();
    $students_result = $students_query->get_result()->fetch_all(MYSQLI_ASSOC);
    $students_query->close();
    
    // --- 7. Fetch All Marks (Same) ---
    $marks_lookup = [];
    $marks_query = $conn->prepare("
        SELECT
            student_id,
            subject_id,
            marks_obtained,
            is_absent
        FROM exam_marks 
        WHERE class_id = ? AND exam_id = ?
    ");
    $marks_query->bind_param("ii", $class_id, $exam_id);
    $marks_query->execute();
    $marks_result = $marks_query->get_result();
    
    while($row = $marks_result->fetch_assoc()) {
        $marks_lookup[$row['student_id']][$row['subject_id']] = [
            'marks_obtained' => (int)$row['marks_obtained'],
            'is_absent' => (bool)$row['is_absent']
        ];
    }
    $marks_query->close();
    
    // --- 8. Compile Initial Report Data ---
    foreach ($students_result as $student_row) {
        $student_id = $student_row['student_id'];
        $total_max_marks = 0;
        $total_obtained_marks = 0;
        $is_failed = false;
        $student_marks = [];
        
        foreach ($subject_details as $subject_id => $subject) {
            $total_marks = $subject['total_marks'];
            $passing_marks = $subject['passing_marks'];
            
            $marks_data = $marks_lookup[$student_id][$subject_id] ?? ['marks_obtained' => 0, 'is_absent' => false];
            $marks_obtained = $marks_data['marks_obtained'];
            $is_absent = $marks_data['is_absent'];
            
            if ($total_marks > 0) {
                $total_max_marks += $total_marks;
            }
            
            if (!$is_absent) {
                $total_obtained_marks += $marks_obtained;
                
                if ($total_marks > 0 && $marks_obtained < $passing_marks) {
                    $is_failed = true;
                }
            } else {
                 $is_failed = true; // Absent means fail
            }

            $student_marks[$subject_id] = [
                'obtained' => $is_absent ? 'A' : $marks_obtained,
                'total' => $total_marks,
                'status' => $is_absent ? 'Absent' : ($total_marks > 0 && $marks_obtained < $passing_marks ? 'Fail' : 'Pass')
            ];
        }

        $percentage = ($total_max_marks > 0) ? round(($total_obtained_marks / $total_max_marks) * 100, 2) : 0;
        $final_status = $is_failed ? 'FAIL' : 'PASS';
        if ($total_max_marks == 0) $final_status = 'N/A';
        
        $report_data[] = [
            'student_id' => $student_id,
            'student_name' => $student_row['student_name'],
            'father_name' => $student_row['father_name'],
            'roll_no' => $student_row['roll_no'],
            'subject_marks' => $student_marks,
            'total_max_marks' => $total_max_marks,
            'total_obtained_marks' => $total_obtained_marks,
            'percentage' => $percentage,
            'final_status' => $final_status,
            'position' => 0 // Default position
        ];
    }
    
    // --- 9. Position Calculation Logic (FINAL FIX) ---
    
    // a. Sort ALL data based on obtained marks (descending)
    usort($report_data, function($a, $b) {
        return $b['total_obtained_marks'] <=> $a['total_obtained_marks'];
    });

    // b. Assign ranks (Rank 1 for the highest score)
    $rank = 0;
    $previous_marks = -1;
    $rank_counter = 0; // Number of students processed

    foreach ($report_data as &$student) {
        $rank_counter++; // Har student ko count karein
        $current_marks = $student['total_obtained_marks'];
        
        // Agar marks pichle student se alag hain, to rank ko update karein
        if ($current_marks != $previous_marks) {
            $rank = $rank_counter; // New rank is the current student count
        }
        
        $student['position'] = $rank;
        $previous_marks = $current_marks;
    }
    unset($student); 

    // c. Final sorting by Roll No (Original Sheet Order)
    usort($report_data, function($a, $b) {
        return $a['roll_no'] <=> $b['roll_no'];
    });
    
    // d. Re-check for N/A cases (No Max Marks defined)
    foreach ($report_data as &$student) {
        if ($student['total_max_marks'] === 0) {
            $student['position'] = 'N/A'; // Agar Max Marks hi 0 hain to position N/A
        }
    }
    unset($student);


    // --- 10. Load the Print View ---
    require_once('class_result_sheet_print.php');

} catch (Exception $e) {
    die("<h1>Database Error:</h1> " . $e->getMessage());
} finally {
    if (isset($conn) && is_object($conn) && $conn->ping()) {
        $conn->close();
    }
}
?>