<?php
// FILE: admin/deduction_process.php - FINAL FIXED CODE (CRUD Operations)
include_once('../config/db.php'); 
$conn = getDBConnection();

header('Content-Type: application/json'); // Set response type to JSON

$action = $_REQUEST['action'] ?? '';

// --- HELPER FUNCTION: Recalculate Main Slip Totals ---
function recalculate_slip_totals($conn, $slip_id) {
    // 1. Calculate Total Deduction from the new teacher_deductions table
    $deduction_sum_stmt = $conn->prepare("
        SELECT SUM(amount) AS total_deduction FROM teacher_deductions WHERE slip_id = ?
    ");
    $deduction_sum_stmt->bind_param("i", $slip_id);
    $deduction_sum_stmt->execute();
    $total_deduction = $deduction_sum_stmt->get_result()->fetch_assoc()['total_deduction'] ?? 0;
    $deduction_sum_stmt->close();

    // 2. Fetch Gross Salary
    $gross_stmt = $conn->prepare("
        SELECT gross_salary FROM teacher_salary_slips WHERE id = ?
    ");
    $gross_stmt->bind_param("i", $slip_id);
    $gross_stmt->execute();
    $gross_salary = $gross_stmt->get_result()->fetch_assoc()['gross_salary'] ?? 0;
    $gross_stmt->close();
    
    // 3. Calculate New Net Paid
    $new_net_paid = $gross_salary - $total_deduction;

    // 4. Update the main slip table
    $update_slip_stmt = $conn->prepare("
        UPDATE teacher_salary_slips 
        SET total_deduction = ?, 
            net_paid = ? 
        WHERE id = ?
    ");
    $update_slip_stmt->bind_param("ddi", $total_deduction, $new_net_paid, $slip_id);
    $update_slip_stmt->execute();
    $update_slip_stmt->close();
    
    return ['total_deduction' => $total_deduction, 'net_paid' => $new_net_paid];
}

try {
    if ($action === 'fetch_deductions') {
        $slip_id = (int)($_GET['slip_id'] ?? 0);
        if ($slip_id <= 0) {
            echo json_encode(['success' => false, 'message' => 'Invalid Slip ID.']);
            exit();
        }
        
        $fetch_stmt = $conn->prepare("
            SELECT id, amount, reason, DATE_FORMAT(created_at, '%d-%m-%Y') as date 
            FROM teacher_deductions 
            WHERE slip_id = ?
            ORDER BY created_at ASC
        ");
        $fetch_stmt->bind_param("i", $slip_id);
        $fetch_stmt->execute();
        $result = $fetch_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        $fetch_stmt->close();
        
        echo json_encode(['success' => true, 'deductions' => $result]);
        
    } elseif ($action === 'add_deduction' || $action === 'update_deduction') {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
             echo json_encode(['success' => false, 'message' => 'Invalid request method.']);
             exit();
        }
        
        $slip_id = (int)($_POST['slip_id'] ?? 0);
        $amount = (float)($_POST['amount'] ?? 0);
        $reason = trim($_POST['reason'] ?? '');
        $deduction_id = (int)($_POST['deduction_id'] ?? 0); // Used for update
        
        if ($slip_id <= 0 || $amount < 0 || empty($reason)) {
            echo json_encode(['success' => false, 'message' => 'Slip ID, Reason are required and Amount must be valid.']);
            exit();
        }
        
        $conn->begin_transaction();
        
        if ($action === 'add_deduction') {
            // 1. Insert new deduction
            $insert_deduction_stmt = $conn->prepare("
                INSERT INTO teacher_deductions (slip_id, amount, reason) VALUES (?, ?, ?)
            ");
            $insert_deduction_stmt->bind_param("ids", $slip_id, $amount, $reason);
            $insert_deduction_stmt->execute();
            $insert_deduction_stmt->close();
            $message = "Deduction successfully added.";
            
        } elseif ($action === 'update_deduction') {
            if ($deduction_id <= 0) {
                throw new Exception("Invalid Deduction ID for update.");
            }
            // 1. Update existing deduction
            $update_deduction_stmt = $conn->prepare("
                UPDATE teacher_deductions SET amount = ?, reason = ? WHERE id = ? AND slip_id = ?
            ");
            $update_deduction_stmt->bind_param("dsii", $amount, $reason, $deduction_id, $slip_id);
            $update_deduction_stmt->execute();
            $update_deduction_stmt->close();
            $message = "Deduction successfully updated.";
        }
        
        // 2. Re-insert/Update item in teacher_salary_items to reflect the change for print details.
        // NOTE: For simplicity, we delete all items and re-insert the current list + base salary.
        
        // First, delete old deduction items for this slip
        $delete_items_stmt = $conn->prepare("DELETE FROM teacher_salary_items WHERE slip_id = ? AND item_type = 'deduction'");
        $delete_items_stmt->bind_param("i", $slip_id);
        $delete_items_stmt->execute();
        $delete_items_stmt->close();

        // Re-fetch all deductions (including the newly added/updated one)
        $fetch_all_deductions_stmt = $conn->prepare("SELECT amount, reason FROM teacher_deductions WHERE slip_id = ?");
        $fetch_all_deductions_stmt->bind_param("i", $slip_id);
        $fetch_all_deductions_stmt->execute();
        $all_deductions = $fetch_all_deductions_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        $fetch_all_deductions_stmt->close();
        
        // Re-insert all current deductions into teacher_salary_items
        $insert_item_stmt = $conn->prepare("
            INSERT INTO teacher_salary_items (slip_id, item_type, item_description, amount) VALUES (?, ?, ?, ?)
        ");
        $item_type = 'deduction';
        foreach ($all_deductions as $deduction) {
            $insert_item_stmt->bind_param("issd", $slip_id, $item_type, $deduction['reason'], $deduction['amount']); 
            $insert_item_stmt->execute();
        }
        $insert_item_stmt->close();
        
        // 3. Recalculate main slip totals and update
        $new_totals = recalculate_slip_totals($conn, $slip_id);

        $conn->commit();
        echo json_encode(['success' => true, 'message' => $message, 'totals' => $new_totals]);
        
    } elseif ($action === 'delete_deduction') {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
             echo json_encode(['success' => false, 'message' => 'Invalid request method.']);
             exit();
        }
        $deduction_id = (int)($_POST['deduction_id'] ?? 0);
        $slip_id = (int)($_POST['slip_id'] ?? 0);
        
        if ($deduction_id <= 0 || $slip_id <= 0) {
            echo json_encode(['success' => false, 'message' => 'Invalid Deduction ID or Slip ID.']);
            exit();
        }
        
        $conn->begin_transaction();

        // 1. Delete deduction from teacher_deductions
        $delete_stmt = $conn->prepare("DELETE FROM teacher_deductions WHERE id = ? AND slip_id = ?");
        $delete_stmt->bind_param("ii", $deduction_id, $slip_id);
        $delete_stmt->execute();
        $delete_stmt->close();
        
        // 2. Recalculate main slip totals (which also updates teacher_salary_items)
        $new_totals = recalculate_slip_totals($conn, $slip_id);
        
        $conn->commit();
        echo json_encode(['success' => true, 'message' => 'Deduction successfully removed.', 'totals' => $new_totals]);
        
    } else {
        echo json_encode(['success' => false, 'message' => 'Unknown action.']);
    }
} catch (Exception $e) {
    if ($conn->in_transaction) {
        $conn->rollback();
    }
    echo json_encode(['success' => false, 'message' => "Error: " . $e->getMessage()]);
} finally {
    if ($conn && $conn->ping()) {
        $conn->close();
    }
}
?>