<?php
// admin/exams_process.php - FINAL CODE (Including Add, Update, Delete, and Result Logic)

session_start();
include_once('../config/db.php');

// ====================================================================
// HELPER FUNCTIONS (Zaroori functions for flow control and grading)
// ====================================================================

/**
 * Redirects the user to a specified page with a message.
 * @param string $page The target file name (e.g., 'manage_exams').
 * @param string $msg The success or error message.
 * @param string $type 'success' or 'err'.
 * @param array $extra_params Additional URL parameters (e.g., ['id' => 1]).
 */
function redirect($page, $msg = '', $type = 'success', $extra_params = []) {
    // Ensure .php extension is present
    if (!str_ends_with($page, '.php')) {
        $page .= '.php';
    }
    
    $param_name = ($type === 'success') ? 'msg' : 'err';
    $url = $page . "?$param_name=" . urlencode($msg);
    foreach ($extra_params as $key => $value) {
        $url .= "&$key=" . urlencode($value);
    }
    header("Location: $url");
    exit;
}

// 1. Minimum Passing Marks
function getPassingPercentage() {
    return 40; // Example: 40%
}

// 2. Grade Calculation
function calculateGrade($percentage) {
    if ($percentage >= 90) return 'A+';
    if ($percentage >= 80) return 'A';
    if ($percentage >= 70) return 'B';
    if ($percentage >= 60) return 'C';
    if ($percentage >= 40) return 'D';
    return 'F'; // Fail
}

// ====================================================================
// MAIN INITIALIZATION
// ====================================================================

$conn = getDBConnection();
// Action ko POST ya GET se lein
$action = $_POST['action'] ?? $_GET['action'] ?? '';

// You should add authentication check here
// if (!isset($_SESSION['user_id'])) { redirect('login'); } 

// ====================================================================
// MAIN PROCESSING LOGIC
// ====================================================================
try {
    // Transaction shuru karein taaki database operations safe rahein
    $conn->begin_transaction(); 

    switch ($action) {
        
        // ====================================================================
        // NEW ACTION: UPDATE EXAM (Used by edit_exam.php)
        // ====================================================================
        case 'update_exam':
            $exam_id = (int)$_POST['exam_id'];
            $name = trim($_POST['exam_name'] ?? ''); 
            $session_id = (int)$_POST['session_id'];
            $start_date = $_POST['start_date'] ?? NULL;
            $end_date = $_POST['end_date'] ?? NULL;
            $status = $_POST['status'] ?? 'active';

            if ($exam_id <= 0 || empty($name) || $session_id <= 0 || empty($start_date)) {
                redirect('edit_exam', 'Invalid or incomplete exam details provided.', 'err', ['id' => $exam_id]);
            }
            
            $stmt = $conn->prepare("UPDATE exams SET name=?, session_id=?, start_date=?, end_date=?, status=? WHERE id=?");
            $stmt->bind_param("sisssi", $name, $session_id, $start_date, $end_date, $status, $exam_id);
            
            if (!$stmt->execute()) {
                throw new Exception("Database update failed: " . $stmt->error);
            }
            $stmt->close();

            $conn->commit();
            redirect('manage_exams', 'Exam "' . htmlspecialchars($name) . '" successfully updated.', 'success'); 
            break;


        // ====================================================================
        // NEW ACTION: DELETE EXAM (Used by manage_exams.php Delete button)
        // ====================================================================
        case 'delete_exam':
            $exam_id = (int)($_GET['id'] ?? $_POST['exam_id'] ?? 0); // ID ko GET ya POST se lein
            
            if ($exam_id <= 0) {
                redirect('manage_exams', 'Invalid Exam ID for deletion.', 'err');
            }
            
            // 1. Delete related records from exam_marks (Marks entered for this exam)
            $stmt_marks = $conn->prepare("DELETE FROM exam_marks WHERE exam_id = ?");
            $stmt_marks->bind_param("i", $exam_id);
            $stmt_marks->execute();
            $stmt_marks->close();
            
            // 2. Delete related records from student_results (Calculated results)
            $stmt_results = $conn->prepare("DELETE FROM student_results WHERE exam_id = ?");
            $stmt_results->bind_param("i", $exam_id);
            $stmt_results->execute();
            $stmt_results->close();

            // 3. Delete related records from class_exam_structure (Subject structure/Max Marks)
            $stmt_structure = $conn->prepare("DELETE FROM class_exam_structure WHERE exam_id = ?");
            $stmt_structure->bind_param("i", $exam_id);
            $stmt_structure->execute();
            $stmt_structure->close();

            // 4. Delete the exam itself
            $stmt_exam = $conn->prepare("DELETE FROM exams WHERE id = ?");
            $stmt_exam->bind_param("i", $exam_id);
            if (!$stmt_exam->execute()) {
                throw new Exception("Exam record could not be deleted: " . $stmt_exam->error);
            }
            $stmt_exam->close();
            
            $conn->commit();
            redirect('manage_exams', 'Exam and all related data successfully deleted.', 'success');
            break;


        // ====================================================================
        // EXISTING ACTION: CALCULATE RESULTS (Aapka original logic)
        // ====================================================================
        case 'calculate_results':
            $exam_id = (int)$_POST['exam_id'];
            $session_id = (int)$_POST['session_id'];
            $passing_percentage = getPassingPercentage();

            if ($exam_id <= 0) { throw new Exception("Invalid Exam ID."); }
            
            // A) Fetch Exam-wise Subjects and Max Marks
            $subjects_query = $conn->prepare("
                SELECT ac.id AS class_id, ac.name AS class_name, ces.subject_id, sub.name AS subject_name, ces.max_marks
                FROM class_exam_structure ces
                JOIN academic_classes ac ON ces.class_id = ac.id
                JOIN subjects sub ON ces.subject_id = sub.id
                WHERE ces.exam_id = ?
            ");
            $subjects_query->bind_param("i", $exam_id);
            $subjects_query->execute();
            $exam_subjects = $subjects_query->get_result()->fetch_all(MYSQLI_ASSOC);
            $subjects_query->close();
            
            if (empty($exam_subjects)) {  
                throw new Exception("No subjects or max marks defined for this exam. Check class_exam_structure.");  
            }
            
            // Reorganize structure
            $exam_structure = [];
            $total_max_marks_by_class = [];
            foreach ($exam_subjects as $sub) {
                $exam_structure[$sub['class_id']][$sub['subject_id']] = (int)$sub['max_marks'];
                $total_max_marks_by_class[$sub['class_id']] = ($total_max_marks_by_class[$sub['class_id']] ?? 0) + (int)$sub['max_marks'];
            }

            // B) Fetch All Student Marks for this Exam
            $marks_query = $conn->prepare("
                SELECT em.student_id, em.marks_obtained, em.is_absent, em.class_id, em.subject_id
                FROM exam_marks em
                WHERE em.exam_id = ? 
            ");
            $marks_query->bind_param("i", $exam_id);
            $marks_query->execute();
            $marks_result = $marks_query->get_result();
            $student_marks = [];
            while ($row = $marks_result->fetch_assoc()) {
                $student_marks[$row['student_id']][$row['subject_id']] = [
                    'marks' => (int)$row['marks_obtained'], 
                    'absent' => (bool)$row['is_absent'],
                    'class_id' => (int)$row['class_id']
                ];
            }
            $marks_query->close();

            // C) Fetch Enrolled Students
            $students_query = $conn->prepare("
                SELECT s.id AS student_id, en.class_id
                FROM students s
                JOIN student_enrollments en ON s.id = en.student_id
                WHERE en.session_id = ? AND s.status = 'active'
            ");
            $students_query->bind_param("i", $session_id);
            $students_query->execute();
            $students_list = $students_query->get_result()->fetch_all(MYSQLI_ASSOC);
            $students_query->close();

            $processed_count = 0;
            
            // D) Calculate Result for each student
            foreach ($students_list as $student) {
                $student_id = $student['student_id'];
                $class_id = $student['class_id'];
                
                if (!isset($exam_structure[$class_id])) { continue; } 
                
                $total_obtained = 0;
                $total_max = $total_max_marks_by_class[$class_id];
                $is_passed = true;
                $subjects_failed = 0;
                $is_result_valid = true; 
                
                // i) Per-Subject Calculation
                foreach ($exam_structure[$class_id] as $subject_id => $max_marks) {
                    $marks_data = $student_marks[$student_id][$subject_id] ?? null;
                    
                    if ($max_marks == 0) { continue; } 

                    if ($marks_data === null) {
                        $is_result_valid = false; 
                        break; 
                    }
                    
                    if ($marks_data['absent']) {
                        $is_passed = false;
                        $subjects_failed++;
                        continue; 
                    }
                    
                    $obtained = $marks_data['marks'];
                    $total_obtained += $obtained;
                    
                    $subject_percentage = ($obtained / $max_marks) * 100;
                    if ($subject_percentage < $passing_percentage) {
                        $is_passed = false;
                        $subjects_failed++;
                    }
                }
                
                // ii) Finalize Results
                if (!$is_result_valid) {
                    $result_status = 'Incomplete';
                    $final_percentage = 0;
                    $final_grade = 'N/A';
                    $subjects_failed = count($exam_structure[$class_id]); 
                    $total_obtained = 0;
                } else {
                    $result_status = $is_passed ? 'Pass' : 'Fail';
                    $final_percentage = $total_max > 0 ? ($total_obtained / $total_max) * 100 : 0; 
                    $final_grade = calculateGrade($final_percentage);
                }
                
                // iii) Insert/Update into student_results
                $existing_res = $conn->query("SELECT id FROM student_results WHERE exam_id = {$exam_id} AND student_id = {$student_id}");
                
                if ($existing_res->num_rows > 0) {
                    // Update existing result
                    $update_stmt = $conn->prepare("UPDATE student_results SET total_max_marks=?, total_obtained_marks=?, final_percentage=?, final_grade=?, result_status=?, subjects_failed=? WHERE exam_id=? AND student_id=?");
                    $update_stmt->bind_param("iidsiisi", $total_max, $total_obtained, $final_percentage, $final_grade, $result_status, $subjects_failed, $exam_id, $student_id);
                    $update_stmt->execute();
                    $update_stmt->close();
                } else {
                    // Insert new result
                    $insert_stmt = $conn->prepare("
                        INSERT INTO student_results (exam_id, student_id, class_id, total_max_marks, total_obtained_marks, final_percentage, final_grade, result_status, subjects_failed)
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
                    ");
                    $insert_stmt->bind_param("iiiiidssi", 
                        $exam_id, $student_id, $class_id, $total_max, $total_obtained, $final_percentage, $final_grade, $result_status, $subjects_failed
                    );
                    $insert_stmt->execute();
                    $insert_stmt->close();
                }
                $processed_count++;
            }
            
            // E) Update Exam Status to 'completed'
            $update_exam_stmt = $conn->prepare("UPDATE exams SET status = 'completed', result_generated_at = NOW() WHERE id = ?");
            $update_exam_stmt->bind_param("i", $exam_id);
            $update_exam_stmt->execute();
            $update_exam_stmt->close();

            $conn->commit();
            redirect('reports_results', "Result successfully calculated and saved for **{$processed_count}** students. Exam finalized!", 'success');
            break;
            
        // ... (Other existing cases like save_marks, manage_exam will be here) ...

        default:
             // Agar koi action specify nahi hua ya action invalid hai
            if (empty($action)) {
                 throw new Exception("No action requested. Please check the form action.");
            }
            throw new Exception("Invalid action requested: " . htmlspecialchars($action));
    }
} catch (Exception $e) {
    // Agar koi galti ho to rollback karein
    if ($conn && $conn->in_transaction) {
        $conn->rollback();
    }
    // Galti hone par wapas pichle page par bhej dein
    $referer_page = basename($_SERVER['HTTP_REFERER'], '.php');
    if (empty($referer_page) || $referer_page === 'exams_process') {
         $referer_page = 'manage_exams'; 
    }
    redirect($referer_page, 'Error processing action: ' . $e->getMessage(), 'err');

} finally {
    if ($conn) {
        $conn->close();
    }
}
?>