<?php
// admin/fees_config.php - FINAL CODE with all functionalities: Save Structure, Add Head, Delete Head, EDIT/UPDATE Head, and Layout Fix

session_start();
// Include the database connection
include_once('../config/db.php'); 

$conn = getDBConnection();

// Fetch settings array
$settings_result = $conn->query("SELECT setting_key, setting_value FROM settings");
$settings = [];
if ($settings_result) {
    while ($row = $settings_result->fetch_assoc()) {
        $settings[$row['setting_key']] = $row['setting_value'];
    }
}

$message = $_GET['msg'] ?? ''; 
$error = $_GET['err'] ?? ''; 
$current_tab = $_GET['tab'] ?? 'heads'; 
$current_session_id = $settings['academic.active_session_id'] ?? 1; 
$head_to_edit = null; 

// ===================================================================
//  1. HANDLE FEE STRUCTURE FORM SUBMISSION (Save/Update Class Fees)
// ===================================================================

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'save_structure') {
    
    if (!isset($_POST['amount']) || !is_array($_POST['amount'])) {
        $error = "No fee amounts submitted. Please enter at least one amount.";
    } else {
        $amounts = $_POST['amount'];
        $conn->begin_transaction();

        try {
            // Purani structure delete karo
            $delete_stmt = $conn->prepare("DELETE FROM fee_structures WHERE session_id = ?");
            $delete_stmt->bind_param("i", $current_session_id);
            $delete_stmt->execute();
            $delete_stmt->close();
            
            $insert_count = 0;
            $insert_stmt = $conn->prepare("INSERT INTO fee_structures (session_id, class_id, head_id, amount) VALUES (?, ?, ?, ?)");

            // Naya data insert karo.
            foreach ($amounts as $class_id => $head_data) {
                $class_id = (int)$class_id;
                foreach ($head_data as $head_id => $amount) {
                    $head_id = (int)$head_id;
                    $amount = (float)$amount; 
                    if ($amount > 0) {
                        $insert_stmt->bind_param("iiid", $current_session_id, $class_id, $head_id, $amount);
                        $insert_stmt->execute();
                        $insert_count++;
                    }
                }
            }

            $conn->commit();
            $msg = "Fee structure successfully saved for {$insert_count} entries.";
            header("Location: fees_config.php?tab=structure&msg=" . urlencode($msg));
            exit;

        } catch (Exception $e) {
            $conn->rollback();
            $error = "Error saving fee structure: " . $e->getMessage();
        } finally {
            if (isset($insert_stmt)) {
                $insert_stmt->close();
            }
        }
    }
}


// ===================================================================
//  2. HANDLE FEE HEADS MANAGEMENT ACTIONS (Add/Update/Delete Head)
// ===================================================================

// A. ADD HEAD (POST)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'add_head') {
    $head_name = trim($_POST['head_name']);
    $head_type = $_POST['head_type'] ?? 'monthly'; 
    $type_map = ['monthly' => 'Recurring', 'one_time' => 'One Time'];
    $is_recurring = ($head_type === 'monthly') ? 1 : 0;
    
    if (empty($head_name)) {
        $err = "Fee Head Name is required.";
    } else {
        $check_stmt = $conn->prepare("SELECT id FROM fee_heads WHERE name = ?");
        $check_stmt->bind_param("s", $head_name);
        $check_stmt->execute();
        $check_stmt->store_result();
        
        if ($check_stmt->num_rows > 0) {
            $err = "Fee Head '{$head_name}' already exists.";
        } else {
            $insert_head_stmt = $conn->prepare("INSERT INTO fee_heads (name, type, is_recurring) VALUES (?, ?, ?)");
            $insert_head_stmt->bind_param("ssi", $head_name, $type_map[$head_type], $is_recurring);
            
            if ($insert_head_stmt->execute()) {
                $msg = "Fee Head '{$head_name}' successfully added.";
            } else {
                $err = "Database error: Could not add Fee Head.";
            }
            $insert_head_stmt->close();
        }
        $check_stmt->close();
    }
    header("Location: fees_config.php?tab=heads&msg=" . urlencode($msg ?? '') . "&err=" . urlencode($err ?? ''));
    exit;
}

// B. UPDATE HEAD (POST)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'update_head') {
    $head_id = (int)$_POST['head_id'];
    $head_name = trim($_POST['head_name']);
    $head_type = $_POST['head_type'] ?? 'monthly'; 
    $type_map = ['monthly' => 'Recurring', 'one_time' => 'One Time'];
    $is_recurring = ($head_type === 'monthly') ? 1 : 0;

    if ($head_id <= 0 || empty($head_name)) {
        $err = "Invalid data for update.";
    } else {
        $update_stmt = $conn->prepare("UPDATE fee_heads SET name = ?, type = ?, is_recurring = ? WHERE id = ?");
        $update_stmt->bind_param("ssii", $head_name, $type_map[$head_type], $is_recurring, $head_id);
        
        if ($update_stmt->execute()) {
            $msg = "Fee Head (ID: {$head_id}) successfully updated.";
        } else {
            $err = "Database error: Could not update Fee Head.";
        }
        $update_stmt->close();
    }
    header("Location: fees_config.php?tab=heads&msg=" . urlencode($msg ?? '') . "&err=" . urlencode($err ?? ''));
    exit;
}

// C. DELETE HEAD (GET)
if ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['action']) && $_GET['action'] === 'delete_head' && isset($_GET['id'])) {
    $head_id = (int)$_GET['id'];
    
    if ($head_id > 0) {
        $conn->begin_transaction();
        try {
            // Check if this head is used in any fee structure
            $check_usage_stmt = $conn->prepare("SELECT COUNT(*) FROM fee_structures WHERE head_id = ?");
            $check_usage_stmt->bind_param("i", $head_id);
            $check_usage_stmt->execute();
            $check_usage_stmt->bind_result($count);
            $check_usage_stmt->fetch();
            $check_usage_stmt->close();

            if ($count > 0) {
                 throw new Exception("Cannot delete. This Fee Head is currently used in {$count} fee structure(s).");
            }

            // Delete the head
            $delete_stmt = $conn->prepare("DELETE FROM fee_heads WHERE id = ?");
            $delete_stmt->bind_param("i", $head_id);
            
            if ($delete_stmt->execute()) {
                $msg = "Fee Head (ID: {$head_id}) successfully deleted.";
                $conn->commit();
            } else {
                throw new Exception("Error deleting Fee Head: " . $delete_stmt->error);
            }
            $delete_stmt->close();

        } catch (Exception $e) {
            $conn->rollback();
            $err = "Deletion failed: " . $e->getMessage();
        }
    } else {
        $err = "Invalid Fee Head ID.";
    }
    header("Location: fees_config.php?tab=heads&msg=" . urlencode($msg ?? '') . "&err=" . urlencode($err ?? ''));
    exit;
}


// D. EDIT HEAD (GET) - Fetch data and store in $head_to_edit
if ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['action']) && $_GET['action'] === 'edit_head' && isset($_GET['id'])) {
    $head_id = (int)$_GET['id'];
    $edit_stmt = $conn->prepare("SELECT id, name, type FROM fee_heads WHERE id = ?");
    $edit_stmt->bind_param("i", $head_id);
    $edit_stmt->execute();
    $result = $edit_stmt->get_result();
    
    if ($result->num_rows === 1) {
        $head_to_edit = $result->fetch_assoc();
        $current_tab = 'heads'; 
    }
    $edit_stmt->close();
}


// ===================================================================
//  3. DATA FETCHING (for displaying tables)
// ===================================================================

// 1. Fetch Fee Heads
$heads_result = $conn->query("SELECT id, name, type, is_recurring FROM fee_heads ORDER BY is_recurring DESC, name ASC"); 
$fee_heads = $heads_result->fetch_all(MYSQLI_ASSOC);

// 2. Fetch Classes
$classes_result = $conn->query("SELECT id, name FROM academic_classes ORDER BY order_no ASC");
$classes = $classes_result->fetch_all(MYSQLI_ASSOC);

// 3. Fetch Existing Structures for the current session
$existing_structure = [];
if (!empty($classes) && !empty($fee_heads)) {
    $structure_query = "SELECT head_id, class_id, amount FROM fee_structures WHERE session_id = {$current_session_id}";
    $structure_result = $conn->query($structure_query);
    if ($structure_result) {
        while ($row = $structure_result->fetch_assoc()) {
            $existing_structure[$row['class_id']][$row['head_id']] = $row['amount'];
        }
    }
}

// --- HTML HEADER (Layout intact) ---
include_once('includes/header.php'); 
?>

<div class="container-fluid">
    <div class="row">
        <main role="main" class="col-md-9 col-lg-10 px-4"> 
            
            <h1 class="h2 mb-4">Fee Configuration & Setup</h1>

            <?php // Success and Error Messages Display
            if (!empty($message)): ?>
                <div class="alert alert-success"><?php echo htmlspecialchars($message); ?></div>
            <?php endif; ?>
            <?php if (!empty($error)): ?>
                <div class="alert alert-danger"><?php echo htmlspecialchars($error); ?></div>
            <?php endif; ?>


            <ul class="nav nav-tabs" id="feeConfigTab" role="tablist">
                <li class="nav-item">
                    <a class="nav-link <?php echo ($current_tab == 'heads') ? 'active' : ''; ?>" id="heads-tab" data-toggle="tab" href="#heads" role="tab" aria-controls="heads" aria-selected="true">Fee Heads Management</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link <?php echo ($current_tab == 'structure') ? 'active' : ''; ?>" id="structure-tab" data-toggle="tab" href="#structure" role="tab" aria-controls="structure" aria-selected="false">Class Fee Structure</a>
                </li>
            </ul>

            <div class="tab-content" id="feeConfigTabContent">
                
                <div class="tab-pane fade <?php echo ($current_tab == 'heads') ? 'show active' : ''; ?> p-3 border border-top-0" id="heads" role="tabpanel" aria-labelledby="heads-tab">
                    
                    <h5 class="mb-3">
                        <?php echo $head_to_edit ? 'Update Fee Head (ID: ' . htmlspecialchars($head_to_edit['id']) . ')' : 'Add New Fee Head'; ?>
                    </h5>
                    
                    <form method="POST" action="fees_config.php?tab=heads" class="form-inline mb-4">
                        <input type="hidden" name="action" value="<?php echo $head_to_edit ? 'update_head' : 'add_head'; ?>">
                        
                        <?php if ($head_to_edit): ?>
                            <input type="hidden" name="head_id" value="<?php echo htmlspecialchars($head_to_edit['id']); ?>">
                        <?php endif; ?>

                        <div class="form-group mr-3">
                            <label for="head_name" class="mr-2">Head Name:</label>
                            <input type="text" name="head_name" id="head_name" class="form-control" 
                                   placeholder="e.g., Annual Fund" 
                                   value="<?php echo $head_to_edit ? htmlspecialchars($head_to_edit['name']) : ''; ?>"
                                   required>
                        </div>
                        
                        <div class="form-group mr-3">
                            <label for="head_type" class="mr-2">Type:</label>
                            <select name="head_type" id="head_type" class="form-control">
                                <?php 
                                    $current_type = $head_to_edit['type'] ?? 'Recurring';
                                    $monthly_selected = ($current_type === 'Recurring') ? 'selected' : '';
                                    $onetime_selected = ($current_type === 'One Time') ? 'selected' : '';
                                ?>
                                <option value="monthly" <?php echo $monthly_selected; ?>>Monthly/Recurring</option>
                                <option value="one_time" <?php echo $onetime_selected; ?>>One Time/Admission</option>
                            </select>
                        </div>
                        
                        <button type="submit" class="btn btn-<?php echo $head_to_edit ? 'warning' : 'primary'; ?>">
                            <i class="fas fa-<?php echo $head_to_edit ? 'sync-alt' : 'plus'; ?>"></i> 
                            <?php echo $head_to_edit ? 'Update Head' : 'Add Head'; ?>
                        </button>

                        <?php if ($head_to_edit): ?>
                            <a href="fees_config.php?tab=heads" class="btn btn-secondary ml-2"><i class="fas fa-times"></i> Cancel Edit</a>
                        <?php endif; ?>
                    </form>

                    <h5 class="mt-4">Existing Fee Heads</h5>
                    <div class="table-responsive">
                        <table class="table table-bordered table-sm">
                            <thead class="thead-light">
                                <tr>
                                    <th>ID</th>
                                    <th>Head Name</th>
                                    <th>Type</th>
                                    <th>Action</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($fee_heads)): ?>
                                    <tr><td colspan="4" class="text-center">No fee heads defined yet.</td></tr>
                                <?php else: ?>
                                    <?php foreach ($fee_heads as $head): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($head['id']); ?></td>
                                        <td><?php echo htmlspecialchars($head['name']); ?></td>
                                        <td><?php echo htmlspecialchars($head['type']); ?></td>
                                        <td>
                                            <a href="fees_config.php?tab=heads&action=edit_head&id=<?php echo $head['id']; ?>" 
                                               class="btn btn-sm btn-info">
                                                <i class="fas fa-edit"></i> Edit
                                            </a>
                                            
                                            <a href="fees_config.php?tab=heads&action=delete_head&id=<?php echo $head['id']; ?>" 
                                               class="btn btn-sm btn-danger" 
                                               onclick="return confirm('Kya aap Fee Head: <?php echo htmlspecialchars($head['name']); ?> ko yakeeni taur par delete karna chahte hain? Isse Fee Structure se iski entries bhi hat jayengi.');">
                                                <i class="fas fa-trash"></i> Delete
                                            </a>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>

                <div class="tab-pane fade <?php echo ($current_tab == 'structure') ? 'show active' : ''; ?> p-3 border border-top-0" id="structure" role="tabpanel" aria-labelledby="structure-tab">
                    
                    <h5 class="mb-3">Define Fee Amounts by Class and Head (Session: <?php echo htmlspecialchars($current_session_id); ?>)</h5>
                    <p class="text-info">Only fill amounts for recurring fees like **Monthly Tuition Fee**. Zero or empty fields will be ignored on saving.</p>
                    
                    <form method="POST" action="fees_config.php?tab=structure">
                        <input type="hidden" name="action" value="save_structure">
                        
                        <div class="table-responsive">
                        <table class="table table-bordered table-sm">
                            <thead class="thead-dark">
                                <tr>
                                    <th>Class Name</th>
                                    <?php foreach ($fee_heads as $head): ?>
                                        <th class="text-center"><?php echo htmlspecialchars($head['name']); ?></th>
                                    <?php endforeach; ?>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($classes as $class): 
                                    $class_id = $class['id'];
                                ?>
                                    <tr>
                                        <td class="font-weight-bold"><?php echo htmlspecialchars($class['name']); ?></td>
                                        
                                        <?php foreach ($fee_heads as $head): 
                                            $head_id = $head['id'];
                                            $amount_val = $existing_structure[$class_id][$head_id] ?? ''; 
                                        ?>
                                            <td style="width: 120px;">
                                            <input type="number" step="1" name="amount[<?php echo $class_id; ?>][<?php echo $head_id; ?>]" 
                                                   class="form-control form-control-sm text-center" 
                                                   placeholder="Amount" value="<?php echo htmlspecialchars($amount_val); ?>"> 
                                            </td>
                                        <?php endforeach; ?>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                        </div>
                        <button type="submit" class="btn btn-success mt-3"><i class="fas fa-save"></i> Save/Update Fee Structure</button>
                    </form>
                </div>

            </div>
            
        </main>
    </div>
</div>

<script>
// Tab functionality (Bootstrap required)
$(document).ready(function() {
    // URL mein tab parameter set karna taake page refresh hone par tab wohi rahe
    $('a[data-toggle="tab"]').on('shown.bs.tab', function (e) {
        var newTab = $(e.target).attr('href').substring(1);
        window.history.pushState(null, null, '?tab=' + newTab);
    });
    // Page load par correct tab activate karna
    var tabParam = new URLSearchParams(window.location.search).get('tab');
    if (tabParam) {
        $('a[href="#' + tabParam + '"]').tab('show');
    }
});
</script>

<?php
// Footer (Layout intact)
include_once('includes/footer.php'); 
//$conn->close(); 
?>