<?php
// admin/fetch_students_by_class.php - Handles AJAX request for filtering students (NO USERS TABLE FILTER)

// Database Connection
include_once('../config/db.php'); 
$conn = getDBConnection(); 

// Input Validation
$class_id = (int)($_GET['class_id'] ?? 0);
$section_id = (int)($_GET['section_id'] ?? 0);

if ($class_id > 0 && $section_id > 0) {
    
    // Query joins students (s) with student_enrollments (se)
    // s.id is returned as user_id because this is the ID used for linking (parent_student_link.student_id)
    $query_students = "
        SELECT 
            s.id AS user_id,                                 
            CONCAT(s.first_name, ' ', s.last_name) AS full_name,  
            se.roll_no                                 
        FROM students s
        
        /* Inner JOIN: Only get students who have an enrollment record */
        JOIN student_enrollments se ON s.id = se.student_id   
        
        WHERE s.status = 'active'
          AND se.class_id = ?
          AND se.section_id = ?
        ORDER BY se.roll_no"; 
        
    $stmt_students = $conn->prepare($query_students);
    
    // Bind parameters: class_id (i), section_id (i)
    $stmt_students->bind_param("ii", $class_id, $section_id);
    
    if ($stmt_students->execute()) {
        $result_students = $stmt_students->get_result();
    
        $all_students = [];
        if ($result_students) {
            while ($row = $result_students->fetch_assoc()) {
                $all_students[] = $row;
            }
        }
        $stmt_students->close();
    
        // Generate the HTML output for the multi-select box
        if (empty($all_students)) {
            echo '<select multiple class="form-control" name="student_id[]" style="min-height: 200px;" disabled>';
            echo '<option value="" disabled selected>-- No active students found in this Class/Section. --</option>';
            echo '</select>';
            echo '<small class="form-text text-muted">No students found matching the selected criteria. Please verify your data entry in the students and student_enrollments tables.</small>';
        } else {
            echo '<select multiple class="form-control" id="student_id_multiselect" name="student_id[]" style="min-height: 200px;" required>';
            foreach ($all_students as $student) {
                $full_name = htmlspecialchars($student['full_name']);
                $roll_no = htmlspecialchars($student['roll_no'] ?? 'N/A');
    
                echo '<option value="' . $student['user_id'] . '">';
                echo "Roll No. {$roll_no} - {$full_name}";
                echo '</option>';
            }
            echo '</select>';
            echo '<small class="form-text text-muted">Hold down Ctrl (Windows) or Command (Mac) to select multiple students.</small>';
        }

    } else {
         // Database execution error
         echo '<div class="alert alert-danger">SQL Query Error: ' . htmlspecialchars($conn->error) . '</div>';
         // Useful for debugging: log the error
         error_log("Student Fetch Error: " . $conn->error); 
    }
    
} else {
    // Input validation failed (Class/Section IDs missing)
    echo '<div class="alert alert-info">Please select both Class and Section.</div>';
}

$conn->close();
?>