<?php
// FILE: admin/generate_slc.php - School Leaving Certificate Management

session_start();
ob_start();
include_once('../config/db.php');
include_once('includes/header.php'); // Aapke system ka header
$conn = getDBConnection();

// --- START: Settings & Session ID ---
$settings_data = [];
$settings_query = $conn->query("SELECT setting_key, setting_value FROM settings");
if ($settings_query) {
    while ($row = $settings_query->fetch_assoc()) {
        $settings_data[trim($row['setting_key'])] = trim($row['setting_value']);
    }
}
$current_session_id = $settings_data['academic.active_session_id'] ?? 1;
// --- END: Settings & Session ID ---

$message = '';
$error = '';

// --- Handle Generation (POST) ---
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['generate_slc'])) {
    $student_id = (int) $_POST['student_id'];
    $leaving_class_id = (int) $_POST['leaving_class_id'];
    $issue_date = date('Y-m-d'); // Current date for issuance

    if ($student_id > 0 && $leaving_class_id > 0) {
        // Check if SLC already exists for this student/leaving class
        $check_stmt = $conn->prepare("SELECT id FROM school_leaving_certificates WHERE student_id = ? AND leaving_class_id = ?");
        $check_stmt->bind_param("ii", $student_id, $leaving_class_id);
        $check_stmt->execute();
        if ($check_stmt->get_result()->num_rows > 0) {
            $error = "Error: School Leaving Certificate for this student in the selected class already exists.";
        } else {
            // Insert new SLC record
            $insert_stmt = $conn->prepare("INSERT INTO school_leaving_certificates (student_id, leaving_class_id, issue_date) VALUES (?, ?, ?)");
            $insert_stmt->bind_param("iis", $student_id, $leaving_class_id, $issue_date);
            if ($insert_stmt->execute()) {
                $message = "Success: School Leaving Certificate generated successfully! Print it from the list below.";
            } else {
                $error = "Error generating certificate: " . $conn->error;
            }
            $insert_stmt->close();
        }
        $check_stmt->close();
    } else {
        $error = "Error: Invalid Student or Class selected.";
    }
}

// --- Handle Deletion (GET) ---
if (isset($_GET['delete_id'])) {
    $delete_id = (int) $_GET['delete_id'];
    $delete_stmt = $conn->prepare("DELETE FROM school_leaving_certificates WHERE id = ?");
    $delete_stmt->bind_param("i", $delete_id);
    if ($delete_stmt->execute()) {
        $message = "Success: Certificate ID {$delete_id} deleted successfully!";
    } else {
        $error = "Error deleting certificate: " . $conn->error;
    }
    $delete_stmt->close();
}

// --- Data Fetching ---
// Modified query to include class information for left students
$students_query = "
    SELECT 
        s.id, 
        s.first_name, 
        s.last_name,
        s.father_name,
        s.leaving_date,
        se.class_id,
        ac.name as class_name
    FROM students s
    LEFT JOIN student_enrollments se ON s.id = se.student_id AND se.session_id = ?
    LEFT JOIN academic_classes ac ON se.class_id = ac.id
    WHERE s.status = 'withdrawn' OR s.status = 'inactive' OR s.leaving_date IS NOT NULL 
    ORDER BY s.first_name ASC
";
$stmt = $conn->prepare($students_query);
$stmt->bind_param("i", $current_session_id);
$stmt->execute();
$students_result = $stmt->get_result();
$students = $students_result->fetch_all(MYSQLI_ASSOC);
$stmt->close();

$classes = $conn->query("SELECT id, name FROM academic_classes ORDER BY order_no ASC")->fetch_all(MYSQLI_ASSOC);

// Fetch Generated SLC List
$slc_query = "
    SELECT 
        slc.id, slc.issue_date, slc.is_edited, 
        s.first_name, s.last_name, s.father_name, s.leaving_date,
        ac.name as leaving_class_name
    FROM school_leaving_certificates slc
    JOIN students s ON slc.student_id = s.id
    JOIN academic_classes ac ON slc.leaving_class_id = ac.id
    ORDER BY slc.id DESC
";
$slc_list = $conn->query($slc_query)->fetch_all(MYSQLI_ASSOC);
?>

<h1 class="mb-4">School Leaving Certificate Management <i class="fas fa-file-invoice"></i></h1>

<?php if ($message): ?>
    <div class="alert alert-success"><?php echo $message; ?></div>
<?php endif; ?>
<?php if ($error): ?>
    <div class="alert alert-danger"><?php echo $error; ?></div>
<?php endif; ?>

<div class="card p-4 mb-4 shadow-sm">
    <h5 class="card-title mb-4"><i class="fas fa-plus-circle"></i> Generate New Certificate (For Left Students Only)</h5>
    
    <form method="POST" action="generate_slc.php" class="row g-3 align-items-start" id="slcForm">
        <!-- Student Selection -->
        <div class="col-md-5">
            <label class="form-label fw-bold">Select Student:</label>
            <select name="student_id" id="student_id" class="form-control" required onchange="updateClassField()">
                <option value="">-- Select Left Student --</option>
                <?php foreach ($students as $student): ?>
                    <option value="<?php echo $student['id']; ?>" 
                            data-class-id="<?php echo $student['class_id'] ?? ''; ?>" 
                            data-class-name="<?php echo htmlspecialchars($student['class_name'] ?? 'Not Available'); ?>">
                        <?php echo htmlspecialchars($student['first_name'] . ' ' . $student['last_name']); ?> 
                        (ID: <?php echo $student['id']; ?>)
                        <?php if ($student['class_name']): ?>
                            - <?php echo htmlspecialchars($student['class_name']); ?>
                        <?php endif; ?>
                    </option>
                <?php endforeach; ?>
            </select>
            <small class="form-text text-muted">Only students with 'Inactive' or 'Withdrawn' status are shown.</small>
        </div>
        
        <!-- Class Display -->
        <div class="col-md-4">
            <label class="form-label fw-bold">Class Passed/Last Attended:</label>
            <div id="classDisplay" class="form-control bg-light" style="height: 38px; line-height: 24px; padding: 6px 12px;">
                -- Select Student First --
            </div>
            <input type="hidden" name="leaving_class_id" id="hidden_leaving_class_id" value="">
            <small class="form-text text-muted">Class auto-selected based on student</small>
        </div>
        
        <!-- Generate Button -->
        <div class="col-md-3">
            <label class="form-label fw-bold">&nbsp;</label>
            <button type="submit" name="generate_slc" class="btn btn-primary w-100" id="generateBtn" disabled style="height: 38px;">
                <i class="fas fa-certificate"></i> Generate SLC
            </button>
        </div>
    </form>
</div>

<!-- JavaScript for auto class selection -->
<script>
function updateClassField() {
    const studentSelect = document.getElementById('student_id');
    const classDisplay = document.getElementById('classDisplay');
    const hiddenClassInput = document.getElementById('hidden_leaving_class_id');
    const generateBtn = document.getElementById('generateBtn');
    
    const selectedOption = studentSelect.options[studentSelect.selectedIndex];
    const classId = selectedOption.getAttribute('data-class-id');
    const className = selectedOption.getAttribute('data-class-name');
    
    if (classId && className && classId !== '') {
        // Update the class display
        classDisplay.textContent = className;
        classDisplay.className = 'form-control bg-light text-success fw-bold';
        
        // Update the hidden input for form submission
        hiddenClassInput.value = classId;
        
        // Enable generate button
        generateBtn.disabled = false;
        generateBtn.className = 'btn btn-success w-100';
    } else {
        // Reset if no class found
        classDisplay.textContent = '-- Class Not Available --';
        classDisplay.className = 'form-control bg-light text-danger';
        hiddenClassInput.value = '';
        generateBtn.disabled = true;
        generateBtn.className = 'btn btn-primary w-100';
    }
}

// Initialize on page load
document.addEventListener('DOMContentLoaded', function() {
    updateClassField();
});
</script>

<!-- Generated Certificates Section -->
<div class="card p-4 shadow-sm">
    <div class="d-flex justify-content-between align-items-center mb-3">
        <h5 class="card-title mb-0"><i class="fas fa-list-alt"></i> Generated Certificates List</h5>
        
        <!-- Search Box for Generated Certificates -->
        <div class="col-md-4">
            <div class="input-group">
                <div class="input-group-prepend">
                    <span class="input-group-text bg-light border-end-0"><i class="fas fa-search"></i></span>
                </div>
                <input type="text" id="certificateSearch" class="form-control border-start-0" placeholder="Search certificates..." onkeyup="filterCertificates()">
            </div>
        </div>
    </div>
    
    <?php if (count($slc_list) > 0): ?>
        <div class="table-responsive">
            <table class="table table-bordered table-striped table-hover mt-3" id="certificatesTable">
                <thead class="table-dark">
                    <tr>
                        <th width="60">ID</th>
                        <th>Student Name</th>
                        <th>Leaving Class</th>
                        <th>Leaving Date</th>
                        <th>Issue Date</th>
                        <th width="100">Status</th>
                        <th width="200">Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($slc_list as $slc): ?>
                    <tr data-search-text="<?php echo htmlspecialchars(strtolower($slc['id'] . ' ' . $slc['first_name'] . ' ' . $slc['last_name'] . ' ' . $slc['leaving_class_name'])); ?>">
                        <td class="fw-bold"><?php echo $slc['id']; ?></td>
                        <td>
                            <strong><?php echo htmlspecialchars($slc['first_name'] . ' ' . $slc['last_name']); ?></strong>
                            <br><small class="text-muted">Father: <?php echo htmlspecialchars($slc['father_name']); ?></small>
                        </td>
                        <td><span class="badge bg-primary"><?php echo htmlspecialchars($slc['leaving_class_name']); ?></span></td>
                        <td><?php echo $slc['leaving_date'] ? date('d-M-Y', strtotime($slc['leaving_date'])) : '<span class="text-muted">N/A</span>'; ?></td>
                        <td><strong><?php echo date('d-M-Y', strtotime($slc['issue_date'])); ?></strong></td>
                        <td>
                            <span class="badge badge-<?php echo $slc['is_edited'] ? 'warning' : 'success'; ?>">
                                <?php echo $slc['is_edited'] ? 'Edited' : 'Generated'; ?>
                            </span>
                        </td>
                        <td>
                            <div class="btn-group btn-group-sm" role="group">
                                <a href="slc_template.php?slc_id=<?php echo $slc['id']; ?>" target="_blank" class="btn btn-success" title="Print">
                                    <i class="fas fa-print"></i>
                                </a>
                                <a href="edit_slc.php?slc_id=<?php echo $slc['id']; ?>" class="btn btn-info" title="Edit">
                                    <i class="fas fa-edit"></i>
                                </a>
                                <a href="generate_slc.php?delete_id=<?php echo $slc['id']; ?>" class="btn btn-danger" title="Delete" onclick="return confirm('Are you sure you want to delete this certificate?')">
                                    <i class="fas fa-trash"></i>
                                </a>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php else: ?>
        <div class="alert alert-warning text-center py-4">
            <i class="fas fa-exclamation-triangle fa-2x mb-3"></i>
            <h5>No School Leaving Certificates Generated Yet</h5>
            <p class="mb-0">Generate your first certificate using the form above.</p>
        </div>
    <?php endif; ?>
</div>

<!-- JavaScript for certificate search -->
<script>
function filterCertificates() {
    const searchInput = document.getElementById('certificateSearch');
    const table = document.getElementById('certificatesTable');
    const rows = table.getElementsByTagName('tbody')[0].getElementsByTagName('tr');
    const searchTerm = searchInput.value.toLowerCase();
    
    for (let i = 0; i < rows.length; i++) {
        const searchText = rows[i].getAttribute('data-search-text');
        if (searchText.includes(searchTerm)) {
            rows[i].style.display = '';
        } else {
            rows[i].style.display = 'none';
        }
    }
}
</script>

<?php
if (isset($conn) && is_object($conn)) {
    $conn->close();
}
include_once('includes/footer.php');
ob_end_flush();
?>