<?php
// FILE: admin/salary_generate.php - FINAL FIXED CODE (Editable Deductions)
include_once('includes/header.php'); 
include_once('includes/sidebar.php'); 
include_once('../config/db.php'); // Ensure this path is correct

$conn = getDBConnection();
$message = $_GET['msg'] ?? $_GET['err'] ?? '';
$is_error = isset($_GET['err']);
$current_month = date('Y-m');

// 1. Fetch Staff List for the form
$staff_result = $conn->query("SELECT id, name, designation, salary FROM teachers WHERE status='active' ORDER BY name ASC");
$staff_members = $staff_result->fetch_all(MYSQLI_ASSOC);

// 2. Fetch Generated Slips List (Last 4 months for robust visibility)
$slips_query = "
    SELECT 
        tss.id, tss.month_year, tss.net_paid, tss.status, tss.paid_date, tss.teacher_id, tss.total_deduction,
        t.name as staff_name, t.emp_no
    FROM teacher_salary_slips tss
    JOIN teachers t ON tss.teacher_id = t.id
    WHERE tss.month_year >= DATE_SUB(LAST_DAY(NOW()), INTERVAL 4 MONTH) 
    ORDER BY tss.month_year DESC, t.name ASC
";
$slips_result = $conn->query($slips_query);
$slips = $slips_result->fetch_all(MYSQLI_ASSOC);

$conn->close();
?>

<h1 class="mb-4">Staff Salary Slip Management</h1>

<?php if ($message): ?>
    <div class="alert alert-<?php echo $is_error ? 'danger' : 'success'; ?>"><?php echo htmlspecialchars($message); ?></div>
<?php endif; ?>

<div class="card mb-4">
    <div class="card-header">
        Generate & Print Salary Slips
    </div>
    <div class="card-body">
        <form action="salary_process.php" method="POST" id="salaryGenerationForm">
            <input type="hidden" name="action" value="generate_slips">
            
            <div class="row">
                <div class="form-group col-md-3">
                    <label for="month_year">Select Month:</label>
                    <input type="month" name="month_year" id="month_year" class="form-control" value="<?php echo $current_month; ?>" required>
                </div>
                <div class="form-group col-md-3">
                    <label for="teacher_id">Select Staff (Optional):</label>
                    <select name="teacher_id" id="teacher_id" class="form-control">
                        <option value="all">-- All Active Staff --</option>
                        <?php foreach ($staff_members as $staff): ?>
                            <option value="<?php echo $staff['id']; ?>"><?php echo htmlspecialchars($staff['name']) . ' (' . htmlspecialchars($staff['designation']) . ')'; ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group col-md-3 align-self-end">
                    <button type="submit" class="btn btn-primary btn-block">Generate Slips</button>
                </div>
                <div class="form-group col-md-3 align-self-end">
                    <button type="button" class="btn btn-secondary btn-block" onclick="printAllSlips()">Print All Slips</button>
                </div>
            </div>
        </form>
    </div>
</div>

<div class="card">
    <div class="card-header">
        Generated Salary Slips (Last 3 Months)
    </div>
    <div class="card-body">
        <table class="table table-bordered table-striped table-sm" id="slipsTable">
            <thead>
                <tr>
                    <th>Slip ID</th>
                    <th>Staff Name</th>
                    <th>Month</th>
                    <th>Net Paid</th>
                    <th>Deduction</th> 
                    <th>Status</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($slips as $slip): ?>
                <tr id="slip-row-<?php echo $slip['id']; ?>">
                    <td><?php echo htmlspecialchars($slip['id']); ?></td>
                    <td><?php echo htmlspecialchars($slip['staff_name']); ?> (<?php echo htmlspecialchars($slip['emp_no']); ?>)</td>
                    <td><?php echo date('F Y', strtotime($slip['month_year'])); ?></td>
                    <td id="net-paid-<?php echo $slip['id']; ?>">Rs. **<?php echo number_format($slip['net_paid']); ?>**</td>
                    <td id="deduction-total-<?php echo $slip['id']; ?>">Rs. **<?php echo number_format($slip['total_deduction']); ?>**</td> 
                    <td><span class="badge badge-<?php 
                        if($slip['status'] === 'paid') echo 'success'; 
                        elseif($slip['status'] === 'cancelled') echo 'danger'; 
                        else echo 'warning';
                    ?>"><?php echo ucfirst($slip['status']); ?></span></td>
                    <td>
                        <button type="button" class="btn btn-sm btn-warning" 
                                onclick="openDeductionModal(<?php echo $slip['id']; ?>, '<?php echo htmlspecialchars($slip['staff_name']); ?>')">
                            Deduct/Edit
                        </button>
                        <a href="salary_slip_print.php?id=<?php echo $slip['id']; ?>" target="_blank" class="btn btn-sm btn-info">Print</a>
                        <?php if ($slip['status'] === 'generated'): ?>
                            <a href="salary_process.php?action=mark_paid&id=<?php echo $slip['id']; ?>" class="btn btn-sm btn-success">Mark Paid</a>
                        <?php endif; ?>
                        <a href="salary_process.php?action=delete_slip&id=<?php echo $slip['id']; ?>" 
                           onclick="return confirm('Are you sure you want to delete this salary slip? This action cannot be undone.');" 
                           class="btn btn-sm btn-danger">Delete</a>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</div>

<?php include_once('includes/footer.php'); ?>

<div class="modal fade" id="deductionModal" tabindex="-1" role="dialog" aria-labelledby="deductionModalLabel" aria-hidden="true">
  <div class="modal-dialog" role="document">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="deductionModalLabel">Manage Deductions for <strong id="modal_staff_name"></strong></h5>
        <button type="button" class="close" data-dismiss="modal" aria-label="Close">
          <span aria-hidden="true">&times;</span>
        </button>
      </div>
      <div class="modal-body">
            <input type="hidden" id="modal_slip_id">
            
            <h6>Current Deductions:</h6>
            <div id="deductions_list_container" class="mb-3">
                </div>

            <hr>

            <h6><span id="form_mode">Add</span> New Deduction</h6>
            <form id="deductionForm" onsubmit="handleDeductionSubmit(event)">
                <input type="hidden" name="action" id="form_action" value="add_deduction">
                <input type="hidden" name="slip_id" id="form_slip_id">
                <input type="hidden" name="deduction_id" id="form_deduction_id" value="0">

                <div class="form-group">
                    <label for="amount">Amount (Rs.)</label>
                    <input type="number" step="0.01" name="amount" id="amount" class="form-control" min="0" required>
                </div>
                <div class="form-group">
                    <label for="reason">Reason</label>
                    <input type="text" name="reason" id="reason" class="form-control" required placeholder="e.g., Loan Installment">
                </div>
                <button type="submit" class="btn btn-primary"><span id="submit_button_text">Add Deduction</span></button>
                <button type="button" class="btn btn-secondary" onclick="resetDeductionForm()">Cancel/Add New</button>
            </form>
            <div id="deduction_message" class="mt-2" style="display: none;"></div>
      </div>
    </div>
  </div>
</div>

<script>
// --- GLOBAL FUNCTIONS ---

// Function to format number as currency (e.g., 1000 to 1,000.00)
function formatCurrency(number) {
    return Number(number).toLocaleString('en-IN', {
        minimumFractionDigits: 2,
        maximumFractionDigits: 2
    });
}

// Function to reset the form for adding new deduction
function resetDeductionForm() {
    document.getElementById('form_mode').innerText = 'Add';
    document.getElementById('form_action').value = 'add_deduction';
    document.getElementById('form_deduction_id').value = '0';
    document.getElementById('submit_button_text').innerText = 'Add Deduction';
    document.getElementById('deductionForm').reset();
    document.getElementById('deductionForm').classList.remove('editing-mode');
}

// Function to populate the form for editing an existing deduction
function editDeduction(id, amount, reason) {
    document.getElementById('form_mode').innerText = 'Edit';
    document.getElementById('form_action').value = 'update_deduction';
    document.getElementById('form_deduction_id').value = id;
    document.getElementById('submit_button_text').innerText = 'Save Changes';
    document.getElementById('amount').value = amount;
    document.getElementById('reason').value = reason;
    document.getElementById('deductionForm').classList.add('editing-mode');
}

// Function to delete a deduction
function deleteDeduction(deductionId, staffName) {
    if (!confirm(`Are you sure you want to delete this deduction item for ${staffName}?`)) {
        return;
    }

    const slipId = document.getElementById('modal_slip_id').value;
    const formData = new FormData();
    formData.append('action', 'delete_deduction');
    formData.append('slip_id', slipId);
    formData.append('deduction_id', deductionId);

    fetch('deduction_process.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        showMessage(data.message, data.success ? 'success' : 'danger');
        if (data.success) {
            updateSlipRow(slipId, data.totals.net_paid, data.totals.total_deduction);
            loadDeductions(slipId, staffName); // Reload the list inside the modal
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showMessage('An error occurred during deletion.', 'danger');
    });
}

// Function to update the main table row after deduction change
function updateSlipRow(slipId, netPaid, totalDeduction) {
    document.getElementById(`net-paid-${slipId}`).innerHTML = `Rs. **${formatCurrency(netPaid)}**`;
    document.getElementById(`deduction-total-${slipId}`).innerHTML = `Rs. **${formatCurrency(totalDeduction)}**`;
}

// Function to show messages in the modal
function showMessage(msg, type) {
    const msgBox = document.getElementById('deduction_message');
    msgBox.innerText = msg;
    msgBox.className = `mt-2 alert alert-${type}`;
    msgBox.style.display = 'block';
    setTimeout(() => {
        msgBox.style.display = 'none';
    }, 5000);
}

// --- MAIN HANDLERS ---

// Handles form submission (Add or Update)
function handleDeductionSubmit(event) {
    event.preventDefault();
    
    const slipId = document.getElementById('modal_slip_id').value;
    const staffName = document.getElementById('modal_staff_name').innerText;
    
    // Set the slip_id in the hidden form field
    document.getElementById('form_slip_id').value = slipId;

    const form = document.getElementById('deductionForm');
    const formData = new FormData(form);

    fetch('deduction_process.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        showMessage(data.message, data.success ? 'success' : 'danger');
        if (data.success) {
            updateSlipRow(slipId, data.totals.net_paid, data.totals.total_deduction);
            loadDeductions(slipId, staffName); // Reload the list
            resetDeductionForm(); // Reset form for new entry
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showMessage('An error occurred during save.', 'danger');
    });
}

// Loads existing deductions into the modal
function loadDeductions(slipId, staffName) {
    const container = document.getElementById('deductions_list_container');
    container.innerHTML = 'Loading...';

    fetch(`deduction_process.php?action=fetch_deductions&slip_id=${slipId}`)
        .then(response => response.json())
        .then(data => {
            container.innerHTML = '';
            if (data.success && data.deductions.length > 0) {
                let html = '<ul class="list-group">';
                data.deductions.forEach(deduction => {
                    const amount = Number(deduction.amount).toFixed(2);
                    html += `
                        <li class="list-group-item d-flex justify-content-between align-items-center">
                            <div>
                                **Rs. ${formatCurrency(deduction.amount)}** (${deduction.reason})
                                <small class="text-muted d-block">(${deduction.date})</small>
                            </div>
                            <div>
                                <button type="button" class="btn btn-sm btn-info mr-2" 
                                    onclick="editDeduction(${deduction.id}, ${amount}, '${deduction.reason.replace(/'/g, "\\'")}')">
                                    Edit
                                </button>
                                <button type="button" class="btn btn-sm btn-danger" 
                                    onclick="deleteDeduction(${deduction.id}, '${staffName.replace(/'/g, "\\'")}')">
                                    X
                                </button>
                            </div>
                        </li>
                    `;
                });
                html += '</ul>';
                container.innerHTML = html;
            } else {
                container.innerHTML = '<p class="text-muted">No deductions applied yet.</p>';
            }
        })
        .catch(error => {
            console.error('Error fetching deductions:', error);
            container.innerHTML = '<p class="text-danger">Error loading deductions.</p>';
        });
}

// Opens the modal and triggers data loading
function openDeductionModal(slipId, staffName) {
    document.getElementById('modal_slip_id').value = slipId;
    document.getElementById('modal_staff_name').innerText = staffName;
    resetDeductionForm(); // Reset form to 'Add New' mode
    
    // Load existing deductions
    loadDeductions(slipId, staffName);

    // Show the modal (assuming Bootstrap's jQuery is available)
    $('#deductionModal').modal('show'); 
}

// JAVASCRIPT FOR BULK PRINT
function printAllSlips() {
    const monthYear = document.getElementById('month_year').value;
    if (monthYear) {
        const url = `salary_slips_bulk_print.php?month=${monthYear}`;
        window.open(url, '_blank');
    } else {
        alert('Please select a month before attempting to print all slips.');
    }
}
</script>