<?php
// admin/salary_slip_print.php - FINAL FIXED CODE with Hybrid Key Logic
include_once('../config/db.php');
$conn = getDBConnection();

$slip_id = (int)($_GET['id'] ?? 0);

if ($slip_id == 0) {
    die("Invalid Slip ID.");
}

// 1. Fetch ALL School Settings from Key-Value Table
$settings_data = [];
try {
    // Fetch all settings data
    $result = $conn->query("SELECT setting_key, setting_value FROM settings");
    if ($result) {
        while ($row = $result->fetch_assoc()) {
            // Trim() function use kiya hai taake keys aur values mein extra space masla na kare
            $settings_data[trim($row['setting_key'])] = trim($row['setting_value']);
        }
        $result->free();
    }
} catch (mysqli_sql_exception $e) {
    error_log("Settings table query failed: " . $e->getMessage());
}

// --- School/Header Details Mapping ---
// Note: settings table mein keys inconsistent hain (e.g., institute.name vs institute logo path)
// Hum har key ko uski possible variations se check karenge.

// Institute Name, Address, Phone (Aapki attendance file mein yeh keys use hui thin)
$school_name = $settings_data['institute.name'] ?? $settings_data['institute name'] ?? "Institute Name Not Set";
$school_address = $settings_data['institute.address'] ?? $settings_data['institute address'] ?? "Address Not Set";
$school_phone = $settings_data['institute.phone'] ?? $settings_data['institute phone'] ?? "Phone Not Set";

// 🔑 FINAL LOGO FIX: 'institute.logo_path' ya 'institute logo path' (Screenshot wali key) se value uthao
$logo_db_path = $settings_data['institute.logo_path'] ?? $settings_data['institute logo path'] ?? '';

// Logo Path Logic: Agar path empty nahi hai, to '../' prefix add karein (Kyunki hum admin folder mein hain).
// Fallback path ab 'uploads/default.png' use kiya hai, jaisa pichli discussion mein tha.
$logo_url = '';
if (!empty($logo_db_path)) {
    // Agar path 'uploads/...' hai, to '../' add karein
    $logo_url = '../' . $logo_db_path;
} else {
    // Agar database mein path nahi mila, to default logo (jo uploads/default.png mein hona chahiye)
    $logo_url = '../uploads/default.png';
}


// 2. Fetch Main Slip Details and Staff Info
$slip_query = "
    SELECT 
        tss.*, t.name, t.emp_no, t.designation, t.cnic
    FROM teacher_salary_slips tss
    JOIN teachers t ON tss.teacher_id = t.id
    WHERE tss.id = ?
";
$stmt = $conn->prepare($slip_query);
$stmt->bind_param("i", $slip_id);
$stmt->execute();
$slip_data = $stmt->get_result()->fetch_assoc();
$stmt->close();

if (!$slip_data) {
    die("Salary Slip not found.");
}

// 3. Fetch Slip Items (Allowances and Deductions)
$items_query = "SELECT item_type, item_description, amount FROM teacher_salary_items WHERE slip_id = ?";
$stmt = $conn->prepare($items_query);
$stmt->bind_param("i", $slip_id);
$stmt->execute();
$items_result = $stmt->get_result();

$allowances = [];
$deductions = [];
$attendance_info = null;

while ($item = $items_result->fetch_assoc()) {
    if ($item['item_type'] === 'allowance') {
        $allowances[] = $item;
    } elseif ($item['item_type'] === 'deduction') {
        $deductions[] = $item;
    } elseif ($item['item_type'] === 'attendance') {
        $attendance_info = $item;
    }
}

$stmt->close();
$conn->close();

$total_deduction = array_sum(array_column($deductions, 'amount'));

// Simple utility function for formatting
function format_amount($amount) {
    return 'Rs. ' . number_format($amount, 2);
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Salary Slip #<?php echo $slip_data['id']; ?></title>
    <style>
        /* Basic Print Styles */
        body { font-family: Arial, sans-serif; margin: 0; padding: 20px; font-size: 10pt; }
        .slip-container { width: 800px; margin: 0 auto; border: 1px solid #333; padding: 20px; }
        /* Header Styles for Logo and Text Alignment */
        .print-header { 
            display: flex; 
            justify-content: space-between; 
            align-items: flex-start; 
            margin-bottom: 20px; 
            border-bottom: 2px solid #333; 
            padding-bottom: 10px; 
        }
        .header-text { 
            text-align: center; 
            flex-grow: 1; 
            padding: 0 10px;
        }
        .header-text h2 { margin: 0; font-size: 16pt; }
        .header-text p { margin: 2px 0; font-size: 9pt; }
        .header-logo { 
            width: 100px; /* Logo Box Size */
            height: 100px; 
            flex-shrink: 0; 
        }
        .header-logo img { 
            width: 100%; 
            height: 100%; 
            object-fit: contain; /* Logo fit adjustment */
        }
        
        .title-bar { text-align: center; background-color: #f2f2f2; padding: 5px; font-weight: bold; margin-bottom: 10px; border: 1px solid #ccc; }
        
        .details-table { width: 100%; margin-bottom: 15px; border-collapse: collapse; }
        .details-table td { padding: 5px 10px; border-bottom: 1px dashed #ccc; }
        .items-section { width: 100%; border-collapse: collapse; margin-top: 20px; }
        .items-section th, .items-section td { border: 1px solid #ccc; padding: 8px; text-align: left; }
        .items-section th { background-color: #e9e9e9; text-align: center; }
        .summary-box { width: 100%; margin-top: 20px; }
        .summary-box td { padding: 5px 10px; }
        .net-total { font-size: 14pt; font-weight: bold; background-color: #e0ffe0; }
        .footer { margin-top: 50px; padding-top: 10px; border-top: 1px solid #ccc; text-align: center; font-size: 8pt; }
        @media print {
            body { background: none; }
            .slip-container { border: none; padding: 0; }
        }
    </style>
</head>
<body>
    <div class="slip-container">
        <div class="print-header">
            <div class="header-logo">
                <img src="<?php echo htmlspecialchars($logo_url); ?>" alt="School Logo" 
                     onerror="this.onerror=null;this.src='../uploads/default.png';">
            </div>
            <div class="header-text">
                <h2><?php echo htmlspecialchars($school_name); ?></h2>
                <p><?php echo htmlspecialchars($school_address); ?></p>
                <p>Phone: <?php echo htmlspecialchars($school_phone); ?></p>
            </div>
            <div style="width: 100px; flex-shrink: 0;">&nbsp;</div> </div>
        
        <div class="title-bar">
            STAFF MONTHLY SALARY SLIP
        </div>

        <table class="details-table">
            <tr>
                <td style="width: 50%;"><strong>Slip Month:</strong> <?php echo date('F Y', strtotime($slip_data['month_year'])); ?></td>
                <td style="width: 50%;"><strong>Slip ID:</strong> <?php echo htmlspecialchars($slip_data['id']); ?></td>
            </tr>
        </table>
        
        <table class="details-table">
            <tr>
                <td style="width: 50%;"><strong>Staff Name:</strong> <?php echo htmlspecialchars($slip_data['name']); ?></td>
                <td style="width: 50%;"><strong>Employee No:</strong> <?php echo htmlspecialchars($slip_data['emp_no'] ?? '-'); ?></td>
            </tr>
            <tr>
                <td><strong>Designation:</strong> <?php echo htmlspecialchars($slip_data['designation']); ?></td>
                <td><strong>CNIC:</strong> <?php echo htmlspecialchars($slip_data['cnic']); ?></td>
            </tr>
            <tr>
                <td colspan="2">
                    <?php if ($attendance_info): ?>
                        <strong>Attendance Info:</strong> <?php echo htmlspecialchars($attendance_info['item_description']); ?>
                    <?php endif; ?>
                </td>
            </tr>
        </table>

        <table class="items-section">
            <thead>
                <tr>
                    <th style="width: 45%;">Allowances / Earnings</th>
                    <th style="width: 10%;">Amount</th>
                    <th style="width: 35%;">Deductions</th>
                    <th style="width: 10%;">Amount</th>
                </tr>
            </thead>
            <tbody>
                <?php
                // Filter Base Monthly Salary to show first
                $base_salary = array_filter($allowances, function($item) {
                    return $item['item_description'] === 'Base Monthly Salary';
                });
                $other_allowances = array_filter($allowances, function($item) {
                    return $item['item_description'] !== 'Base Monthly Salary';
                });
                $sorted_allowances = array_merge($base_salary, $other_allowances);
                
                $max_rows = max(count($sorted_allowances), count($deductions));
                for ($i = 0; $i < $max_rows; $i++):
                    $allowance = $sorted_allowances[$i] ?? null;
                    $deduction = $deductions[$i] ?? null;
                ?>
                <tr>
                    <td><?php echo htmlspecialchars($allowance['item_description'] ?? '-'); ?></td>
                    <td style="text-align: right;"><?php echo $allowance ? format_amount($allowance['amount']) : '-'; ?></td>
                    <td><?php echo htmlspecialchars($deduction['item_description'] ?? '-'); ?></td>
                    <td style="text-align: right;"><?php echo $deduction ? format_amount($deduction['amount']) : '-'; ?></td>
                </tr>
                <?php endfor; ?>
            </tbody>
        </table>

        <table class="summary-box">
            <tr>
                <td style="width: 45%; border-top: 1px solid #333;"><strong>Total Gross Salary:</strong></td>
                <td style="width: 10%; text-align: right; border-top: 1px solid #333; font-weight: bold;"><?php echo format_amount($slip_data['gross_salary']); ?></td>
                <td style="width: 35%; border-top: 1px solid #333;"><strong>Total Deductions:</strong></td>
                <td style="width: 10%; text-align: right; border-top: 1px solid #333; font-weight: bold;"><?php echo format_amount($total_deduction); ?></td>
            </tr>
            <tr>
                <td colspan="3" class="net-total" style="text-align: right; background-color: #ddf;">Net Payable Amount:</td>
                <td class="net-total" style="text-align: right; background-color: #ddf;"><?php echo format_amount($slip_data['net_paid']); ?></td>
            </tr>
        </table>

        <div class="footer">
            <p>Status: <?php echo ucfirst($slip_data['status']); ?> <?php echo $slip_data['paid_date'] ? '(Paid on: ' . $slip_data['paid_date'] . ')' : ''; ?></p>
            <p>This is a computer generated slip and does not require a signature.</p>
        </div>
        
        <button onclick="window.print()" class="btn btn-primary" style="margin-top: 20px; display: block; width: 100%; padding: 10px;">Print Slip</button>
    </div>
</body>
</html>