<?php
// FILE: admin/slc_template.php - School Leaving Certificate Print Template (Final Fix: Serial No. and Date Above Heading)

session_start();
ob_start();
include_once('../config/db.php');
$conn = getDBConnection();

// --- START: Settings & Session ID ---
$settings_data = [];
$settings_query = $conn->query("SELECT setting_key, setting_value FROM settings");
if ($settings_query) {
    while ($row = $settings_query->fetch_assoc()) {
        $settings_data[trim($row['setting_key'])] = trim($row['setting_value']);
    }
}
$settings['institute_name'] = $settings_data['institute.name'] ?? "SCHOOL NAME HERE";
$settings['institute_address'] = $settings_data['institute.address'] ?? "School Address, City, Contact Info";
$settings['institute_phone'] = $settings_data['institute.phone'] ?? "Phone Not Set";
$rawLogoPath = $settings_data['institute.logo_path'] ?? '';
$logoPathParts = explode('|', $rawLogoPath);
$cleanLogoPath = trim($logoPathParts[0]);
$settings['logo_url'] = !empty($cleanLogoPath) ? '../' . $cleanLogoPath : null;
// --- END: Settings & Session ID ---

$slc_id = (int) ($_GET['slc_id'] ?? 0);
$slc_data = null;

if ($slc_id > 0) {
    $query = "
        SELECT 
            slc.*, 
            s.first_name, s.last_name, s.father_name, s.dob, 
            ac.name as leaving_class_name
        FROM school_leaving_certificates slc
        JOIN students s ON slc.student_id = s.id
        JOIN academic_classes ac ON slc.leaving_class_id = ac.id
        WHERE slc.id = ?
    ";
    $stmt = $conn->prepare($query);
    $stmt->bind_param("i", $slc_id);
    $stmt->execute();
    $slc_data = $stmt->get_result()->fetch_assoc();
    $stmt->close();
}

if (!$slc_data) {
    die("Error: School Leaving Certificate Not Found or Invalid ID.");
}

$student_full_name = htmlspecialchars($slc_data['first_name'] . ' ' . $slc_data['last_name']);
$father_name = htmlspecialchars($slc_data['father_name']);
$dob = date('d/m/Y', strtotime($slc_data['dob']));
$leaving_class = htmlspecialchars($slc_data['leaving_class_name']);
$issue_date_db = $slc_data['issue_date'];
$issue_date_display = date('d/m/Y', strtotime($issue_date_db));
$sdo = "S/D/O"; 

// --- DYNAMIC INSTITUTE CODE LOGIC ---
$institute_name_words = explode(' ', $settings['institute_name']);
$institute_code = '';
foreach ($institute_name_words as $word) {
    if (!empty($word)) {
        $institute_code .= strtoupper($word[0]);
    }
}
if (empty($institute_code)) {
    $institute_code = 'SCH'; 
}

$slc_id_padded = str_pad($slc_id, 4, '0', STR_PAD_LEFT); 
$serial_no = "{$institute_code}/Certificate/{$slc_id_padded}";
// --- END: Serial Number Logic ---

?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>School Leaving Certificate - <?php echo $student_full_name; ?></title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">
    <style>
        /* Base styles for display */
        body { 
            font-family: serif; 
            background-color: #f7f3e8; 
            margin: 0; 
            padding: 20px;
            display: flex;
            flex-direction: column; 
            align-items: center; 
        }
        
        .no-print {
            text-align: center; 
            margin-bottom: 20px;
            width: 100%; 
        }

        .certificate-container {
            width: 95%; 
            max-width: 1000px; 
            margin: 0 auto;
            border: 5px solid #d4af37; 
            padding: 50px;
            background-color: #fff9e6; 
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
            display: flex;
            flex-direction: column;
            justify-content: space-between; 
            position: relative; 
            min-height: 500px;
        }

        .header { text-align: center; margin-bottom: 30px; }
        .header img { height: 100px; margin-bottom: 10px; }
        .header h1 { font-size: 36px; color: #333; margin: 0; }
        .header p { margin: 5px 0 0 0; color: #666; font-size: 14px; }
        
        /* Heading style */
        h2 { 
            text-align: center; 
            font-size: 28px; 
            margin-bottom: 30px; /* Space below heading */
            text-decoration: underline; 
            margin-top: 0; 
        }

        /* Container for Serial No and Date aligned with content area */
        .top-info {
            display: flex;
            justify-content: space-between;
            align-items: center;
            /* Moved up: Now positioned below the main header line but above H2 */
            margin: 0 auto 10px auto; 
            font-size: 16px;
            font-weight: bold;
            
            /* Alignment to Content Area */
            width: 90%; 
            text-align: left;
        }
        
        .content-area {
            text-align: center; 
            flex-grow: 1; 
            margin-bottom: 80px; 
        }
        .content { 
            font-size: 18px; 
            line-height: 1.8; 
            display: inline-block; 
            max-width: 90%; 
            text-align: left; 
        }
        .content p {
            text-align: left; 
            margin: 10px 0;
            display: block; 
        }

        .data-underline { border-bottom: 1px dotted #000; padding: 0 10px; font-weight: bold; }
        .date-underline { padding: 0 10px; font-weight: bold; }

        /* Signature Area */
        .signature-area { 
            position: absolute;
            bottom: 50px; 
            left: 50px;
            right: 50px;
            display: flex; 
            justify-content: flex-end; 
            width: calc(100% - 100px); 
            z-index: 10;
        }
        
        .signature-area .signature-item.date { display: none; } 
        
        .signature-item { text-align: center; width: 40%; }
        .signature-item p { margin-top: 5px; font-weight: bold; }
        .signature-item.principal { width: 40%; margin-left: auto; } 
        .signature-item.principal p { border-top: 1px solid #333; padding-top: 5px; }

        /* Print media specific styles */
        @media print {
            @page {
                size: landscape;
                margin: 0.1cm; 
                -webkit-print-color-adjust: exact;
                print-color-adjust: exact;
            }
            body { 
                margin: 0;
                padding: 0;
                width: 29.5cm; 
                height: 20.8cm; 
                display: flex; 
                align-items: center;
                justify-content: center;
                overflow: hidden; 
                box-sizing: border-box; 
            }
            
            .certificate-container { 
                border: 5px solid #d4af37 !important; 
                box-shadow: none; 
                width: 28.5cm; 
                height: 19.5cm; 
                padding: 15px; 
                margin: 0; 
                position: relative;
                display: flex; 
                flex-direction: column;
                justify-content: space-between;
                page-break-after: avoid !important;
                page-break-before: avoid !important;
                box-sizing: border-box;
            }
            
            .header { margin-bottom: 10px; } 
            
            .top-info {
                width: 90%; 
                margin: 0 auto 5px auto; /* Reduced margin in print */
            }

            h2 { margin-bottom: 15px; } 
            
            .content-area { margin-bottom: 50px; } 
            .no-print { display: none !important; }
            
            .signature-area {
                bottom: 15px; 
                left: 15px; 
                right: 15px;
                width: calc(100% - 30px); 
            }
        }
    </style>
</head>
<body>

<div class="no-print">
    <button onclick="window.print()" style="padding: 10px 20px; font-size: 16px; cursor: pointer;">
        <i class="fas fa-print"></i> Print Certificate
    </button>
    <p style="font-size: 12px; color: #555; margin-top: 10px; max-width: 400px; margin-left: auto; margin-right: auto;">
        **Note for Printing:** To remove date/time/URL, please check your printer settings (usually "More settings" -> "Options" -> uncheck "Headers and footers").
    </p>
</div>

<div class="certificate-container">
    
    <div> 
        <div class="header">
            <?php if ($settings['logo_url']): ?>
                <img src="<?php echo $settings['logo_url']; ?>" alt="School Logo" style="height: 100px; margin-bottom: 10px;"/>
            <?php endif; ?>
            <h1><?php echo htmlspecialchars($settings['institute_name']); ?></h1>
            <p><?php echo htmlspecialchars($settings['institute_address']); ?></p>
            <p>Phone: <?php echo htmlspecialchars($settings['institute_phone']); ?></p>
            <hr style="border-top: 2px solid #333; margin: 20px 0;">
        </div>
        
        <div class="top-info">
            <span class="serial-no">Sr No: <?php echo htmlspecialchars($serial_no); ?></span>
            <span class="issue-date">Issue Date: <?php echo htmlspecialchars($issue_date_display); ?></span>
        </div>
        
        <h2>School Leaving Certificate</h2>
        
        <div class="content-area">
            <div class="content">
                <p>
                    It is certified that <span class="data-underline"><?php echo $student_full_name; ?></span> 
                    **<?php echo $sdo; ?>** <span class="data-underline"><?php echo $father_name; ?></span> 
                    is a Bonafide student of this school. His/Her date of birth per school record is <span class="data-underline"><?php echo $dob; ?></span>. 
                    He/She has passed class <span class="data-underline"><?php echo $leaving_class; ?></span>. 
                </p>
                <p>
                    We wish him/her a very bright future. This certificate is being issued at the request of the above-named student for whatever legal purpose it may serve.
                </p>
                
                <?php if (!empty($slc_data['leaving_reason']) || !empty($slc_data['character_remark'])): ?>
                <hr style="margin: 20px 0;">
                <p><strong>Leaving Reason:</strong> <?php echo htmlspecialchars($slc_data['leaving_reason'] ?? 'N/A'); ?></p>
                <p><strong>Character/Conduct:</strong> <?php echo htmlspecialchars($slc_data['character_remark'] ?? 'Good'); ?></p>
                <?php endif; ?>
            </div>
        </div>
    </div>

    
    <div class="signature-area">
        <div class="signature-item principal">
            <br>
            <p>Principal / Headmaster Signature</p>
        </div>
    </div>
    
</div>

<?php
if (isset($conn) && is_object($conn)) {
    $conn->close();
}
ob_end_flush();
?>
</body>
</html>