<?php
// admin/student_add.php - New Student Admission Form (FIXED)
ob_start(); // ✅ Output Buffering Start
session_start();
// Include necessary files
include_once('../config/db.php'); 
include_once('includes/header.php'); 
include_once('includes/sidebar.php'); 

// Database Connection
$conn = getDBConnection();
$message = $_GET['msg'] ?? '';
$error = $_GET['err'] ?? '';
$form_data = []; // For holding form data in case of error

// --- Data Fetching for Dropdowns ---
$sessions_result = $conn->query("SELECT id, name FROM academic_sessions ORDER BY id DESC");
$sessions = $sessions_result ? $sessions_result->fetch_all(MYSQLI_ASSOC) : [];

$classes_result = $conn->query("SELECT id, name FROM academic_classes ORDER BY order_no ASC");
$classes = $classes_result ? $classes_result->fetch_all(MYSQLI_ASSOC) : [];

$sections_result = $conn->query("SELECT id, name FROM academic_sections WHERE status='active' ORDER BY name ASC");
$sections = $sections_result ? $sections_result->fetch_all(MYSQLI_ASSOC) : [];

// Active Session ID fetch karein - Check for sessions with future end date
$active_session_query = $conn->query("SELECT id, name FROM academic_sessions WHERE end_date > CURDATE() ORDER BY id DESC LIMIT 1");
$active_session = $active_session_query->fetch_assoc();

// Agar koi active session nahi hai
if (!$active_session) {
    $error = "❌ Koi active academic session nahi hai! Pehle academic sessions page ja kar naya session add karein.";
    $no_active_session = true;
} else {
    $current_session_id = $active_session['id'];
    $current_session_name = $active_session['name'];
    $no_active_session = false;
}

// --- Settings se Institute Name fetch karna ---
$settings_data = [];
$settings_query = $conn->query("SELECT setting_key, setting_value FROM settings");
if ($settings_query) {
    while ($row = $settings_query->fetch_assoc()) {
        $settings_data[trim($row['setting_key'])] = trim($row['setting_value']);
    }
}

// Institute name se prefixes generate karna - EXACTLY generate_slc.php ki tarah
$institute_name = $settings_data['institute.name'] ?? 'Mubeshir Computer Academy';

function generatePrefixesFromInstituteName($institute_name) {
    // Institute name ko words mein break karo
    $words = explode(' ', $institute_name);
    
    // Pehle 3 words ke first letters le kar abbreviation banayein
    $abbreviation = '';
    $word_count = 0;
    
    foreach ($words as $word) {
        if (trim($word) !== '' && $word_count < 3) {
            $abbreviation .= strtoupper(substr(trim($word), 0, 1));
            $word_count++;
        }
    }
    
    // Agar 3 words nahi milay, toh abbreviation ko complete karein
    if (strlen($abbreviation) < 3) {
        $abbreviation = str_pad($abbreviation, 3, 'X', STR_PAD_RIGHT);
    }
    
    // Pehla prefix: first 3 letters of abbreviation
    $prefix1 = substr($abbreviation, 0, 3);
    
    // Dusra prefix: hamesha "REG" rahega (Registration ke liye)
    $prefix2 = "REG";
    
    return [$prefix1, $prefix2];
}

// Function to generate unique GR Number (Registration No)
function generateGRNumber($conn, $institute_name) {
    // Institute name se prefixes generate karo
    list($prefix1, $prefix2) = generatePrefixesFromInstituteName($institute_name);
    
    // Get the last GR number from database with same prefixes
    $result = $conn->query("SELECT gr_no FROM students WHERE gr_no LIKE '{$prefix1}/{$prefix2}/%' ORDER BY id DESC LIMIT 1");
    
    if ($result && $result->num_rows > 0) {
        $last_gr = $result->fetch_assoc()['gr_no'];
        // Extract the number part and increment
        $parts = explode('/', $last_gr);
        $last_number = intval(end($parts));
        $new_number = $last_number + 1;
    } else {
        $new_number = 1; // First student
    }
    
    // Format number with leading zeros (4 digits)
    $formatted_number = str_pad($new_number, 4, '0', STR_PAD_LEFT);
    
    return "{$prefix1}/{$prefix2}/{$formatted_number}";
}

// Auto-generate GR number for new form
$auto_generated_gr_no = generateGRNumber($conn, $institute_name);

// --- Handle Form Submission ---
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    
    // Agar koi active session nahi hai to form submit block kar do
    if (!$active_session) {
        $error = "❌ Koi active academic session nahi hai! Pehle academic sessions page ja kar naya session add karein.";
        goto display_form;
    }
    
    // Store submitted data back into $form_data for form persistence in case of error
    $form_data = $_POST;
    
    // --------------------------------------------------------------------------------
    // 🛑 STEP 1: MAX STUDENT LIMIT CHECK (UPGRADE PLAN)
    // --------------------------------------------------------------------------------
    $MAX_STUDENT_LIMIT = 100; // ⚠️ Aap apni limit yahan set kar sakte hain
    
    try {
        $count_query = "SELECT COUNT(id) AS total_students FROM students";
        $result = $conn->query($count_query);

        if ($result && $row = $result->fetch_assoc()) {
            $current_student_count = (int)$row['total_students'];
            
            if ($current_student_count >= $MAX_STUDENT_LIMIT) {
                // Limit cross ho chuki hai, error set karo
                $error = "🚨 Plan Limit Reached! Aapke plan mein sirf {$MAX_STUDENT_LIMIT} bachche hi add ho sakte hain. Zyada bachche add karne ke liye, apna plan Upgrade karein.";
                goto display_form; 
            }
        }
    } catch (Exception $e) {
        $error = "Database Error while checking limit: " . $e->getMessage();
        goto display_form; 
    }
    // --------------------------------------------------------------------------------
    
    // --------------------------------------------------------------------------------
    // 🛑 STEP 2: Data Sanitation and Validation (Agar Limit Check pass ho jaye)
    // --------------------------------------------------------------------------------
    $first_name = trim($_POST['first_name'] ?? '');
    $last_name = trim($_POST['last_name'] ?? '');
    $father_name = trim($_POST['father_name'] ?? '');
    $gender = trim($_POST['gender'] ?? '');
    $dob = trim($_POST['dob'] ?? '');
    $admission_date = trim($_POST['admission_date'] ?? date('Y-m-d'));
    $roll_no = (int)($_POST['roll_no'] ?? 0);
    $class_id = (int)($_POST['class_id'] ?? 0);
    $section_id = (int)($_POST['section_id'] ?? 0);
    $session_id = (int)($_POST['session_id'] ?? $current_session_id);
    
    // FIXED: GR Number (Registration No) field - use existing gr_no column
    $gr_no = trim($_POST['gr_no'] ?? '');
    if (empty($gr_no)) {
        $gr_no = generateGRNumber($conn, $institute_name);
    }
    
    // FIXED: CNIC/B-Form field - agar empty hai toh NULL set karo
    $cnic_bform = trim($_POST['cnic_bform'] ?? '');
    $cnic_bform = empty($cnic_bform) ? NULL : $cnic_bform;
    
    // FIXED: Phone field - agar empty hai toh NULL set karo
    $phone = trim($_POST['phone'] ?? '');
    $phone = empty($phone) ? NULL : $phone;
    
    // FIXED: Email field - agar empty hai toh NULL set karo
    $email = trim($_POST['email'] ?? '');
    $email = empty($email) ? NULL : $email;
    
    // FIXED: Address field - agar empty hai toh NULL set karo
    $address = trim($_POST['address'] ?? '');
    $address = empty($address) ? NULL : $address;
    
    $status = 'active'; // Naye student ka status 'active' by default
    
    if (empty($first_name) || empty($father_name) || empty($gender) || empty($dob) || empty($class_id) || empty($roll_no)) {
        $error = "Zaroori field (Name, Father Name, Gender, DOB, Roll No, Class) bharna zaroori hai.";
        goto display_form;
    }
    
    // 🛑 NEW: Minimum Age Validation (3 years)
    $dob_timestamp = strtotime($dob);
    $today_timestamp = time();
    $age_in_seconds = $today_timestamp - $dob_timestamp;
    $age_in_years = $age_in_seconds / (365 * 24 * 60 * 60); // Convert seconds to years
    
    if ($age_in_years < 3) {
        $error = "❌ Bachay ki age kam az kam 3 saal honi chahiye! Aap ne jo date of birth di hai us ke mutabiq bachay ki age " . number_format($age_in_years, 1) . " saal hai.";
        goto display_form;
    }
    
    // Check if GR Number already exists
    $check_gr_no_q = $conn->prepare("SELECT COUNT(*) FROM students WHERE gr_no = ?");
    $check_gr_no_q->bind_param("s", $gr_no);
    $check_gr_no_q->execute();
    $check_gr_no_q->bind_result($gr_no_exists);
    $check_gr_no_q->fetch();
    $check_gr_no_q->close();
    
    if ($gr_no_exists > 0) {
        $error = "Error: GR Number **{$gr_no}** already exists. Kripya page refresh karein ya naya GR number generate karein.";
        goto display_form;
    }
    
    // Check if Roll No already exists in the selected Class/Section/Session
    $check_roll_no_q = $conn->prepare("SELECT COUNT(*) FROM student_enrollments WHERE session_id = ? AND class_id = ? AND section_id = ? AND roll_no = ?");
    $check_roll_no_q->bind_param("iiii", $session_id, $class_id, $section_id, $roll_no);
    $check_roll_no_q->execute();
    $check_roll_no_q->bind_result($roll_no_exists);
    $check_roll_no_q->fetch();
    $check_roll_no_q->close();
    
    if ($roll_no_exists > 0) {
        $error = "Error: Roll Number **{$roll_no}** is already assigned in this Class and Section for the selected Session. Kripya Roll Number change karein.";
        goto display_form;
    }

    // --------------------------------------------------------------------------------
    // 🛑 STEP 3: Database Insertion (Student aur Enrollment dono)
    // --------------------------------------------------------------------------------
    $conn->begin_transaction(); // Transaction start karein
    try {
        // 1. Student Table Insertion - gr_no column use karein
        $stmt = $conn->prepare("INSERT INTO students (gr_no, first_name, last_name, father_name, gender, dob, admission_date, cnic_bform, phone, email, address, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        
        // FIXED: NULL values properly handle karne ke liye
        $stmt->bind_param("ssssssssssss", 
            $gr_no,           // GR Number (Registration No)
            $first_name, 
            $last_name, 
            $father_name, 
            $gender, 
            $dob, 
            $admission_date, 
            $cnic_bform, 
            $phone, 
            $email, 
            $address, 
            $status
        );
        
        if (!$stmt->execute()) {
            throw new Exception("Student record insert nahi hua: " . $stmt->error);
        }
        $new_student_id = $conn->insert_id;
        $stmt->close();
        
        // 2. Enrollment Table Insertion (FIXED: enrollment_date column removed)
        $stmt_en = $conn->prepare("INSERT INTO student_enrollments (student_id, session_id, class_id, section_id, roll_no, status) VALUES (?, ?, ?, ?, ?, 'active')");
        $stmt_en->bind_param("iiiii", $new_student_id, $session_id, $class_id, $section_id, $roll_no);
        
        if (!$stmt_en->execute()) {
             throw new Exception("Student enrollment record insert nahi hua: " . $stmt_en->error);
        }
        $stmt_en->close();
        
        // Agar sab theek raha toh commit karo
        $conn->commit();
        
        $message = "Student **{$first_name} {$last_name}** successfully add ho gaya hai (GR No: {$gr_no}, Roll No: {$roll_no})!";
        header("Location: students.php?msg=" . urlencode($message));
        ob_end_flush(); // Output buffering flush aur end karein
        exit;
        
    } catch (Exception $e) {
        // Agar koi error aaya toh rollback karo
        $conn->rollback();
        $error = "Insertion Error: " . $e->getMessage();
        // $form_data will hold the values to redisplay
    }
}


display_form:
?>
<div class="container-fluid">
    <h1 class="mt-4">New Student Admission Form <i class="fas fa-user-plus"></i></h1>

    <?php if (!empty($message)): ?>
        <div class="alert alert-success"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>
    <?php if (!empty($error)): ?>
        <div class="alert alert-danger"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <?php if ($no_active_session): ?>
        <!-- Agar koi active session nahi hai to yeh section show karo -->
        <div class="card mb-4">
            <div class="card-header bg-danger text-white">
                <i class="fas fa-exclamation-triangle"></i> No Active Academic Session
            </div>
            <div class="card-body text-center">
                <h4 class="text-danger">❌ Student Add Nahin Kar Sakte!</h4>
                <p class="lead">Koi active academic session nahi hai. Pehle academic session add karna zaroori hai.</p>
                
                <div class="mt-4">
                    <a href="academic_sessions.php" class="btn btn-primary btn-lg mr-3">
                        <i class="fas fa-calendar-plus"></i> Add Academic Session
                    </a>
                    <a href="dashboard.php" class="btn btn-secondary btn-lg">
                        <i class="fas fa-tachometer-alt"></i> Go to Dashboard
                    </a>
                </div>
                
                <div class="mt-4 alert alert-info">
                    <strong>Instruction:</strong> 
                    <ol class="text-left">
                        <li>Academic Sessions page par jayein</li>
                        <li>"Add New Academic Session" form bharein</li>
                        <li>Session ko active karein</li>
                        <li>Phir student add kar sakte hain</li>
                    </ol>
                </div>
            </div>
        </div>
    <?php else: ?>
        <!-- Agar active session hai to normal form show karo -->
        <div class="card mb-4">
            <div class="card-header bg-primary text-white">
                <i class="fas fa-graduation-cap"></i> Student Details
            </div>
            <div class="card-body">
                <form action="student_add.php" method="POST">
                    
                    <!-- ROW 1: Registration No, Academic Session, Admission Date -->
                    <div class="form-row">
                        <div class="form-group col-md-4">
                            <label for="gr_no">Registration No (GR Number) *</label>
                            <input type="text" class="form-control" id="gr_no" name="gr_no" value="<?php echo htmlspecialchars($form_data['gr_no'] ?? $auto_generated_gr_no); ?>" required readonly style="background-color: #e9ecef;">
                            
                        </div>
                        <div class="form-group col-md-4">
                            <label for="session_id">Academic Session *</label>
                            <input type="text" class="form-control" value="<?php echo htmlspecialchars($current_session_name); ?>" readonly style="background-color: #e9ecef;">
                            <input type="hidden" name="session_id" value="<?php echo $current_session_id; ?>">
                            
                        </div>
                        <div class="form-group col-md-4">
                            <label for="admission_date">Admission Date *</label>
                            <input type="date" class="form-control" id="admission_date" name="admission_date" value="<?php echo htmlspecialchars($form_data['admission_date'] ?? date('Y-m-d')); ?>" required>
                        </div>
                    </div>

                    <!-- ROW 2: Class, Section, Roll No -->
                    <div class="form-row">
                        <div class="form-group col-md-4">
                            <label for="class_id">Class *</label>
                            <select id="class_id" name="class_id" class="form-control" required>
                                <option value="">Select Class</option>
                                <?php foreach ($classes as $class): ?>
                                    <option value="<?php echo $class['id']; ?>" 
                                        <?php if (($form_data['class_id'] ?? '') == $class['id']) echo 'selected'; ?>>
                                        <?php echo htmlspecialchars($class['name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group col-md-4">
                            <label for="section_id">Section *</label>
                            <select id="section_id" name="section_id" class="form-control" required>
                                <option value="">Select Section</option>
                                <?php foreach ($sections as $section): ?>
                                    <option value="<?php echo $section['id']; ?>" 
                                        <?php if (($form_data['section_id'] ?? '') == $section['id']) echo 'selected'; ?>>
                                        <?php echo htmlspecialchars($section['name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group col-md-4">
                            <label for="roll_no">Roll Number *</label>
                            <input type="number" class="form-control" id="roll_no" name="roll_no" value="<?php echo htmlspecialchars($form_data['roll_no'] ?? ''); ?>" required min="1">
                        </div>
                    </div>

                    <!-- ROW 3: Date of Birth -->
                    <div class="form-row">
                        <div class="form-group col-md-4">
                            <label for="dob">Date of Birth *</label>
                            <input type="date" class="form-control" id="dob" name="dob" value="<?php echo htmlspecialchars($form_data['dob'] ?? ''); ?>" required>
                            <small class="form-text text-muted">Bachay ki age kam az kam 3 saal honi chahiye</small>
                        </div>
                    </div>
                    
                    <hr>

                    <h5>Personal Information</h5>
                    <div class="form-row">
                        <div class="form-group col-md-4">
                            <label for="first_name">First Name *</label>
                            <input type="text" class="form-control" id="first_name" name="first_name" value="<?php echo htmlspecialchars($form_data['first_name'] ?? ''); ?>" required>
                        </div>
                        <div class="form-group col-md-4">
                            <label for="last_name">Last Name</label>
                            <input type="text" class="form-control" id="last_name" name="last_name" value="<?php echo htmlspecialchars($form_data['last_name'] ?? ''); ?>">
                        </div>
                        <div class="form-group col-md-4">
                            <label for="gender">Gender *</label>
                            <select id="gender" name="gender" class="form-control" required>
                                <option value="">Select Gender</option>
                                <option value="Male" <?php if (($form_data['gender'] ?? '') == 'Male') echo 'selected'; ?>>Male</option>
                                <option value="Female" <?php if (($form_data['gender'] ?? '') == 'Female') echo 'selected'; ?>>Female</option>
                                <option value="Other" <?php if (($form_data['gender'] ?? '') == 'Other') echo 'selected'; ?>>Other</option>
                            </select>
                        </div>
                    </div>

                    <div class="form-row">
                        <div class="form-group col-md-12">
                            <label for="father_name">Father's Name *</label>
                            <input type="text" class="form-control" id="father_name" name="father_name" value="<?php echo htmlspecialchars($form_data['father_name'] ?? ''); ?>" required>
                        </div>
                    </div>
                    
                    <hr>
                    
                    <h5>Contact Information</h5>
                    <div class="form-row">
                        <div class="form-group col-md-4">
                            <label for="cnic_bform">CNIC / B-Form</label>
                            <input type="text" class="form-control" id="cnic_bform" name="cnic_bform" value="<?php echo htmlspecialchars($form_data['cnic_bform'] ?? ''); ?>" placeholder="e.g., 3840393218793">
                        </div>
                        <div class="form-group col-md-4">
                            <label for="phone">Phone/Mobile</label>
                            <input type="text" class="form-control" id="phone" name="phone" value="<?php echo htmlspecialchars($form_data['phone'] ?? ''); ?>" placeholder="e.g., 03XX-XXXXXXX">
                        </div>
                        <div class="form-group col-md-4">
                            <label for="email">Email</label>
                            <input type="email" class="form-control" id="email" name="email" value="<?php echo htmlspecialchars($form_data['email'] ?? ''); ?>" placeholder="Optional">
                        </div>
                    </div>
                    <div class="form-group">
                        <label for="address">Address</label>
                        <textarea class="form-control" id="address" name="address" rows="2" placeholder="Full residential address"><?php echo htmlspecialchars($form_data['address'] ?? ''); ?></textarea>
                    </div>

                    <button type="submit" class="btn btn-success btn-lg mt-4">
                        <i class="fas fa-save"></i> Save Student Record
                    </button>
                    <a href="students.php" class="btn btn-secondary btn-lg mt-4">
                        <i class="fas fa-list"></i> Go to Student List
                    </a>
                </form>
            </div>
        </div>
    <?php endif; ?>
</div>

<?php 
ob_end_flush(); 
include_once('includes/footer.php'); 
?>