<?php
// admin/students_process.php (Complete Student Management Logic)
session_start();

// Authentication Check
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

// Database Connection
include_once('../config/db.php');

$action = $_POST['action'] ?? $_GET['action'] ?? '';
$conn = getDBConnection();

// --- Helper Function ---
function redirect($page, $msg = '', $type = 'success') {
    $param_name = ($type === 'success') ? 'msg' : 'err';
    if ($page === 'student_edit' && isset($_POST['student_id'])) {
        $url = "$page.php?id=" . (int)$_POST['student_id'] . "&$param_name=" . urlencode($msg);
    } else {
        $url = "$page.php?$param_name=" . urlencode($msg);
    }
    header("Location: $url");
    exit;
}

// ====================================================================
// FEE ASSIGNMENT ACTION - NEW
// ====================================================================
if ($_POST['action'] == 'assign_fees') {
    $student_id = intval($_POST['student_id']);
    $admission_fee = floatval($_POST['admission_fee']);
    $monthly_fee = floatval($_POST['monthly_fee']);
    $examination_fee = floatval($_POST['examination_fee']);
    $examination_type = $_POST['examination_type'];
    $months = $_POST['months'] ?? [];
    $fee_notes = $_POST['fee_notes'];
    
    if ($student_id <= 0) {
        $_SESSION['error'] = "Invalid student ID";
        header("Location: students.php?err=Invalid+student+ID");
        exit();
    }
    
    try {
        $conn->begin_transaction();
        
        // Get current academic year
        $current_year = date('Y');
        
        // Admission Fee (One-time) - sirf pehle month ke liye
        if ($admission_fee > 0) {
            // Check if admission fee already assigned
            $check_admission = $conn->prepare("SELECT id FROM student_fees WHERE student_id = ? AND fee_type = 'admission' AND academic_year = ?");
            $check_admission->bind_param("ii", $student_id, $current_year);
            $check_admission->execute();
            $admission_exists = $check_admission->get_result()->num_rows > 0;
            
            if (!$admission_exists) {
                $admission_query = "INSERT INTO student_fees 
                    (student_id, fee_type, amount, month, academic_year, is_one_time, notes, created_at) 
                    VALUES (?, 'admission', ?, 'August', ?, 1, ?, NOW())";
                $stmt = $conn->prepare($admission_query);
                // CORRECTED: 'idiss' - integer, double, integer, string, string
                $stmt->bind_param("idiss", $student_id, $admission_fee, $current_year, $fee_notes);
                $stmt->execute();
                $stmt->close();
            }
        }
        
        // Monthly Fee - har selected month ke liye
        if ($monthly_fee > 0 && !empty($months)) {
            $monthly_query = "INSERT INTO student_fees 
                (student_id, fee_type, amount, month, academic_year, is_recurring, notes, created_at) 
                VALUES (?, 'monthly', ?, ?, ?, 1, ?, NOW())";
            $stmt = $conn->prepare($monthly_query);
            
            foreach ($months as $month) {
                // Check if monthly fee already exists for this month
                $check_monthly = $conn->prepare("SELECT id FROM student_fees WHERE student_id = ? AND fee_type = 'monthly' AND month = ? AND academic_year = ?");
                $check_monthly->bind_param("isi", $student_id, $month, $current_year);
                $check_monthly->execute();
                $monthly_exists = $check_monthly->get_result()->num_rows > 0;
                
                if (!$monthly_exists) {
                    // CORRECTED: 'idiss' - integer, double, string, integer, string
                    $stmt->bind_param("idiss", $student_id, $monthly_fee, $month, $current_year, $fee_notes);
                    $stmt->execute();
                }
            }
            $stmt->close();
        }
        
        // Examination Fee - selected months ke liye
        if ($examination_fee > 0 && !empty($months)) {
            $exam_query = "INSERT INTO student_fees 
                (student_id, fee_type, amount, month, academic_year, examination_type, notes, created_at) 
                VALUES (?, 'examination', ?, ?, ?, ?, ?, NOW())";
            $stmt = $conn->prepare($exam_query);
            
            foreach ($months as $month) {
                // Check if exam fee already exists for this month
                $check_exam = $conn->prepare("SELECT id FROM student_fees WHERE student_id = ? AND fee_type = 'examination' AND month = ? AND academic_year = ? AND examination_type = ?");
                $check_exam->bind_param("isis", $student_id, $month, $current_year, $examination_type);
                $check_exam->execute();
                $exam_exists = $check_exam->get_result()->num_rows > 0;
                
                if (!$exam_exists) {
                    // CORRECTED: 'idisss' - integer, double, string, integer, string, string
                    $stmt->bind_param("idisss", $student_id, $examination_fee, $month, $current_year, $examination_type, $fee_notes);
                    $stmt->execute();
                }
            }
            $stmt->close();
        }
        
        $conn->commit();
        
        $_SESSION['message'] = "Fees successfully assigned to student!";
        header("Location: students.php?msg=Fees+assigned+successfully");
        exit();
        
    } catch (Exception $e) {
        $conn->rollback();
        $_SESSION['error'] = "Error assigning fees: " . $e->getMessage();
        header("Location: students.php?err=Error+assigning+fees");
        exit();
    }
}

// ====================================================================
// PROMOTE, DEMOTE, RETAIN STUDENT ACTIONS - FIXED
// ====================================================================

// Promote Student
if (isset($_GET['action']) && $_GET['action'] == 'promote_student' && isset($_GET['id'])) {
    $student_id = intval($_GET['id']);
    promoteStudent($student_id);
    header("Location: students.php?msg=Student promoted successfully");
    exit();
}

// Demote Student  
if (isset($_GET['action']) && $_GET['action'] == 'demote_student' && isset($_GET['id'])) {
    $student_id = intval($_GET['id']);
    demoteStudent($student_id);
    header("Location: students.php?msg=Student demoted successfully");
    exit();
}

// Retain Student
if (isset($_GET['action']) && $_GET['action'] == 'retain_student' && isset($_GET['id'])) {
    $student_id = intval($_GET['id']);
    retainStudent($student_id);
    header("Location: students.php?msg=Student retained successfully");
    exit();
}

// Functions for promotion/demotion/retention
function promoteStudent($student_id) {
    global $conn;
    
    // Session ID get karein settings table se
    $settings_query = "SELECT setting_value FROM settings WHERE setting_key = 'academic.active_session_id'";
    $settings_result = $conn->query($settings_query);
    
    if ($settings_result && $settings_result->num_rows > 0) {
        $current_session = $settings_result->fetch_assoc();
        $current_session_id = $current_session['setting_value'];
    } else {
        $current_session_id = 1;
    }
    
    // Get current class information
    $query = "SELECT en.class_id, ac.name as class_name FROM student_enrollments en 
              JOIN academic_classes ac ON en.class_id = ac.id 
              WHERE en.student_id = ? AND en.session_id = ?";
    $stmt = $conn->prepare($query);
    $stmt->bind_param("ii", $student_id, $current_session_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $current_class = $result->fetch_assoc();
    
    if ($current_class) {
        $current_class_name = $current_class['class_name'];
        
        // Define class sequence
        $class_sequence = [
            'Play Group' => 'Nursery',
            'Nursery' => 'Prep', 
            'Prep' => 'One',
            'One' => 'Two',
            'Two' => 'Three',
            'Three' => 'Four',
            'Four' => 'Five',
            'Five' => 'Six',
            'Six' => 'Seven',
            'Seven' => 'Eight',
            'Eight' => 'Nine',
            'Nine' => 'Ten',
            'Ten' => 'Ten'
        ];
        
        // Get next class
        $next_class_name = $class_sequence[$current_class_name] ?? $current_class_name;
        
        if ($next_class_name != $current_class_name) {
            // Get next class ID
            $class_query = "SELECT id FROM academic_classes WHERE name = ?";
            $class_stmt = $conn->prepare($class_query);
            $class_stmt->bind_param("s", $next_class_name);
            $class_stmt->execute();
            $class_result = $class_stmt->get_result();
            $next_class = $class_result->fetch_assoc();
            
            if ($next_class) {
                // Update enrollment with next class
                $update_query = "UPDATE student_enrollments SET class_id = ? WHERE student_id = ? AND session_id = ?";
                $update_stmt = $conn->prepare($update_query);
                $update_stmt->bind_param("iii", $next_class['id'], $student_id, $current_session_id);
                $update_stmt->execute();
            }
        }
    }
}

function demoteStudent($student_id) {
    global $conn;
    
    // Session ID get karein settings table se
    $settings_query = "SELECT setting_value FROM settings WHERE setting_key = 'academic.active_session_id'";
    $settings_result = $conn->query($settings_query);
    
    if ($settings_result && $settings_result->num_rows > 0) {
        $current_session = $settings_result->fetch_assoc();
        $current_session_id = $current_session['setting_value'];
    } else {
        $current_session_id = 1;
    }
    
    // Get current class information
    $query = "SELECT en.class_id, ac.name as class_name FROM student_enrollments en 
              JOIN academic_classes ac ON en.class_id = ac.id 
              WHERE en.student_id = ? AND en.session_id = ?";
    $stmt = $conn->prepare($query);
    $stmt->bind_param("ii", $student_id, $current_session_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $current_class = $result->fetch_assoc();
    
    if ($current_class) {
        $current_class_name = $current_class['class_name'];
        
        // Define class sequence (reverse for demotion)
        $class_sequence = [
            'Nursery' => 'Play Group',
            'Prep' => 'Nursery',
            'One' => 'Prep',
            'Two' => 'One',
            'Three' => 'Two', 
            'Four' => 'Three',
            'Five' => 'Four',
            'Six' => 'Five',
            'Seven' => 'Six',
            'Eight' => 'Seven',
            'Nine' => 'Eight',
            'Ten' => 'Nine',
            'Play Group' => 'Play Group'
        ];
        
        // Get previous class
        $prev_class_name = $class_sequence[$current_class_name] ?? $current_class_name;
        
        if ($prev_class_name != $current_class_name) {
            // Get previous class ID
            $class_query = "SELECT id FROM academic_classes WHERE name = ?";
            $class_stmt = $conn->prepare($class_query);
            $class_stmt->bind_param("s", $prev_class_name);
            $class_stmt->execute();
            $class_result = $class_stmt->get_result();
            $prev_class = $class_result->fetch_assoc();
            
            if ($prev_class) {
                // Update enrollment with previous class
                $update_query = "UPDATE student_enrollments SET class_id = ? WHERE student_id = ? AND session_id = ?";
                $update_stmt = $conn->prepare($update_query);
                $update_stmt->bind_param("iii", $prev_class['id'], $student_id, $current_session_id);
                $update_stmt->execute();
            }
        }
    }
}

function retainStudent($student_id) {
    // Retention ke liye koi change nahi karna
    return true;
}

// ====================================================================
// ACTION 1: ADD NEW STUDENT
// ====================================================================
if ($action === 'add_student' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    
    // Student Details
    $first_name = trim($_POST['first_name'] ?? '');
    $last_name = trim($_POST['last_name'] ?? '');
    $father_name = trim($_POST['father_name'] ?? '');
    $dob = $_POST['dob'] ?? NULL;
    $phone = trim($_POST['phone'] ?? '');
    $admission_date = $_POST['admission_date'] ?? date('Y-m-d');
    $status = $_POST['status'] ?? 'active';
    $address = trim($_POST['address'] ?? '');
    
    // Enrollment Details
    $enrollment_session_id = (int)$_POST['enrollment_session_id'];
    $enrollment_class_id = (int)$_POST['enrollment_class_id'];
    $roll_no = trim($_POST['roll_no'] ?? NULL);
    
    if (empty($first_name) || $enrollment_session_id == 0 || $enrollment_class_id == 0) {
        redirect('student_add', 'Please fill all required fields.', 'danger');
    }

    $conn->begin_transaction();
    
    try {
        // A. Insert into Students table
        $stmt_s = $conn->prepare("INSERT INTO students (first_name, last_name, father_name, dob, phone, admission_date, status, address) 
                                 VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
        $stmt_s->bind_param("ssssssss", $first_name, $last_name, $father_name, $dob, $phone, $admission_date, $status, $address);
        if (!$stmt_s->execute()) { throw new Exception("Error inserting student: " . $stmt_s->error); }
        $student_id = $stmt_s->insert_id;
        $stmt_s->close();
        
        // B. Insert into Student_Enrollments table
        $stmt_e = $conn->prepare("INSERT INTO student_enrollments (student_id, session_id, class_id, roll_no) VALUES (?, ?, ?, ?)");
        $stmt_e->bind_param("iiis", $student_id, $enrollment_session_id, $enrollment_class_id, $roll_no);
        if (!$stmt_e->execute()) { throw new Exception("Error inserting enrollment: " . $stmt_e->error); }
        $stmt_e->close();
        
        $conn->commit();
        redirect('students', 'Student "' . $first_name . '" successfully added and enrolled.', 'success');
        
    } catch (Exception $e) {
        $conn->rollback();
        redirect('student_add', 'Student addition failed: ' . $e->getMessage(), 'danger'); 
    }
}

// ====================================================================
// ACTION 2: UPDATE STUDENT INFORMATION
// ====================================================================
elseif ($action === 'update_student' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    
    $student_id = (int)$_POST['student_id'];
    // Student Details
    $first_name = trim($_POST['first_name'] ?? '');
    $last_name = trim($_POST['last_name'] ?? '');
    $father_name = trim($_POST['father_name'] ?? '');
    $dob = $_POST['dob'] ?? NULL;
    $phone = trim($_POST['phone'] ?? '');
    $admission_date = $_POST['admission_date'] ?? date('Y-m-d');
    $status = $_POST['status'] ?? 'active';
    $address = trim($_POST['address'] ?? '');
    // Enrollment Details
    $enrollment_session_id = (int)$_POST['enrollment_session_id'];
    $enrollment_class_id = (int)$_POST['enrollment_class_id'];
    $roll_no = trim($_POST['roll_no'] ?? NULL);

    if ($student_id <= 0 || empty($first_name) || $enrollment_session_id == 0 || $enrollment_class_id == 0) {
        redirect('student_edit', 'Invalid or missing required data.', 'danger');
    }

    $conn->begin_transaction();
    
    try {
        // A. Update Basic Student Record
        $stmt_s = $conn->prepare("UPDATE students SET first_name=?, last_name=?, father_name=?, dob=?, phone=?, admission_date=?, status=?, address=? WHERE id=?");
        $stmt_s->bind_param("ssssssssi", $first_name, $last_name, $father_name, $dob, $phone, $admission_date, $status, $address, $student_id);
        if (!$stmt_s->execute()) { throw new Exception("Error updating student details: " . $stmt_s->error); }
        $stmt_s->close();
        
        // B. Update/Insert Enrollment Record
        $check_enrollment = $conn->query("SELECT id FROM student_enrollments WHERE student_id = $student_id AND session_id = $enrollment_session_id");
        
        if ($check_enrollment->num_rows > 0) {
            // Update existing enrollment
            $stmt_e = $conn->prepare("UPDATE student_enrollments SET class_id=?, roll_no=? WHERE student_id=? AND session_id=?");
            $stmt_e->bind_param("isii", $enrollment_class_id, $roll_no, $student_id, $enrollment_session_id);
        } else {
            // Delete old enrollments and insert new
            $conn->query("DELETE FROM student_enrollments WHERE student_id = {$student_id}"); 
            
            // Insert new enrollment record
            $stmt_e = $conn->prepare("INSERT INTO student_enrollments (student_id, session_id, class_id, roll_no) VALUES (?, ?, ?, ?)");
            $stmt_e->bind_param("iiis", $student_id, $enrollment_session_id, $enrollment_class_id, $roll_no);
        }

        if (!$stmt_e->execute()) { throw new Exception("Error updating/inserting enrollment: " . $stmt_e->error); }
        $stmt_e->close();
        
        $conn->commit();
        redirect('student_profile', 'Student record successfully updated.', 'success'); 
        
    } catch (Exception $e) {
        $conn->rollback();
        redirect('student_edit', 'Update failed: ' . $e->getMessage(), 'danger'); 
    }
}

// ====================================================================
// ACTION 3: DELETE STUDENT
// ====================================================================
elseif ($action === 'delete_student' && $_SERVER['REQUEST_METHOD'] === 'GET') {
    
    $student_id = (int)($_GET['id'] ?? 0);
    
    if ($student_id <= 0) {
        redirect('students', 'Invalid Student ID.', 'danger');
    }

    $conn->begin_transaction();
    
    try {
        // Delete related data first
        $conn->query("DELETE FROM fee_payments WHERE invoice_id IN (SELECT id FROM fee_invoices WHERE student_id = {$student_id})");
        $conn->query("DELETE FROM fee_invoice_items WHERE invoice_id IN (SELECT id FROM fee_invoices WHERE student_id = {$student_id})");
        $conn->query("DELETE FROM fee_invoices WHERE student_id = {$student_id}");
        $conn->query("DELETE FROM student_fees WHERE student_id = {$student_id}"); // NEW: Delete assigned fees
        $conn->query("DELETE FROM exam_marks WHERE student_id = {$student_id}"); 
        $conn->query("DELETE FROM student_results WHERE student_id = {$student_id}");
        $conn->query("DELETE FROM daily_attendance_records WHERE student_id = {$student_id}");
        $conn->query("DELETE FROM student_enrollments WHERE student_id = {$student_id}");

        // Delete main student record
        $stmt = $conn->prepare("DELETE FROM students WHERE id = ?");
        $stmt->bind_param("i", $student_id);
        
        if (!$stmt->execute()) { throw new Exception("Error deleting student: " . $stmt->error); }
        $stmt->close();
        
        $conn->commit();
        redirect('students', 'Student aur us se mutalliq tamaam records safaltamami se delete ho chuke hain.', 'success');

    } catch (Exception $e) {
        $conn->rollback();
        redirect('students', 'Student delete karne mein ghalti hui: ' . $e->getMessage(), 'danger');
    }
}

// ====================================================================
// Default redirect
// ====================================================================
else {
    redirect('students', 'Invalid action.', 'danger');
}

$conn->close();
?>