<?php
// admin/subject_process.php - Handles Subject and Course Allocation CRUD operations (Batch Allocation logic)

session_start();
// Security Check
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}
include_once('../config/db.php');
$conn = getDBConnection();

$action = $_POST['action'] ?? $_GET['action'] ?? '';

// Function to handle redirection
function redirect($page, $msg = '', $type = 'success', $tab = 'subjects') {
    $param_name = ($type === 'success') ? 'msg' : 'err';
    header("Location: $page?tab=$tab&$param_name=" . urlencode($msg));
    exit;
}

// ... (ACTION 1: ADD NEW SUBJECT - NO CHANGE)
if ($action === 'add_subject' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $name = trim($_POST['name'] ?? '');
    $code = !empty(trim($_POST['code'] ?? '')) ? trim($_POST['code']) : NULL; 
    $type = $_POST['type'] ?? 'core';

    if (empty($name)) {
        redirect('subjects.php', 'Subject Name cannot be empty.', 'danger', 'subjects');
    }
    
    try {
        $stmt = $conn->prepare("INSERT INTO subjects (name, code, type) VALUES (?, ?, ?)");
        $stmt->bind_param("sss", $name, $code, $type); 
        
        if ($stmt->execute()) {
            redirect('subjects.php', 'Subject "' . $name . '" successfully added.', 'success', 'subjects');
        } else {
            if ($conn->errno == 1062) {
                redirect('subjects.php', 'Error adding subject: Subject Code/Name already exists.', 'danger', 'subjects');
            } else {
                redirect('subjects.php', 'Database Error: Could not add subject. ' . $conn->error, 'danger', 'subjects');
            }
        }
        $stmt->close();
    } catch (Exception $e) {
        redirect('subjects.php', 'An error occurred: ' . $e->getMessage(), 'danger', 'subjects');
    }
}


// ... (ACTION 2: UPDATE EXISTING SUBJECT - NO CHANGE)
elseif ($action === 'update_subject' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $id = (int)($_POST['id'] ?? 0);
    $name = trim($_POST['name'] ?? '');
    $code = !empty(trim($_POST['code'] ?? '')) ? trim($_POST['code']) : NULL; 
    $type = $_POST['type'] ?? 'core';

    if ($id <= 0 || empty($name)) {
        $_SESSION['edit_err'] = 'Invalid or incomplete subject data for update. (Subject Name is required).';
        header("Location: subject_edit.php?id=" . $id);
        exit;
    }

    try {
        $stmt = $conn->prepare("UPDATE subjects SET name = ?, code = ?, type = ? WHERE id = ?");
        $stmt->bind_param("sssi", $name, $code, $type, $id); 
        
        if ($stmt->execute()) {
            $message = "Subject '{$name}' updated successfully.";
            redirect('subjects.php', $message, 'success', 'subjects'); 
        } else {
            $error = "Update failed.";
            if ($conn->errno == 1062) {
                $error = "Subject Code already exists. Please use a unique code.";
            } else {
                $error = "Database Error: Could not update subject. " . $conn->error;
            }
            $_SESSION['edit_err'] = $error;
            header("Location: subject_edit.php?id=" . $id);
            exit;
        }
        $stmt->close();
    } catch (Exception $e) {
        $_SESSION['edit_err'] = 'An error occurred during update: ' . $e->getMessage();
        header("Location: subject_edit.php?id=" . $id);
        exit;
    }
}


// ... (ACTION 3: DELETE SUBJECT - NO CHANGE)
elseif ($action === 'delete_subject' && $_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['id'])) {
    $id = (int)($_GET['id'] ?? 0);

    if ($id > 0) {
        try {
            $conn->begin_transaction();
            // Delete related records (Allocations, Marks)
            $conn->query("DELETE FROM course_allocation WHERE subject_id = $id");
            $conn->query("DELETE FROM exam_marks WHERE subject_id = $id");
            
            $stmt_sub = $conn->prepare("DELETE FROM subjects WHERE id = ?");
            $stmt_sub->bind_param("i", $id);

            if ($stmt_sub->execute()) {
                $conn->commit();
                redirect('subjects.php', "Subject successfully deleted, along with its related data.", 'success', 'subjects');
            } else {
                $conn->rollback();
                redirect('subjects.php', "Subject delete nahi ho saka: " . $conn->error, 'danger', 'subjects');
            }
            $stmt_sub->close();

        } catch (Exception $e) {
            $conn->rollback();
            redirect('subjects.php', 'Deletion error: ' . $e->getMessage(), 'danger', 'subjects');
        }
    } else {
        redirect('subjects.php', "Invalid Subject ID.", 'danger', 'subjects');
    }
}


// ====================================================================
// ACTION 4: ALLOCATE MULTIPLE SUBJECTS (NEW BATCH LOGIC)
// ====================================================================
elseif ($action === 'allocate_subjects_batch' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $session_id = (int)$_POST['session_id'];
    $class_id = (int)$_POST['class_id'];
    $section_id = !empty($_POST['section_id']) ? (int)$_POST['section_id'] : NULL;  
    $teacher_id = (int)$_POST['teacher_id'];
    $selected_subject_ids = $_POST['subject_ids'] ?? []; 
    
    if (!$session_id || !$class_id || !$teacher_id || empty($selected_subject_ids)) {
        redirect('subjects.php', 'Class, Teacher, aur kam se kam ek Subject select karna zaroori hai.', 'danger', 'allocation');
    }

    try {
        $conn->begin_transaction();
        $allocated_count = 0;
        $duplicate_count = 0;
        
        foreach ($selected_subject_ids as $subject_id) {
            $subject_id = (int)$subject_id;
            
            // 1. Duplicate check (same Class/Section/Subject combination)
            $check_query = "SELECT id FROM course_allocation WHERE session_id = ? AND class_id = ? AND subject_id = ?";
            
            if ($section_id !== NULL) {
                $check_query .= " AND section_id = ?";
                $stmt_check = $conn->prepare($check_query);
                $stmt_check->bind_param("iiii", $session_id, $class_id, $subject_id, $section_id);
            } else {
                $check_query .= " AND section_id IS NULL";
                $stmt_check = $conn->prepare($check_query);
                $stmt_check->bind_param("iii", $session_id, $class_id, $subject_id);
            }
            
            $stmt_check->execute();
            $stmt_check->store_result();
            
            if ($stmt_check->num_rows > 0) {
                // Skip if duplicate allocation exists
                $duplicate_count++;
                $stmt_check->close();
                continue; 
            }
            $stmt_check->close();

            // 2. Insert the new allocation
            if ($section_id !== NULL) {
                $sql = "INSERT INTO course_allocation (session_id, class_id, section_id, subject_id, teacher_id) VALUES (?, ?, ?, ?, ?)";
                $stmt_insert = $conn->prepare($sql);
                $stmt_insert->bind_param("iiiii", $session_id, $class_id, $section_id, $subject_id, $teacher_id);
            } else {
                $sql = "INSERT INTO course_allocation (session_id, class_id, section_id, subject_id, teacher_id) VALUES (?, ?, NULL, ?, ?)";
                $stmt_insert = $conn->prepare($sql);
                $stmt_insert->bind_param("iiii", $session_id, $class_id, $subject_id, $teacher_id);
            }
            
            if ($stmt_insert->execute()) {
                $allocated_count++;
            } else {
                // Agar Foreign Key ya koi aur error aaya to transaction rollback kar denge
                $conn->rollback();
                $error_msg = 'Allocation failed for a subject. Database Error: ' . $stmt_insert->error;
                if ($conn->errno == 1452) { 
                     $error_msg = 'Allocation failed: The selected Subject or Teacher ID does not exist in the database.';
                }
                redirect('subjects.php', $error_msg, 'danger', 'allocation');
            }
            $stmt_insert->close();
        }

        $conn->commit();
        $success_message = "{$allocated_count} subjects successfully allocated.";
        if ($duplicate_count > 0) {
            $success_message .= " ({$duplicate_count} duplicates were skipped.)";
        }
        redirect('subjects.php', $success_message, 'success', 'allocation');

    } catch (Exception $e) {
        $conn->rollback();
        redirect('subjects.php', 'Allocation error: ' . $e->getMessage(), 'danger', 'allocation');
    }
}


// ... (ACTION 5: DELETE ALLOCATION - NO CHANGE)
elseif ($action === 'delete_allocation' && $_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['id'])) {
    $allocation_id = (int)$_GET['id'];
    
    $stmt = $conn->prepare("DELETE FROM course_allocation WHERE id = ?");
    $stmt->bind_param("i", $allocation_id);
    
    if ($stmt->execute()) {
        redirect('subjects.php', 'Allocation successfully removed.', 'success', 'allocation');
    } else {
        redirect('subjects.php', 'Error removing allocation: ' . $conn->error, 'danger', 'allocation');
    }
    $stmt->close();
}


// ... (DEFAULT - NO CHANGE)
else {
    redirect('subjects.php', 'Invalid action: The requested operation could not be identified.', 'danger');
}

$conn->close();
?>