<?php
// FILE: admin/syllabus.php

session_start();
ob_start();
include_once('../config/db.php');

$conn = getDBConnection();

$message = '';
$error = '';

// Get active academic year from academic_sessions table
$active_academic_year = '';
$active_session_name = '';
$settings_query = "SELECT id, name, start_date, end_date FROM academic_sessions WHERE is_active = 1 LIMIT 1";
$settings_result = $conn->query($settings_query);
if ($settings_result && $settings_result->num_rows > 0) {
    $session_data = $settings_result->fetch_assoc();
    $active_academic_year = $session_data['name'];
    $active_session_name = $session_data['name'];
} else {
    // Fallback: if no active session found, use current year
    $current_year = date('Y');
    $next_year = $current_year + 1;
    $active_academic_year = $current_year . '-' . $next_year;
    $active_session_name = $current_year . '-' . $next_year;
}

// Handle file upload
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $class_id = $_POST['class_id'] ?? '';
    $syllabus_type = $_POST['syllabus_type'] ?? '';
    $academic_year = $_POST['academic_year'] ?? '';
    
    // Validation
    if (empty($class_id) || empty($syllabus_type) || empty($academic_year)) {
        $error = "All fields are required!";
    } elseif (empty($_FILES['syllabus_file']['name'])) {
        $error = "Please select a PDF file to upload!";
    } else {
        // Check if file is PDF
        $file_type = $_FILES['syllabus_file']['type'];
        $file_size = $_FILES['syllabus_file']['size'];
        $file_name = $_FILES['syllabus_file']['name'];
        
        if ($file_type != 'application/pdf') {
            $error = "Only PDF files are allowed!";
        } elseif ($file_size > 1048576) { // 1MB in bytes
            $error = "File size must be less than 1MB!";
        } else {
            // Check if syllabus already exists for this class and type
            $check_stmt = $conn->prepare("SELECT id FROM class_syllabus WHERE class_id = ? AND syllabus_type = ? AND academic_year = ?");
            $check_stmt->bind_param("iss", $class_id, $syllabus_type, $academic_year);
            $check_stmt->execute();
            $check_result = $check_stmt->get_result();
            
            if ($check_result->num_rows > 0) {
                $error = "Syllabus already exists for this class and type! Please delete the existing one first.";
            } else {
                // Generate unique filename
                $file_extension = pathinfo($file_name, PATHINFO_EXTENSION);
                $new_filename = "syllabus_" . $class_id . "_" . $syllabus_type . "_" . time() . "." . $file_extension;
                $upload_path = "../uploads/syllabus/" . $new_filename;
                
                // Create uploads directory if it doesn't exist
                if (!is_dir('../uploads/syllabus')) {
                    mkdir('../uploads/syllabus', 0777, true);
                }
                
                if (move_uploaded_file($_FILES['syllabus_file']['tmp_name'], $upload_path)) {
                    // Insert into database
                    $insert_stmt = $conn->prepare("INSERT INTO class_syllabus (class_id, syllabus_type, academic_year, file_name, original_name, file_path, file_size, uploaded_at) VALUES (?, ?, ?, ?, ?, ?, ?, NOW())");
                    $insert_stmt->bind_param("isssssi", $class_id, $syllabus_type, $academic_year, $new_filename, $file_name, $upload_path, $file_size);
                    
                    if ($insert_stmt->execute()) {
                        $message = "Syllabus uploaded successfully!";
                    } else {
                        $error = "Failed to save syllabus details in database!";
                        // Delete uploaded file if database insert fails
                        unlink($upload_path);
                    }
                    $insert_stmt->close();
                } else {
                    $error = "Failed to upload file!";
                }
            }
            $check_stmt->close();
        }
    }
}

// Handle delete action
if (isset($_GET['delete_id'])) {
    $delete_id = intval($_GET['delete_id']);
    
    // Get file path before deleting
    $select_stmt = $conn->prepare("SELECT file_path FROM class_syllabus WHERE id = ?");
    $select_stmt->bind_param("i", $delete_id);
    $select_stmt->execute();
    $select_result = $select_stmt->get_result();
    
    if ($select_result->num_rows > 0) {
        $syllabus = $select_result->fetch_assoc();
        $file_path = $syllabus['file_path'];
        
        // Delete from database
        $delete_stmt = $conn->prepare("DELETE FROM class_syllabus WHERE id = ?");
        $delete_stmt->bind_param("i", $delete_id);
        
        if ($delete_stmt->execute()) {
            // Delete physical file
            if (file_exists($file_path)) {
                unlink($file_path);
            }
            $message = "Syllabus deleted successfully!";
        } else {
            $error = "Failed to delete syllabus from database!";
        }
        $delete_stmt->close();
    } else {
        $error = "Syllabus not found!";
    }
    $select_stmt->close();
}

// Fetch classes for dropdown
$classes = [];
$classes_query = "SELECT id, name FROM academic_classes ORDER BY 
                  CASE 
                    WHEN name LIKE '%Play Group%' THEN 1
                    WHEN name LIKE '%Nursery%' THEN 2  
                    WHEN name LIKE '%Prep%' THEN 3
                    WHEN name LIKE '%One%' THEN 4
                    WHEN name LIKE '%Two%' THEN 5
                    WHEN name LIKE '%Three%' THEN 6
                    WHEN name LIKE '%Four%' THEN 7
                    WHEN name LIKE '%Five%' THEN 8
                    WHEN name LIKE '%Six%' THEN 9
                    WHEN name LIKE '%Seven%' THEN 10
                    WHEN name LIKE '%Eight%' THEN 11
                    WHEN name LIKE '%Nine%' THEN 12
                    WHEN name LIKE '%Ten%' THEN 13
                    ELSE 14
                  END ASC, order_no ASC";
$classes_result = $conn->query($classes_query);
if ($classes_result) {
    $classes = $classes_result->fetch_all(MYSQLI_ASSOC);
}

// Fetch existing syllabus
$syllabus_list = [];
$syllabus_query = "SELECT cs.*, ac.name as class_name 
                   FROM class_syllabus cs 
                   JOIN academic_classes ac ON cs.class_id = ac.id 
                   ORDER BY cs.uploaded_at DESC";
$syllabus_result = $conn->query($syllabus_query);
if ($syllabus_result) {
    $syllabus_list = $syllabus_result->fetch_all(MYSQLI_ASSOC);
}

// Includes for Page Layout
include('includes/header.php'); 
include_once('includes/sidebar.php');
?>

<div class="content-wrapper">
    <div class="content-header">
        <div class="container-fluid">
            <div class="row mb-2">
                <div class="col-sm-6">
                    <h1 class="m-0">📚 Syllabus Management</h1>
                </div>
                <div class="col-sm-6">
                    <ol class="breadcrumb float-sm-right">
                        <li class="breadcrumb-item"><a href="index.php">Home</a></li>
                        <li class="breadcrumb-item active">Syllabus</li>
                    </ol>
                </div>
            </div>
        </div>
    </div>
    
    <section class="content">
        <div class="container-fluid">
            <!-- Alert Messages -->
            <?php if ($message): ?>
                <div class="alert alert-success alert-dismissible fade show" role="alert">
                    <i class="fas fa-check-circle me-2"></i><?php echo htmlspecialchars($message); ?>
                    <button type="button" class="close" data-dismiss="alert" aria-label="Close">
                        <span aria-hidden="true">&times;</span>
                    </button>
                </div>
            <?php endif; ?>
            
            <?php if ($error): ?>
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <i class="fas fa-exclamation-triangle me-2"></i><?php echo htmlspecialchars($error); ?>
                    <button type="button" class="close" data-dismiss="alert" aria-label="Close">
                        <span aria-hidden="true">&times;</span>
                    </button>
                </div>
            <?php endif; ?>

            <!-- Upload Syllabus Card -->
            <div class="card card-primary">
                <div class="card-header">
                    <h3 class="card-title">
                        <i class="fas fa-upload me-2"></i>Upload New Syllabus
                    </h3>
                    <div class="card-tools">
                        <span class="badge badge-success">
                            <i class="fas fa-calendar me-1"></i>
                            Active Session: <?php echo htmlspecialchars($active_session_name); ?>
                        </span>
                    </div>
                </div>
                <div class="card-body">
                    <form method="POST" action="syllabus.php" enctype="multipart/form-data" id="syllabusForm">
                        <div class="row">
                            <div class="form-group col-md-4">
                                <label for="class_id">Select Class <span class="text-danger">*</span></label>
                                <select name="class_id" id="class_id" class="form-control" required>
                                    <option value="">-- Select Class --</option>
                                    <?php foreach ($classes as $class): ?>
                                        <option value="<?php echo $class['id']; ?>">
                                            <?php echo htmlspecialchars($class['name']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            
                            <div class="form-group col-md-4">
                                <label for="syllabus_type">Syllabus Type <span class="text-danger">*</span></label>
                                <select name="syllabus_type" id="syllabus_type" class="form-control" required>
                                    <option value="">-- Select Type --</option>
                                    <option value="term_wise">Term Wise Syllabus</option>
                                    <option value="summer_vacation">Summer Vacation Syllabus</option>
                                </select>
                            </div>
                            
                            <div class="form-group col-md-4">
                                <label for="academic_year">Academic Session <span class="text-danger">*</span></label>
                                <input type="text" class="form-control bg-light" 
                                       value="<?php echo htmlspecialchars($active_academic_year); ?>" 
                                       readonly disabled
                                       placeholder="Auto-detected from active session">
                                <small class="form-text text-muted">
                                    <i class="fas fa-info-circle me-1"></i>
                                    Automatically set to active academic session
                                </small>
                                <!-- Hidden field to actually submit the value -->
                                <input type="hidden" name="academic_year" value="<?php echo htmlspecialchars($active_academic_year); ?>">
                            </div>
                        </div>

                        <div class="form-group">
                            <label for="syllabus_file">Syllabus File (PDF only, max 1MB) <span class="text-danger">*</span></label>
                            <div class="custom-file">
                                <input type="file" class="custom-file-input" id="syllabus_file" name="syllabus_file" accept=".pdf" required>
                                <label class="custom-file-label" for="syllabus_file">Choose PDF file...</label>
                            </div>
                            <small class="form-text text-muted">
                                <i class="fas fa-exclamation-triangle me-1"></i>
                                Only PDF files are allowed. Maximum file size: 1MB
                            </small>
                            <div id="filePreview" class="mt-2" style="display: none;">
                                <div class="alert alert-info p-2">
                                    <i class="fas fa-file-pdf me-2"></i>
                                    <span id="fileName"></span>
                                    <span id="fileSize" class="badge badge-light ml-2"></span>
                                </div>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <button type="submit" class="btn btn-success">
                                <i class="fas fa-upload me-2"></i>Upload Syllabus
                            </button>
                            <button type="reset" class="btn btn-secondary">
                                <i class="fas fa-redo me-2"></i>Reset Form
                            </button>
                        </div>
                    </form>
                </div>
            </div>

            <!-- Existing Syllabus List -->
            <div class="card card-info mt-4">
                <div class="card-header">
                    <h3 class="card-title">
                        <i class="fas fa-list me-2"></i>Existing Syllabus Files
                    </h3>
                    <div class="card-tools">
                        <span class="badge badge-light">Total: <?php echo count($syllabus_list); ?> files</span>
                    </div>
                </div>
                <div class="card-body">
                    <?php if (empty($syllabus_list)): ?>
                        <div class="alert alert-info text-center">
                            <i class="fas fa-info-circle fa-2x mb-3"></i>
                            <h5>No Syllabus Files Found</h5>
                            <p class="mb-0">Upload your first syllabus using the form above.</p>
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-bordered table-striped table-hover">
                                <thead class="thead-dark">
                                    <tr>
                                        <th>#</th>
                                        <th>Class</th>
                                        <th>Syllabus Type</th>
                                        <th>Academic Session</th>
                                        <th>File Name</th>
                                        <th>File Size</th>
                                        <th>Uploaded At</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($syllabus_list as $index => $syllabus): ?>
                                    <tr>
                                        <td><?php echo $index + 1; ?></td>
                                        <td>
                                            <span class="badge badge-primary"><?php echo htmlspecialchars($syllabus['class_name']); ?></span>
                                        </td>
                                        <td>
                                            <?php if ($syllabus['syllabus_type'] == 'term_wise'): ?>
                                                <span class="badge badge-success">Term Wise</span>
                                            <?php else: ?>
                                                <span class="badge badge-warning">Summer Vacation</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($syllabus['academic_year'] == $active_academic_year): ?>
                                                <span class="badge badge-success">
                                                    <?php echo htmlspecialchars($syllabus['academic_year']); ?>
                                                    <i class="fas fa-star ml-1"></i>
                                                </span>
                                            <?php else: ?>
                                                <span class="badge badge-secondary"><?php echo htmlspecialchars($syllabus['academic_year']); ?></span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <i class="fas fa-file-pdf text-danger me-2"></i>
                                            <?php echo htmlspecialchars($syllabus['original_name']); ?>
                                        </td>
                                        <td>
                                            <span class="badge badge-light">
                                                <?php echo round($syllabus['file_size'] / 1024, 2); ?> KB
                                            </span>
                                        </td>
                                        <td><?php echo date('d-M-Y h:i A', strtotime($syllabus['uploaded_at'])); ?></td>
                                        <td>
                                            <div class="btn-group btn-group-sm">
                                                <a href="<?php echo $syllabus['file_path']; ?>" 
                                                   class="btn btn-info" 
                                                   target="_blank"
                                                   title="View PDF">
                                                    <i class="fas fa-eye"></i> View
                                                </a>
                                                <a href="<?php echo $syllabus['file_path']; ?>" 
                                                   class="btn btn-secondary" 
                                                   download
                                                   title="Download PDF">
                                                    <i class="fas fa-download"></i> Download
                                                </a>
                                                <a href="syllabus.php?delete_id=<?php echo $syllabus['id']; ?>" 
                                                   class="btn btn-danger" 
                                                   onclick="return confirmDelete()"
                                                   title="Delete Syllabus">
                                                    <i class="fas fa-trash"></i> Delete
                                                </a>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
            
        </div>
    </section>
</div>

<script>
// File input preview
document.getElementById('syllabus_file').addEventListener('change', function(e) {
    const file = e.target.files[0];
    const filePreview = document.getElementById('filePreview');
    const fileName = document.getElementById('fileName');
    const fileSize = document.getElementById('fileSize');
    const fileLabel = document.querySelector('.custom-file-label');
    
    if (file) {
        // Update file label
        fileLabel.textContent = file.name;
        
        // Show preview
        fileName.textContent = file.name;
        fileSize.textContent = (file.size / 1024).toFixed(2) + ' KB';
        filePreview.style.display = 'block';
        
        // Validate file type and size
        if (file.type !== 'application/pdf') {
            alert('Error: Only PDF files are allowed!');
            this.value = '';
            filePreview.style.display = 'none';
            fileLabel.textContent = 'Choose PDF file...';
        } else if (file.size > 1048576) {
            alert('Error: File size must be less than 1MB!');
            this.value = '';
            filePreview.style.display = 'none';
            fileLabel.textContent = 'Choose PDF file...';
        }
    } else {
        filePreview.style.display = 'none';
        fileLabel.textContent = 'Choose PDF file...';
    }
});

// Form validation
document.getElementById('syllabusForm').addEventListener('submit', function(e) {
    const classId = document.getElementById('class_id').value;
    const syllabusType = document.getElementById('syllabus_type').value;
    const syllabusFile = document.getElementById('syllabus_file').files[0];
    
    let errors = [];
    
    if (!classId) errors.push('Please select a class');
    if (!syllabusType) errors.push('Please select syllabus type');
    if (!syllabusFile) errors.push('Please select a PDF file');
    
    if (errors.length > 0) {
        e.preventDefault();
        alert('Please fix the following errors:\n\n• ' + errors.join('\n• '));
        return false;
    }
});

// Delete confirmation
function confirmDelete() {
    return confirm('Are you sure you want to delete this syllabus?\n\nThis action cannot be undone and the file will be permanently deleted.');
}

// Auto-hide alerts after 5 seconds
setTimeout(function() {
    $('.alert').fadeOut('slow');
}, 5000);

// Reset form
document.querySelector('button[type="reset"]').addEventListener('click', function() {
    document.getElementById('filePreview').style.display = 'none';
    document.querySelector('.custom-file-label').textContent = 'Choose PDF file...';
});
</script>

<style>
.card-header {
    background: linear-gradient(45deg, #f8f9fa, #e9ecef);
    border-bottom: 1px solid #dee2e6;
}
.table th {
    background-color: #495057;
    color: white;
    font-weight: 600;
}
.btn-group-sm > .btn {
    padding: 0.25rem 0.5rem;
    font-size: 0.875rem;
    margin: 0 2px;
}
.badge {
    font-size: 0.85em;
    font-weight: 500;
}
.alert {
    border-radius: 0.5rem;
    border: none;
    box-shadow: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
}
.custom-file-label::after {
    content: "Browse";
}
.bg-light {
    background-color: #f8f9fa !important;
    cursor: not-allowed;
}
</style>

<?php 
// Connection Closure 
if (isset($conn) && $conn instanceof mysqli) {
    //@$conn->close();
}

include_once('includes/footer.php'); 
ob_end_flush();
?>