<?php
// admin/user_management.php - Manage System Users (NON-PARENT USERS)

// 1. Start Session
session_start();

// 2. Database Connection
include_once('../config/db.php'); 
$conn = getDBConnection(); 
$message = $_GET['msg'] ?? ''; 
$error = $_GET['err'] ?? ''; 

// 3. Authorization Check
$user_role_id = $_SESSION['role_id'] ?? 0;
if (!isset($user_role_id) || $user_role_id != 1) { 
    header('Location: dashboard.php?err=' . urlencode('Aapke paas is page ka access nahi hai.'));
    exit;
}

// --------------------------------------------------------------------------------
// --- GET Request Handling (Edit/Delete Actions) ---
// --------------------------------------------------------------------------------
if (isset($_GET['action']) && isset($_GET['id'])) {
    $user_id = (int)$_GET['id'];
    $current_logged_in_user_id = $_SESSION['user_id'] ?? 0;
    
    // Security Check
    if ($user_id == $current_logged_in_user_id && $user_id != 0) {
        $error = "Security Error: Aap khud ko delete ya status change nahi kar sakte.";
        header("Location: user_management.php?err=" . urlencode($error));
        exit;
    }

    try {
        if ($_GET['action'] == 'delete') {
            // Check if the user is a Parent first (if yes, redirect to Parent Management)
            $role_check_q = $conn->query("SELECT ur.name FROM users u JOIN user_roles ur ON u.role_id = ur.id WHERE u.id = {$user_id}");
            if ($role_check_q && strpos(strtolower($role_check_q->fetch_assoc()['name'] ?? ''), 'parent') !== false) {
                 $error = "Parent users ko 'Parent Management' tab se delete karein.";
                 header("Location: user_management.php?err=" . urlencode($error));
                 exit;
            }
            
            // Transaction start for simple user deletion
            $conn->begin_transaction();
            
            // 1. Delete user (Note: Parent deletion logic is in parent_management.php)
            $stmt = $conn->prepare("DELETE FROM users WHERE id = ? AND role_id != 1"); 
            $stmt->bind_param("i", $user_id);
            
            if ($stmt->execute() && $stmt->affected_rows > 0) {
                $conn->commit();
                $message = "User ID {$user_id} successfully deleted!";
            } else if ($stmt->affected_rows == 0) {
                 $conn->rollback();
                 $error = "Deletion failed. User not found, ya woh Admin role ka user hai.";
            } else {
                $conn->rollback();
                $error = "Error deleting user: " . $conn->error;
            }
            $stmt->close();
            
        } else if ($_GET['action'] == 'toggle_status') {
            $current_status_q = $conn->query("SELECT status FROM users WHERE id = {$user_id}");
            $current_status = $current_status_q ? ($current_status_q->fetch_assoc()['status'] ?? 'inactive') : 'inactive';
            $new_status = ($current_status == 'active') ? 'inactive' : 'active';
            
            $stmt = $conn->prepare("UPDATE users SET status = ? WHERE id = ?");
            $stmt->bind_param("si", $new_status, $user_id);
            
            if ($stmt->execute()) {
                $message = "User ID {$user_id} ka status successfully " . ucfirst($new_status) . " kar diya gaya.";
            } else {
                $error = "Error updating status: " . $conn->error;
            }
            $stmt->close();
        }
        
    } catch (Exception $e) {
        if (isset($conn) && $conn->in_transaction) {
            $conn->rollback();
        }
        $error = "Action Error: " . $e->getMessage();
    }
    
    header("Location: user_management.php?msg=" . urlencode($message) . "&err=" . urlencode($error));
    exit;
}

// --------------------------------------------------------------------------------
// --- POST Request Handling (Add New User) ---
// --------------------------------------------------------------------------------
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $action = $_POST['action'] ?? '';
    
    try {
        if ($action == 'add_user') {
            $username = trim($_POST['username'] ?? '');
            $password = $_POST['password'] ?? '';
            $user_name = trim($_POST['user_name'] ?? '');
            $email = trim($_POST['email'] ?? NULL);
            $role_id = (int)($_POST['role_id'] ?? 0);
            
            // Dynamic Role Name Fetch
            $role_name_q = $conn->prepare("SELECT name FROM user_roles WHERE id = ?");
            $role_name_q->bind_param("i", $role_id);
            $role_name_q->execute();
            $result_role_name = $role_name_q->get_result();
            $role_data = $result_role_name->fetch_assoc();
            $role_name = strtolower($role_data['name'] ?? '');
            $role_name_q->close();
            
            // Check if the role is 'Parent' - agar haan, toh error do
            if (strpos($role_name, 'parent') !== false) { 
                $error = "Parent user ko 'Parent Management' tab se add karein.";
            } else if (empty($username) || empty($password) || empty($role_id) || empty($user_name)) {
                $error = "Please fill in Username, Password, Name, and Role.";
            } else {
                $password_hash = password_hash($password, PASSWORD_DEFAULT);

                // 1. Insert into users table
                $stmt = $conn->prepare("INSERT INTO users (username, password_hash, role_id, name, email, status) VALUES (?, ?, ?, ?, ?, 'active')");
                $stmt->bind_param("ssiss", $username, $password_hash, $role_id, $user_name, $email);
                
                if ($stmt->execute()) {
                    $message = "User '{$username}' successfully added!";
                } else {
                    $error = "Error adding user: " . $conn->error;
                    if ($conn->errno == 1062) {
                          $error = "Error: Username ya Email already exists.";
                    }
                }
                $stmt->close();
            }
        }
    } catch (Exception $e) {
        $error = "Error: " . $e->getMessage();
    }
    
    header("Location: user_management.php?msg=" . urlencode($message) . "&err=" . urlencode($error));
    exit;
}

// --------------------------------------------------------------------------------
// --- Fetch Data ---
// --------------------------------------------------------------------------------

// 1. Fetch all User Roles (excluding 'Parent' role for this form dropdown)
$roles = [];
$parent_role_id = 0;
$roles_result = $conn->query("SELECT id, name FROM user_roles ORDER BY name");
if ($roles_result) {
    while ($row = $roles_result->fetch_assoc()) {
        // Parent role ID store karo
        if (strpos(strtolower($row['name']), 'parent') !== false) {
            $parent_role_id = (int)$row['id'];
            continue; // Skip the Parent role for the dropdown list in this file
        }
        $roles[] = $row;
    }
}

// 2. Fetch all Users with their Role Names 
$users_data = [];
$query = "SELECT 
            u.id, 
            u.username, 
            u.name, 
            u.email, 
            u.status, 
            ur.name AS role_name,
            ur.id AS role_id
          FROM users u
          JOIN user_roles ur ON u.role_id = ur.id
          ORDER BY ur.name, u.username";
$result = $conn->query($query);
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $users_data[] = $row;
    }
}

// --------------------------------------------------------------------------------
// --- HTML Output Starts Here ---
// --------------------------------------------------------------------------------
include('includes/header.php'); 
?>

<h1 class="mb-4">User Management <i class="fas fa-users-cog"></i></h1>

<?php if (!empty($message)): ?>
    <div class="alert alert-success"><?php echo htmlspecialchars($message); ?></div>
<?php endif; ?>
<?php if (!empty($error)): ?>
    <div class="alert alert-danger"><?php echo htmlspecialchars($error); ?></div>
<?php endif; ?>

<ul class="nav nav-tabs mb-3">
    <li class="nav-item">
        <a class="nav-link active" href="user_management.php"><i class="fas fa-user-shield"></i> Core User Management</a>
    </li>
    <li class="nav-item">
        <a class="nav-link" href="parent_management.php"><i class="fas fa-user-tie"></i> Parent Management</a>
    </li>
</ul>

<div class="card mb-4 shadow-sm">
    <div class="card-header bg-primary text-white">
        <h5><i class="fas fa-user-plus"></i> Add New Core User (Admin, Teacher, Staff, Student)</h5>
    </div>
    <div class="card-body">
        <form action="user_management.php" method="POST">
            <input type="hidden" name="action" value="add_user">
            
            <div class="form-row">
                <div class="form-group col-md-6">
                    <label for="username">Username (Login ID)</label>
                    <input type="text" class="form-control" id="username" name="username" required>
                </div>
                <div class="form-group col-md-6">
                    <label for="password">Password</label>
                    <input type="password" class="form-control" id="password" name="password" required>
                </div>
            </div>
            
            <div class="form-row">
                <div class="form-group col-md-6">
                    <label for="user_name">Full Name</label>
                    <input type="text" class="form-control" id="user_name" name="user_name" required>
                </div>
                <div class="form-group col-md-6">
                    <label for="role_id">User Role</label>
                    <select id="role_id" name="role_id" class="form-control" required>
                        <option value="">Select Role</option>
                        <?php foreach ($roles as $role): ?>
                            <option value="<?php echo $role['id']; ?>">
                                <?php echo htmlspecialchars($role['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
            </div>

             <div class="form-group">
                <label for="email">Email (Optional)</label>
                <input type="email" class="form-control" id="email" name="email">
            </div>
            
            <button type="submit" class="btn btn-primary mt-3"><i class="fas fa-save"></i> Create User</button>
        </form>
    </div>
</div>

---
<div class="card shadow-sm">
    <div class="card-header">
        <h5>All System Users</h5>
    </div>
    <div class="card-body">
        <?php if (empty($users_data)): ?>
            <div class="alert alert-info">No users found. Please add the first user above.</div>
        <?php else: ?>
            <div class="table-responsive">
                <table class="table table-striped table-bordered table-hover">
                    <thead class="thead-dark">
                        <tr>
                            <th>ID</th>
                            <th>Username</th>
                            <th>Full Name</th>
                            <th>Email</th>
                            <th>Role</th>
                            <th>Status</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($users_data as $user): ?>
                            <tr>
                                <td><?php echo $user['id']; ?></td>
                                <td><?php echo htmlspecialchars($user['username']); ?></td>
                                <td><?php echo htmlspecialchars($user['name']); ?></td>
                                <td><?php echo htmlspecialchars($user['email'] ?? 'N/A'); ?></td>
                                <td>
                                    <span class="badge badge-<?php echo ($user['role_id'] == $parent_role_id) ? 'warning' : 'info'; ?>">
                                        <?php echo htmlspecialchars($user['role_name']); ?>
                                    </span>
                                </td>
                                <td>
                                    <?php 
                                        $status_class = ($user['status'] == 'active') ? 'badge-success' : 'badge-danger';
                                        echo "<span class='badge {$status_class}'>" . htmlspecialchars(ucfirst($user['status'])). "</span>";
                                    ?>
                                </td>
                                <td>
                                    <?php if ($user['role_id'] == $parent_role_id): ?>
                                    <a href="parent_management.php?action=edit&id=<?php echo $user['id']; ?>" 
                                       class="btn btn-sm btn-info" 
                                       title="Manage Parent Details and Student Links">
                                        <i class="fas fa-user-tie"></i> Manage Parent
                                    </a>
                                    <?php else: ?>
                                    <a href="user_management.php?action=toggle_status&id=<?php echo $user['id']; ?>" 
                                       class="btn btn-sm btn-warning" 
                                       title="Edit User: Status Toggle (Active/Inactive)">
                                        <i class="fas fa-edit"></i> Status
                                    </a>
                                    <?php endif; ?>

                                    <?php if ($user['id'] != 1): ?>
                                        <?php if ($user['role_id'] == $parent_role_id): ?>
                                            <a href="parent_management.php?action=delete&id=<?php echo $user['id']; ?>" 
                                                class="btn btn-sm btn-danger" 
                                                title="Delete Parent and Links"
                                                onclick="return confirm('Are you sure you want to permanently delete parent <?php echo htmlspecialchars($user['username']); ?>? All student links will be removed.');">
                                                <i class="fas fa-trash"></i> Delete
                                            </a>
                                        <?php else: ?>
                                            <a href="user_management.php?action=delete&id=<?php echo $user['id']; ?>" 
                                                class="btn btn-sm btn-danger" 
                                                title="Delete User"
                                                onclick="return confirm('Are you sure you want to permanently delete user <?php echo htmlspecialchars($user['username']); ?>?');">
                                                <i class="fas fa-trash"></i> Delete
                                            </a>
                                        <?php endif; ?>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>
</div>

<?php
// Note: Is file mein koi custom JavaScript nahi hai
include('includes/footer.php');
?>