<?php
// install/process.php - Single-Step Installation (Final Robust Version with Hashing Fix)

// Set headers for JSON response
header('Content-Type: application/json');

// --- Main Execution Block (Runs only on POST request) ---
if ($_SERVER["REQUEST_METHOD"] === "POST") {
    
    // Sanitize and define all config variables from POST data
    $servername = trim($_POST['servername'] ?? 'localhost');
    $username = trim($_POST['username'] ?? '');
    $password = $_POST['password'] ?? '';
    $dbname = trim($_POST['dbname'] ?? 'sms_database');
    $admin_username = trim($_POST['admin_username'] ?? 'admin');
    $admin_password_plain = $_POST['admin_password'] ?? '123456';
    $institute_name = trim($_POST['institute_name'] ?? 'Default School Name');
    $institute_address = trim($_POST['institute_address'] ?? '123 Main Street');
    
    // Simple validation: Check required fields
    if (empty($username) || empty($dbname) || empty($admin_username) || empty($admin_password_plain)) {
        echo json_encode(['status' => 'error', 'message' => 'Please fill out all required fields.', 'next_step' => '1']);
        exit;
    }
    
    try {
        // 1. Database Connection and Creation
        // Connect without selecting DB to test credentials and create DB
        $conn = new mysqli($servername, $username, $password);

        if ($conn->connect_error) {
            throw new Exception("Connection failed: " . $conn->connect_error);
        }

        // Create database if it doesn't exist
        $dbname_safe = $conn->real_escape_string($dbname);
        $db_check_query = "SELECT SCHEMA_NAME FROM INFORMATION_SCHEMA.SCHEMATA WHERE SCHEMA_NAME = '{$dbname_safe}'";
        $db_exists_result = $conn->query($db_check_query);
        
        if ($db_exists_result->num_rows == 0) {
            if (!$conn->query("CREATE DATABASE IF NOT EXISTS `{$dbname_safe}`")) {
                throw new Exception("Error creating database: Check user permissions. " . $conn->error);
            }
        }
        
        $conn->close();

        // 2. Save configuration file (config/db.php)
        $config_content = "<?php\n";
        $config_content .= "// config/db.php - Generated during installation\n";
        $config_content .= "function getDBConnection() {\n";
        $config_content .= "    \$servername = \"{$servername}\";\n";
        $config_content .= "    \$username = \"{$username}\";\n";
        $config_content .= "    \$password = \"{$password}\";\n";
        $config_content .= "    \$dbname = \"{$dbname}\";\n";
        $config_content .= "    \$conn = new mysqli(\$servername, \$username, \$password, \$dbname);\n";
        $config_content .= "    if (\$conn->connect_error) {\n";
        $config_content .= "        die(\"Connection failed: \" . \$conn->connect_error);\n";
        $config_content .= "    }\n";
        $config_content .= "    \$conn->set_charset(\"utf8mb4\");\n";
        $config_content .= "    return \$conn;\n";
        $config_content .= "}\n";
        $config_content .= "?>";

        // Ensure config folder exists
        if (!is_dir('../config')) {
            mkdir('../config', 0777, true);
        }
        if (file_put_contents('../config/db.php', $config_content) === false) {
             throw new Exception("Error writing to config/db.php. Check folder permissions.");
        }
        
        // --- THIRD PHASE: Execute Schema and Setup ---
        
        // 3. Load DB connection and execute schema
        // config file ko dobara include karna zaroori hai naye database se connect karne ke liye
        include_once('../config/db.php'); 
        $conn = getDBConnection();
        $schema_file = 'final_schema.sql'; 
        $sql = file_get_contents($schema_file);
        
        if (empty($sql)) throw new Exception("SQL Schema file (final_schema.sql) is empty or not found. Ensure it is in the 'install' folder.");

        // Execute schema queries
        if (!$conn->multi_query($sql)) throw new Exception("Schema execution failed: " . $conn->error);
        // Wait for all queries to finish
        do { if ($result = $conn->store_result()) $result->free(); } while ($conn->more_results() && $conn->next_result());
        
        // 4. Insert Admin User (using the credentials provided by the user)
        $password_hash = password_hash($admin_password_plain, PASSWORD_DEFAULT);
        
        // Agar hashing fail ho jaye (rare case, but possible), to error throw karein.
        if ($password_hash === false || empty($password_hash)) {
            throw new Exception("Password hashing failed. Cannot proceed with Admin creation.");
        }
        
        $role_id = 1; // Admin role ID
        $admin_name = 'System Admin';

        $stmt = $conn->prepare("INSERT INTO users (username, password_hash, role_id, name, status) VALUES (?, ?, ?, ?, 'active') ON DUPLICATE KEY UPDATE password_hash = VALUES(password_hash), name = VALUES(name)");
        
        // FIX APPLIED HERE: Changed "siss" to "ssis". password_hash is always a string ('s')
        $stmt->bind_param("ssis", $admin_username, $password_hash, $role_id, $admin_name); 
        
        if (!$stmt->execute()) throw new Exception("Admin user creation failed: " . $stmt->error);
        $stmt->close();
        
        // 5. Insert Institute Settings
        $settings_data = [
            'institute.name' => $institute_name,
            'institute.address' => $institute_address,
            'academic.active_session_id' => '1',
        ];

        foreach ($settings_data as $key => $value) {
            $group = ($key === 'academic.active_session_id') ? 'academic' : 'institute';
            $stmt = $conn->prepare("INSERT INTO settings (setting_key, setting_value, group_name) VALUES (?, ?, ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)");
            $stmt->bind_param("sss", $key, $value, $group);
            $stmt->execute();
            $stmt->close();
        }

        $conn->close();
        
        // Final success response
        echo json_encode([
            'status' => 'complete', 
            'message' => "Installation successful! Login with username: '{$admin_username}' and the password you provided.",
            'admin_url' => '../admin/login.php'
        ]);

    } catch (Exception $e) {
        if (isset($conn)) $conn->close();
        echo json_encode(['status' => 'error', 'message' => "Installation Failed: " . $e->getMessage(), 'next_step' => '1']);
    }
} else {
    // If accessed directly without POST data (jo pehle 'Invalid step' ka error de raha tha)
    echo json_encode(['status' => 'error', 'message' => 'Invalid installation attempt. Please submit the form from index.php.', 'next_step' => '1']);
}
?>