<?php
// FILE: parent/class_planner.php - Class Planner/Homework View for Parent
session_start();

if (!isset($_SESSION['user_id']) || $_SESSION['role_id'] != 2) {
    header('Location: ../login.php');
    exit;
}

include_once('../config/db.php');
include_once('includes/header.php');
$conn = getDBConnection();

$parent_user_id = (int)$_SESSION['user_id'];
$student_id = (int)($_GET['student_id'] ?? 0);
$error = '';
$planner_data = [];

// --- Date Filter ---
$date_filter = $_GET['date'] ?? date('Y-m-d');

// --- 1. Security Check ---
$is_linked_q = $conn->prepare("SELECT COUNT(*) FROM parent_student_link WHERE parent_user_id = ? AND student_id = ?");
$is_linked_q->bind_param("ii", $parent_user_id, $student_id);
$is_linked_q->execute();
$is_linked_q->bind_result($is_linked);
$is_linked_q->fetch();
$is_linked_q->close();

if ($student_id == 0 || $is_linked == 0) {
    $error = "Error: Invalid Student ID or the student is not linked to your account.";
    goto display_planner;
}

// --- 2. Fetch Student Details & Enrollment ---
$details_query = "
    SELECT 
        s.first_name, s.last_name, 
        en.class_id, en.section_id,
        ac.name AS class_name
    FROM students s
    LEFT JOIN student_enrollments en ON s.id = en.student_id AND en.is_active = 1
    LEFT JOIN academic_classes ac ON en.class_id = ac.id
    WHERE s.id = ?
";
$student_details = [];
if ($stmt_det = $conn->prepare($details_query)) {
    $stmt_det->bind_param("i", $student_id);
    $stmt_det->execute();
    $student_details = $stmt_det->get_result()->fetch_assoc();
    $stmt_det->close();
}

if (!$student_details || empty($student_details['class_id'])) {
    $error = "Error: Student details or active enrollment not found.";
    goto display_planner;
}

$class_id = (int)$student_details['class_id'];
$section_id = (int)$student_details['section_id'];
$student_full_name = htmlspecialchars($student_details['first_name'] . ' ' . $student_details['last_name']);
$student_class = htmlspecialchars($student_details['class_name'] ?? 'N/A');


// --- 3. Fetch Class Planner Data ---
$planner_query = "
    SELECT 
        dp.subject_id, 
        s.name AS subject_name, 
        dp.cw_topic, 
        dp.hw_assignment,
        t.name AS teacher_name
    FROM daily_planner dp
    JOIN academic_subjects s ON dp.subject_id = s.id
    LEFT JOIN teachers t ON dp.teacher_id = t.id -- Assuming teacher_id is in daily_planner
    WHERE dp.class_id = ? 
    AND dp.section_id = ? 
    AND dp.planner_date = ?
    ORDER BY s.name ASC
";

if ($stmt = $conn->prepare($planner_query)) {
    $stmt->bind_param("iis", $class_id, $section_id, $date_filter);
    $stmt->execute();
    $planner_data = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
} else {
    $error = "Error fetching planner data: " . $conn->error;
}


display_planner:
$conn->close();
?>

<div class="container-fluid">
    <h1 class="mt-4">Daily Class Planner / Homework <i class="fas fa-book-open"></i></h1>

    <?php if (!empty($error)): ?>
        <div class="alert alert-danger"><?php echo $error; ?></div>
    <?php elseif ($student_id != 0 && $is_linked == 1): ?>
        <p class="lead">Viewing Planner for: **<?php echo $student_full_name; ?>** (Class: **<?php echo $student_class; ?>**)</p>
        
        <form method="GET" class="form-inline mb-4">
            <input type="hidden" name="student_id" value="<?php echo $student_id; ?>">
            <label class="my-1 mr-2" for="date">Select Date:</label>
            <input type="date" class="form-control my-1 mr-sm-2" id="date" name="date" value="<?php echo htmlspecialchars($date_filter); ?>" max="<?php echo date('Y-m-d'); ?>" required>
            <button type="submit" class="btn btn-primary my-1"><i class="fas fa-search"></i> View Planner</button>
        </form>

        <div class="card shadow-sm">
            <div class="card-header bg-success text-white">
                <i class="fas fa-calendar-day"></i> Planner Details for **<?php echo date('d-M-Y', strtotime($date_filter)); ?>**
            </div>
            <div class="card-body">
                <?php if (empty($planner_data)): ?>
                    <div class="alert alert-info text-center">Is date ke liye koi **Class Planner** entry available nahi hai.</div>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-bordered table-striped">
                            <thead class="thead-light">
                                <tr>
                                    <th>Subject</th>
                                    <th>Class Work / Topic Covered</th>
                                    <th>Homework / Assignment</th>
                                    <th>Teacher</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($planner_data as $item): ?>
                                    <tr>
                                        <td>**<?php echo htmlspecialchars($item['subject_name']); ?>**</td>
                                        <td><?php echo nl2br(htmlspecialchars($item['cw_topic'])); ?></td>
                                        <td class="text-danger font-weight-bold"><?php echo nl2br(htmlspecialchars($item['hw_assignment'])); ?></td>
                                        <td><?php echo htmlspecialchars($item['teacher_name'] ?? 'N/A'); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>

    <?php else: ?>
         <div class="alert alert-info mt-4">Kripya <a href="dashboard.php" class="alert-link">Dashboard</a> par wapas jaakar bachcha select karein.</div>
    <?php endif; ?>
</div>

<?php include_once('includes/footer.php'); ?>