<?php
// FILE: parent/dashboard.php - Parent's Main Dashboard
session_start();

// Authorization Check: Ensure the user is logged in AND is a Parent (role_id = 5)
if (!isset($_SESSION['user_id']) || $_SESSION['role_id'] != 5) { 
    header('Location: login.php');
    exit;
}

include_once('../config/db.php');
include_once('includes/header.php');
$conn = getDBConnection();

$parent_user_id = (int)$_SESSION['user_id'];
$parent_name = htmlspecialchars($_SESSION['name'] ?? 'Parent');
$message = '';
$error = '';

// --- Fetch Assigned Students ---
$students_data = [];
$active_session_id = 0;

// Step 1: Active Session ID nikalna
$settings_query = $conn->query("SELECT setting_value FROM settings WHERE setting_key='academic.active_session_id'");
if ($settings_query && $row = $settings_query->fetch_assoc()) {
    $active_session_id = (int)$row['setting_value'];
}

// Step 2: Parent se linked students ki details fetch karna
$query = "
    SELECT 
        s.id, s.first_name, s.last_name, s.father_name, s.gender, s.status,
        en.roll_no, 
        ac.id as class_id, ac.name AS class_name,
        asec.name AS section_name
    FROM students s
    JOIN parent_student_link psl ON s.id = psl.student_id
    LEFT JOIN student_enrollments en ON s.id = en.student_id AND en.session_id = ?
    LEFT JOIN academic_classes ac ON en.class_id = ac.id
    LEFT JOIN academic_sections asec ON en.section_id = asec.id
    WHERE psl.parent_user_id = ?
    ORDER BY ac.order_no ASC, en.roll_no ASC
";

if ($stmt = $conn->prepare($query)) {
    $stmt->bind_param("ii", $active_session_id, $parent_user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $students_data = $result->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
} else {
    $error = "Error fetching student data: " . $conn->error;
}

// --- Fetch Active Notices ---
$notices_data = [];
$notices_query = $conn->query("
    SELECT id, title, description, notice_date, created_at 
    FROM notices 
    WHERE is_active = 1 
    AND notice_date <= CURDATE() 
    ORDER BY notice_date DESC, created_at DESC 
    LIMIT 5
");

if ($notices_query) {
    $notices_data = $notices_query->fetch_all(MYSQLI_ASSOC);
}

// --- Fetch Syllabus Data for Each Student ---
$syllabus_data = [];
foreach ($students_data as $student) {
    if (!empty($student['class_id'])) {
        $class_id = (int)$student['class_id'];
        $syllabus_query = $conn->prepare("
            SELECT id, class_id, syllabus_type, academic_year, file_name, original_name, file_path, file_size, uploaded_at 
            FROM class_syllabus 
            WHERE class_id = ? 
            ORDER BY uploaded_at DESC
        ");
        if ($syllabus_query) {
            $syllabus_query->bind_param("i", $class_id);
            $syllabus_query->execute();
            $syllabus_result = $syllabus_query->get_result();
            $syllabus_data[$student['id']] = $syllabus_result->fetch_all(MYSQLI_ASSOC);
            $syllabus_query->close();
        }
    }
}

// Debug: Check if syllabus data is being fetched
error_log("Syllabus Data: " . print_r($syllabus_data, true));

$conn->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Parent Dashboard</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .student-card {
            border-radius: 15px;
            border: none;
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
            transition: all 0.3s ease;
            background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
            overflow: hidden;
        }

        .student-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 25px rgba(0,0,0,0.15);
        }

        .student-card-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 15px 20px;
            border-bottom: none;
        }

        .student-avatar {
            width: 60px;
            height: 60px;
            border-radius: 50%;
            background: linear-gradient(45deg, #ff6b6b, #feca57);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
            color: white;
            margin-right: 15px;
            border: 3px solid white;
        }

        .student-info {
            flex: 1;
        }

        .student-name {
            font-size: 1.3rem;
            font-weight: 600;
            margin-bottom: 5px;
        }

        .student-details {
            font-size: 0.9rem;
            opacity: 0.9;
        }

        .action-btn {
            border-radius: 25px;
            padding: 8px 15px;
            font-size: 0.85rem;
            font-weight: 500;
            transition: all 0.3s ease;
            border: 2px solid;
            margin: 3px;
            min-width: 110px;
            text-align: center;
            display: inline-block;
            text-decoration: none;
        }

        .action-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.2);
            text-decoration: none;
        }

        .btn-attendance {
            background: linear-gradient(45deg, #17a2b8, #20c997);
            border-color: #17a2b8;
            color: white;
        }

        .btn-fees {
            background: linear-gradient(45deg, #ffc107, #fd7e14);
            border-color: #ffc107;
            color: white;
        }

        .btn-result {
            background: linear-gradient(45deg, #dc3545, #e83e8c);
            border-color: #dc3545;
            color: white;
        }

        .btn-planner {
            background: linear-gradient(45deg, #28a745, #20c997);
            border-color: #28a745;
            color: white;
        }

        .btn-syllabus {
            background: linear-gradient(45deg, #6f42c1, #e83e8c);
            border-color: #6f42c1;
            color: white;
            border: none;
        }

        .status-badge {
            border-radius: 20px;
            padding: 4px 12px;
            font-size: 0.75rem;
            font-weight: 600;
        }

        .welcome-section {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-radius: 15px;
            color: white;
            padding: 30px;
            margin-bottom: 30px;
            box-shadow: 0 8px 25px rgba(0,0,0,0.15);
        }

        .welcome-title {
            font-size: 2.2rem;
            font-weight: 700;
            margin-bottom: 10px;
        }

        .welcome-subtitle {
            font-size: 1.1rem;
            opacity: 0.9;
            margin-bottom: 0;
        }

        /* Syllabus Modal Styles */
        .syllabus-modal .modal-content {
            border-radius: 15px;
            border: none;
            box-shadow: 0 10px 30px rgba(0,0,0,0.2);
        }

        .syllabus-modal .modal-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-bottom: none;
            border-radius: 15px 15px 0 0;
            padding: 20px 25px;
        }

        .syllabus-modal .modal-title {
            font-weight: 600;
            font-size: 1.3rem;
        }

        .syllabus-modal .modal-body {
            padding: 25px;
            max-height: 60vh;
            overflow-y: auto;
        }

        .syllabus-item {
            border: 1px solid #e9ecef;
            border-radius: 10px;
            padding: 15px;
            margin-bottom: 15px;
            transition: all 0.3s ease;
            background: #f8f9fa;
        }

        .syllabus-item:hover {
            background: white;
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
            transform: translateY(-2px);
        }

        .syllabus-title {
            font-weight: 600;
            color: #2c3e50;
            margin-bottom: 8px;
            font-size: 1.1rem;
        }

        .syllabus-meta {
            font-size: 0.8rem;
            color: #95a5a6;
            margin-bottom: 10px;
        }

        .syllabus-type {
            display: inline-block;
            background: #e9ecef;
            padding: 3px 8px;
            border-radius: 12px;
            font-size: 0.75rem;
            font-weight: 600;
            margin-right: 8px;
        }

        .syllabus-type.term_wise {
            background: #d4edda;
            color: #155724;
        }

        .syllabus-type.summer_vacation {
            background: #fff3cd;
            color: #856404;
        }

        .syllabus-year {
            display: inline-block;
            background: #d1ecf1;
            color: #0c5460;
            padding: 3px 8px;
            border-radius: 12px;
            font-size: 0.75rem;
            font-weight: 600;
        }

        .syllabus-file-info {
            font-size: 0.8rem;
            color: #6c757d;
            margin-bottom: 10px;
        }

        .syllabus-actions {
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
        }

        .btn-view, .btn-download {
            border-radius: 20px;
            padding: 6px 15px;
            font-size: 0.8rem;
            font-weight: 500;
            border: none;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-block;
        }

        .btn-view {
            background: #17a2b8;
            color: white;
        }

        .btn-view:hover {
            background: #138496;
            color: white;
            transform: translateY(-1px);
            text-decoration: none;
        }

        .btn-download {
            background: #28a745;
            color: white;
        }

        .btn-download:hover {
            background: #218838;
            color: white;
            transform: translateY(-1px);
            text-decoration: none;
        }

        .no-syllabus {
            text-align: center;
            padding: 40px 20px;
            color: #6c757d;
        }

        .no-syllabus i {
            font-size: 3rem;
            margin-bottom: 15px;
            opacity: 0.5;
        }

        /* Mobile Responsive */
        @media (max-width: 768px) {
            .student-card-header {
                padding: 12px 15px;
            }
            
            .student-avatar {
                width: 50px;
                height: 50px;
                font-size: 20px;
                margin-right: 12px;
            }
            
            .student-name {
                font-size: 1.1rem;
            }
            
            .action-btn {
                min-width: 100%;
                margin: 5px 0;
                padding: 10px;
            }
            
            .welcome-section {
                padding: 20px;
                margin-bottom: 20px;
            }
            
            .welcome-title {
                font-size: 1.8rem;
            }
        }

        @media (max-width: 576px) {
            .student-avatar {
                width: 45px;
                height: 45px;
                font-size: 18px;
            }
            
            .student-name {
                font-size: 1rem;
            }
            
            .action-btn {
                font-size: 0.8rem;
                padding: 8px 12px;
            }
            
            .syllabus-modal .modal-body {
                padding: 15px;
            }
            
            .syllabus-actions {
                flex-direction: column;
            }
            
            .btn-view, .btn-download {
                width: 100%;
                text-align: center;
            }
        }
    </style>
</head>
<body>
    <?php include_once('includes/header.php'); ?>

    <div class="container-fluid">
        <!-- Welcome Section -->
        <div class="welcome-section">
            <div class="row align-items-center">
                <div class="col-md-8">
                    <h1 class="welcome-title">Welcome, <?php echo $parent_name; ?>! 👋</h1>
                    <p class="welcome-subtitle">Track your children's academic progress, attendance, fees, and weekly study plans in one place.</p>
                </div>
                <div class="col-md-4 text-md-end">
                    <div class="d-inline-block bg-white text-dark rounded-pill px-4 py-2 shadow">
                        <small class="text-muted">Session ID</small>
                        <div class="fw-bold"><?php echo $active_session_id; ?></div>
                    </div>
                </div>
            </div>
        </div>

        <?php if (!empty($error)): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <i class="fas fa-exclamation-triangle me-2"></i>
                <?php echo $error; ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <!-- Students Section -->
        <div class="card mb-4 border-0 shadow-sm">
            <div class="card-header bg-transparent border-0 py-3">
                <h4 class="mb-0 text-dark">
                    <i class="fas fa-users me-2 text-primary"></i>
                    My Children 
                    <span class="badge bg-primary ms-2"><?php echo count($students_data); ?></span>
                </h4>
            </div>
            <div class="card-body">
                <?php if (empty($students_data)): ?>
                    <div class="text-center py-5">
                        <i class="fas fa-user-slash fa-3x text-muted mb-3"></i>
                        <h5 class="text-muted">No Students Linked</h5>
                        <p class="text-muted">Please contact school administration to link your children to your account.</p>
                    </div>
                <?php else: ?>
                    <div class="row">
                        <?php foreach ($students_data as $student): 
                            $gender_icon = ($student['gender'] == 'Female') ? 'fas fa-female' : 'fas fa-male';
                            $status_class = ($student['status'] == 'active') ? 'bg-success' : 'bg-warning';
                            $has_syllabus = !empty($syllabus_data[$student['id']]) && count($syllabus_data[$student['id']]) > 0;
                        ?>
                            <div class="col-xl-6 col-lg-12 mb-4">
                                <div class="card student-card">
                                    <div class="card-header student-card-header d-flex align-items-center">
                                        <div class="student-avatar">
                                            <i class="<?php echo $gender_icon; ?>"></i>
                                        </div>
                                        <div class="student-info">
                                            <div class="student-name"><?php echo htmlspecialchars($student['first_name'] . ' ' . $student['last_name']); ?></div>
                                            <div class="student-details">
                                                Roll No: <?php echo htmlspecialchars($student['roll_no'] ?? 'N/A'); ?>
                                                <span class="badge <?php echo $status_class; ?> status-badge ms-2"><?php echo ucfirst($student['status']); ?></span>
                                                <?php if ($has_syllabus): ?>
                                                    <span class="badge bg-info status-badge ms-2" data-bs-toggle="tooltip" title="Syllabus Available">
                                                        <i class="fas fa-book me-1"></i>Syllabus
                                                    </span>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="card-body">
                                        <div class="row mb-3">
                                            <div class="col-6">
                                                <small class="text-muted">Class & Section</small>
                                                <div class="fw-bold text-dark"><?php echo htmlspecialchars($student['class_name'] ?? 'N/A'); ?> - <?php echo htmlspecialchars($student['section_name'] ?? 'N/A'); ?></div>
                                            </div>
                                            <div class="col-6">
                                                <small class="text-muted">Father's Name</small>
                                                <div class="fw-bold text-dark"><?php echo htmlspecialchars($student['father_name']); ?></div>
                                            </div>
                                        </div>
                                        
                                        <div class="actions-container">
                                            <div class="row g-2">
                                                <div class="col-sm-6 col-12">
                                                    <a href="attendance.php?student_id=<?php echo $student['id']; ?>" class="btn action-btn btn-attendance w-100">
                                                        <i class="fas fa-clipboard-check me-1"></i> Attendance
                                                    </a>
                                                </div>
                                                <div class="col-sm-6 col-12">
                                                    <a href="fees.php?student_id=<?php echo $student['id']; ?>" class="btn action-btn btn-fees w-100">
                                                        <i class="fas fa-money-bill-wave me-1"></i> Fees
                                                    </a>
                                                </div>
                                                <div class="col-sm-6 col-12">
                                                    <a href="result_card.php?student_id=<?php echo $student['id']; ?>" class="btn action-btn btn-result w-100">
                                                        <i class="fas fa-poll me-1"></i> Result
                                                    </a>
                                                </div>
                                                <div class="col-sm-6 col-12">
                                                    <a href="student_planner_view.php?student_id=<?php echo $student['id']; ?>" class="btn action-btn btn-planner w-100">
                                                        <i class="fas fa-calendar-alt me-1"></i> Study Plan
                                                    </a>
                                                </div>
                                                <?php if ($has_syllabus): ?>
                                                <div class="col-12 mt-2">
                                                    <button type="button" class="btn action-btn btn-syllabus w-100" data-bs-toggle="modal" data-bs-target="#syllabusModal<?php echo $student['id']; ?>">
                                                        <i class="fas fa-book me-1"></i> View Syllabus
                                                    </button>
                                                </div>
                                                <?php else: ?>
                                                <div class="col-12 mt-2">
                                                    <button type="button" class="btn action-btn btn-secondary w-100" disabled>
                                                        <i class="fas fa-book me-1"></i> No Syllabus Available
                                                    </button>
                                                </div>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Syllabus Modal for each student -->
                            <?php if ($has_syllabus): ?>
                            <div class="modal fade syllabus-modal" id="syllabusModal<?php echo $student['id']; ?>" tabindex="-1" aria-labelledby="syllabusModalLabel<?php echo $student['id']; ?>" aria-hidden="true">
                                <div class="modal-dialog modal-lg">
                                    <div class="modal-content">
                                        <div class="modal-header">
                                            <h5 class="modal-title" id="syllabusModalLabel<?php echo $student['id']; ?>">
                                                <i class="fas fa-book me-2"></i>
                                                Syllabus - <?php echo htmlspecialchars($student['first_name'] . ' ' . $student['last_name']); ?>
                                            </h5>
                                            <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                                        </div>
                                        <div class="modal-body">
                                            <div class="mb-3">
                                                <strong>Class:</strong> <?php echo htmlspecialchars($student['class_name'] ?? 'N/A'); ?><br>
                                                <strong>Section:</strong> <?php echo htmlspecialchars($student['section_name'] ?? 'N/A'); ?>
                                            </div>
                                            
                                            <?php foreach ($syllabus_data[$student['id']] as $syllabus): ?>
                                                <div class="syllabus-item">
                                                    <div class="syllabus-title">
                                                        <?php echo htmlspecialchars($syllabus['original_name']); ?>
                                                    </div>
                                                    <div class="syllabus-meta">
                                                        <span class="syllabus-type <?php echo $syllabus['syllabus_type']; ?>">
                                                            <?php echo ucfirst(str_replace('_', ' ', $syllabus['syllabus_type'])); ?>
                                                        </span>
                                                        <span class="syllabus-year">
                                                            <?php echo htmlspecialchars($syllabus['academic_year']); ?>
                                                        </span>
                                                    </div>
                                                    <div class="syllabus-file-info">
                                                        <small>
                                                            <i class="fas fa-file me-1"></i>
                                                            <?php echo htmlspecialchars($syllabus['file_name']); ?>
                                                            • 
                                                            <?php 
                                                            $file_size = $syllabus['file_size'];
                                                            if ($file_size < 1024) {
                                                                echo $file_size . ' bytes';
                                                            } elseif ($file_size < 1048576) {
                                                                echo round($file_size / 1024, 2) . ' KB';
                                                            } else {
                                                                echo round($file_size / 1048576, 2) . ' MB';
                                                            }
                                                            ?>
                                                            • 
                                                            <i class="fas fa-calendar me-1"></i>
                                                            <?php echo date('M j, Y g:i A', strtotime($syllabus['uploaded_at'])); ?>
                                                        </small>
                                                    </div>
                                                    <div class="syllabus-actions">
                                                        <?php if (!empty($syllabus['file_path'])): ?>
                                                            <a href="<?php echo htmlspecialchars($syllabus['file_path']); ?>" target="_blank" class="btn btn-view">
                                                                <i class="fas fa-eye me-1"></i> View
                                                            </a>
                                                            <a href="<?php echo htmlspecialchars($syllabus['file_path']); ?>" download="<?php echo htmlspecialchars($syllabus['original_name']); ?>" class="btn btn-download">
                                                                <i class="fas fa-download me-1"></i> Download
                                                            </a>
                                                        <?php else: ?>
                                                            <span class="text-danger">File path not available</span>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>
                                            <?php endforeach; ?>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <?php endif; ?>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
    // Initialize tooltips
    var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'))
    var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
        return new bootstrap.Tooltip(tooltipTriggerEl)
    });

    // Debug: Log modal events
    document.addEventListener('DOMContentLoaded', function() {
        var modals = document.querySelectorAll('.syllabus-modal');
        modals.forEach(function(modal) {
            modal.addEventListener('show.bs.modal', function (event) {
                console.log('Modal opening:', event.target.id);
            });
        });
    });
    </script>

    <?php include_once('includes/footer.php'); ?>
</body>
</html>