<?php
// FILE: parent/fees.php - Student Fee History/Status View for Parent
session_start();

// Simplified login check - sirf check karo ke user logged in hai ya nahi
if (!isset($_SESSION['user_id'])) {
    header('Location: ../login.php');
    exit;
}

include_once('../config/db.php');
include_once('includes/header.php');
$conn = getDBConnection();

$parent_user_id = (int)$_SESSION['user_id'];
$student_id = (int)($_GET['student_id'] ?? 0);
$error = '';
$fees_data = [];

// --- 1. Security Check ---
$is_linked_q = $conn->prepare("SELECT COUNT(*) FROM parent_student_link WHERE parent_user_id = ? AND student_id = ?");
$is_linked_q->bind_param("ii", $parent_user_id, $student_id);
$is_linked_q->execute();
$is_linked_q->bind_result($is_linked);
$is_linked_q->fetch();
$is_linked_q->close();

if ($student_id == 0 || $is_linked == 0) {
    $error = "Error: Invalid Student ID or the student is not linked to your account.";
    goto display_fees;
}

// --- 2. Fetch Student Details & Enrollment ---
$details_query = "
    SELECT 
        s.first_name, s.last_name, 
        ac.name AS class_name,
        se.roll_no,
        se.status as enrollment_status
    FROM students s
    LEFT JOIN student_enrollments se ON s.id = se.student_id
    LEFT JOIN academic_classes ac ON se.class_id = ac.id
    WHERE s.id = ?
    ORDER BY se.id DESC LIMIT 1
";
$student_details = [];
if ($stmt_det = $conn->prepare($details_query)) {
    $stmt_det->bind_param("i", $student_id);
    $stmt_det->execute();
    $student_details = $stmt_det->get_result()->fetch_assoc();
    $stmt_det->close();
}

if (!$student_details) {
    $error = "Error: Student details not found.";
    goto display_fees;
}

$student_full_name = htmlspecialchars($student_details['first_name'] . ' ' . $student_details['last_name']);
$student_class = htmlspecialchars($student_details['class_name'] ?? 'N/A');
$roll_no = htmlspecialchars($student_details['roll_no'] ?? 'N/A');
$enrollment_status = htmlspecialchars($student_details['enrollment_status'] ?? 'active');

// --- 3. Fetch Fees History and Status ---
$fees_data = [];

// Fetch fee invoices with payments - USING CORRECT COLUMN NAMES
try {
    $fees_query = "
        SELECT 
            fi.id AS invoice_id,
            fi.voucher_no AS invoice_number,
            fi.month_year AS billing_month, 
            fi.total_amount, 
            COALESCE(SUM(fp.amount), 0) AS amount_paid,
            fi.due_date,
            fi.status,
            fi.issue_date,
            fi.net_payable,
            fi.fine_charged
        FROM fee_invoices fi
        LEFT JOIN fee_payments fp ON fi.id = fp.invoice_id
        WHERE fi.student_id = ?
        GROUP BY fi.id
        ORDER BY fi.due_date DESC
    ";

    if ($stmt = $conn->prepare($fees_query)) {
        $stmt->bind_param("i", $student_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            $fees_data = $result->fetch_all(MYSQLI_ASSOC);
        } else {
            $error = "No fee records found for this student.";
        }
        $stmt->close();
    } else {
        $error = "Database error: Unable to fetch fee data.";
    }
} catch (Exception $e) {
    $error = "Fee system error: " . $e->getMessage();
}

display_fees:
$conn->close();
?>

<div class="page-content">
    <main>
        <div class="container-fluid px-4">
            <h1 class="mt-4">Fee History & Status</h1>
            <ol class="breadcrumb mb-4">
                <li class="breadcrumb-item"><a href="dashboard.php">Dashboard</a></li>
                <li class="breadcrumb-item active">Fee History</li>
            </ol>

            <?php if (!empty($error)): ?>
                <div class="alert alert-danger">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    <?php echo $error; ?>
                </div>
            <?php endif; ?>

            <?php if ($student_id != 0 && $is_linked == 1 && empty($error)): ?>
                
                <div class="card mb-4">
                    <div class="card-header">
                        <i class="fas fa-user-graduate me-1"></i>
                        Student: <strong><?php echo $student_full_name; ?></strong> | 
                        Class: <strong><?php echo $student_class; ?></strong> | 
                        Roll No: <strong><?php echo $roll_no; ?></strong>
                    </div>
                </div>
                
                <?php if (!empty($fees_data)): ?>
                <div class="card mb-4">
                    <div class="card-header text-white" style="background-color: #6c757d;">
                        <i class="fas fa-money-bill-wave me-1"></i>
                        Monthly Fee Vouchers
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-bordered table-striped">
                                <thead class="table-dark">
                                    <tr>
                                        <th>Voucher #</th>
                                        <th>Billing Month</th>
                                        <th>Issue Date</th>
                                        <th>Total Amount</th>
                                        <th>Net Payable</th>
                                        <th>Fine Charged</th>
                                        <th>Amount Paid</th>
                                        <th>Balance Due</th>
                                        <th>Due Date</th>
                                        <th>Status</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php 
                                    $total_balance = 0;
                                    $total_paid = 0;
                                    $total_invoiced = 0;
                                    $total_net_payable = 0;
                                    $total_fine = 0;
                                    
                                    foreach ($fees_data as $invoice): 
                                        $amount_paid = (float)($invoice['amount_paid'] ?? 0);
                                        $total_amount = (float)($invoice['total_amount'] ?? 0);
                                        $net_payable = (float)($invoice['net_payable'] ?? $total_amount);
                                        $fine_charged = (float)($invoice['fine_charged'] ?? 0);
                                        $balance = $net_payable - $amount_paid;
                                        
                                        $total_balance += $balance;
                                        $total_paid += $amount_paid;
                                        $total_invoiced += $total_amount;
                                        $total_net_payable += $net_payable;
                                        $total_fine += $fine_charged;
                                        
                                        // Status determination
                                        $status_class = 'bg-secondary';
                                        $status_text = 'Pending';
                                        
                                        if ($balance <= 0 && $net_payable > 0) {
                                            $status_class = 'bg-success';
                                            $status_text = 'Paid';
                                        } elseif ($balance > 0) {
                                            $due_date = strtotime($invoice['due_date']);
                                            $today = time();
                                            if ($due_date < $today) {
                                                $status_class = 'bg-danger';
                                                $status_text = 'Overdue';
                                            } else {
                                                $status_class = 'bg-warning';
                                                $status_text = 'Due';
                                            }
                                        }
                                    ?>
                                        <tr>
                                            <td><strong><?php echo htmlspecialchars($invoice['invoice_number'] ?? 'VOUCHER-' . $invoice['invoice_id']); ?></strong></td>
                                            <td><?php echo htmlspecialchars($invoice['billing_month'] ?? 'N/A'); ?></td>
                                            <td><?php echo date('d-M-Y', strtotime($invoice['issue_date'])); ?></td>
                                            <td>Rs. <?php echo number_format($total_amount, 0); ?></td>
                                            <td>Rs. <?php echo number_format($net_payable, 0); ?></td>
                                            <td class="text-danger">Rs. <?php echo number_format($fine_charged, 0); ?></td>
                                            <td class="text-success">Rs. <?php echo number_format($amount_paid, 0); ?></td>
                                            <td class="<?php echo ($balance > 0) ? 'text-danger fw-bold' : 'text-success'; ?>">
                                                Rs. <?php echo number_format($balance, 0); ?>
                                            </td>
                                            <td><?php echo date('d-M-Y', strtotime($invoice['due_date'])); ?></td>
                                            <td>
                                                <span class="badge <?php echo $status_class; ?>"><?php echo $status_text; ?></span>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                                <tfoot class="table-light">
                                    <tr>
                                        <td colspan="3" class="text-end fw-bold">Totals:</td>
                                        <td class="fw-bold">Rs. <?php echo number_format($total_invoiced, 0); ?></td>
                                        <td class="fw-bold">Rs. <?php echo number_format($total_net_payable, 0); ?></td>
                                        <td class="text-danger fw-bold">Rs. <?php echo number_format($total_fine, 0); ?></td>
                                        <td class="text-success fw-bold">Rs. <?php echo number_format($total_paid, 0); ?></td>
                                        <td class="fw-bold <?php echo ($total_balance > 0) ? 'text-danger' : 'text-success'; ?>">
                                            Rs. <?php echo number_format($total_balance, 0); ?>
                                        </td>
                                        <td colspan="2">
                                            <span class="badge <?php echo ($total_balance > 0) ? 'bg-warning' : 'bg-success'; ?>">
                                                <?php echo ($total_balance > 0) ? 'Balance Due' : 'All Paid'; ?>
                                            </span>
                                        </td>
                                    </tr>
                                </tfoot>
                            </table>
                        </div>
                        
                        <!-- Summary Cards -->
                        <div class="row mt-4">
                            <div class="col-md-2">
                                <div class="card bg-primary text-white">
                                    <div class="card-body text-center p-2">
                                        <small>Total Invoiced</small>
                                        <h6>Rs. <?php echo number_format($total_invoiced, 0); ?></h6>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-2">
                                <div class="card bg-info text-white">
                                    <div class="card-body text-center p-2">
                                        <small>Net Payable</small>
                                        <h6>Rs. <?php echo number_format($total_net_payable, 0); ?></h6>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-2">
                                <div class="card bg-danger text-white">
                                    <div class="card-body text-center p-2">
                                        <small>Total Fine</small>
                                        <h6>Rs. <?php echo number_format($total_fine, 0); ?></h6>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-2">
                                <div class="card bg-success text-white">
                                    <div class="card-body text-center p-2">
                                        <small>Total Paid</small>
                                        <h6>Rs. <?php echo number_format($total_paid, 0); ?></h6>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-2">
                                <div class="card <?php echo ($total_balance > 0) ? 'bg-warning' : 'bg-secondary'; ?> text-white">
                                    <div class="card-body text-center p-2">
                                        <small>Balance Due</small>
                                        <h6>Rs. <?php echo number_format($total_balance, 0); ?></h6>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-2">
                                <div class="card <?php echo ($total_balance > 0) ? 'bg-danger' : 'bg-success'; ?> text-white">
                                    <div class="card-body text-center p-2">
                                        <small>Overall Status</small>
                                        <h6><?php echo ($total_balance > 0) ? 'Due' : 'Clear'; ?></h6>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <?php elseif (empty($error)): ?>
                    <div class="alert alert-info text-center">
                        <i class="fas fa-info-circle me-2"></i>
                        Is bachche ke liye koi fee record nahi mila.
                    </div>
                <?php endif; ?>

            <?php elseif ($student_id == 0): ?>
                 <div class="alert alert-info mt-4">
                     <i class="fas fa-info-circle me-2"></i>
                     Kripya <a href="dashboard.php" class="alert-link">Dashboard</a> par wapas jaakar bachcha select karein.
                 </div>
            <?php endif; ?>
        </div>
    </main>

    <?php include_once('includes/footer.php'); ?>
</div>