<?php
// FILE: ajax_handler.php - Handles dynamic AJAX requests (FINAL)

// 1. Session Start and DB Connection
session_start();
// NOTE: Is path ko double-check karein. Agar yeh file 'admin' folder ke bahar hai, 
// aur 'includes' folder bhi uske bahar hai, toh yeh path theek hoga.
include_once('includes/db_connection.php'); 

header('Content-Type: application/json');

// 2. Security Check: Only Admin (role_id = 1) should access this handler
if (!isset($_SESSION['role_id']) || $_SESSION['role_id'] != 1) {
    http_response_code(403);
    echo json_encode(['success' => false, 'message' => 'Access Denied: Admin role required.']);
    exit;
}

$conn = getDBConnection();
if ($conn->connect_error) {
    // Agar DB connection fail ho jaye
    echo json_encode(['success' => false, 'message' => 'Database connection failed.']);
    exit;
}

$action = $_GET['action'] ?? '';
$response = ['success' => false, 'data' => []]; // Standardized response format

try {
    switch ($action) {
        
        // --- Existing Action: Get Subjects by Class (Included for integrity) ---
        case 'get_subjects_by_class':
            $class_id = (int)($_GET['class_id'] ?? 0);
            $session_id = (int)($_GET['session_id'] ?? 0);
            
            if ($class_id > 0 && $session_id > 0) {
                $query = "
                    SELECT DISTINCT sub.id, sub.name, sub.code 
                    FROM course_allocation ca
                    JOIN subjects sub ON ca.subject_id = sub.id
                    WHERE ca.class_id = ? AND ca.session_id = ?
                    ORDER BY sub.name
                ";
                
                $stmt = $conn->prepare($query);
                $stmt->bind_param("ii", $class_id, $session_id);
                $stmt->execute();
                $result = $stmt->get_result();
                
                while ($row = $result->fetch_assoc()) {
                    $response['data'][] = $row;
                }
                $response['success'] = true;
                $stmt->close();
            } else {
                 $response['message'] = 'Class ID or Session ID missing.';
            }
            break;

        // --- NEW ACTION 1: Get all Classes (for Parent Assignment) ---
        case 'get_classes':
            // academic_classes table use ho rahi hai
            $result = $conn->query("SELECT id, name FROM academic_classes ORDER BY order_no ASC"); 
            if ($result) {
                while ($row = $result->fetch_assoc()) {
                    $response['data'][] = $row;
                }
                $response['success'] = true;
            } else {
                 $response['message'] = 'Error fetching classes: ' . $conn->error;
            }
            break;

        // --- NEW ACTION 2: Get Sections by Class ID (for Parent Assignment) ---
        case 'get_sections':
            $class_id = (int)($_GET['class_id'] ?? 0);
            if ($class_id > 0) {
                // Assuming 'academic_sections' table exists and links to classes
                $stmt = $conn->prepare("SELECT id, name FROM academic_sections WHERE class_id = ? AND status='active' ORDER BY name");
                $stmt->bind_param("i", $class_id);
                $stmt->execute();
                $result = $stmt->get_result();
                
                while ($row = $result->fetch_assoc()) {
                    $response['data'][] = $row;
                }
                $response['success'] = true;
                $stmt->close();
            } else {
                $response['message'] = 'Class ID missing.';
            }
            break;

        // --- NEW ACTION 3: Get Students by Class and Section (CRITICAL FOR LINKING) ---
        case 'get_students':
            $class_id = (int)($_GET['class_id'] ?? 0);
            $section_id = (int)($_GET['section_id'] ?? 0);
            
            // 1. Fetch Active Session ID - CRITICAL FOR ENROLLMENT
            $active_session_id = 0;
            // Assuming settings table mein active session ID stored hai
            $settings_query = $conn->query("SELECT setting_value FROM settings WHERE setting_key='academic.active_session_id'");
            if ($settings_query && $row = $settings_query->fetch_assoc()) {
                $active_session_id = (int)$row['setting_value'];
            }
            
            if ($class_id > 0 && $section_id > 0 && $active_session_id > 0) {
                // Joining students table with student_enrollments to get roll_no and filter by active session/class/section
                $query = "
                    SELECT 
                        s.id, 
                        CONCAT(s.first_name, ' ', s.last_name) AS full_name, 
                        en.roll_no 
                    FROM students s
                    JOIN student_enrollments en ON s.id = en.student_id
                    WHERE en.class_id = ? 
                    AND en.section_id = ? 
                    AND en.session_id = ? 
                    AND s.status = 'active' 
                    ORDER BY en.roll_no
                ";
                
                $stmt = $conn->prepare($query);
                $stmt->bind_param("iii", $class_id, $section_id, $active_session_id);
                $stmt->execute();
                $result = $stmt->get_result();
                
                while ($row = $result->fetch_assoc()) {
                    $response['data'][] = [
                        'id' => $row['id'],
                        'name' => $row['full_name'], // Full Name (First + Last)
                        'roll_no' => $row['roll_no']
                    ];
                }
                $response['success'] = true;
                $stmt->close();
            } else {
                $response['message'] = 'Required IDs (Class/Section/Session) missing or invalid.';
            }
            break;

        default:
            $response['message'] = 'Invalid action.';
            break;
    }

} catch (Exception $e) {
    $response['success'] = false;
    $response['message'] = 'Server Error: ' . $e->getMessage();
}

$conn->close();
echo json_encode($response);


<?php
// admin/ajax_handler.php - Handles all AJAX requests for user_management.php

session_start();

// Authorization Check (Admin Only)
$user_role_id = $_SESSION['role_id'] ?? 0;
if (!isset($user_role_id) || $user_role_id != 1) { 
    // Unauthorized access
    header('Content-Type: application/json');
    echo json_encode(['success' => false, 'message' => 'Unauthorized access.']);
    exit;
}

// Database Connection
include_once('../config/db.php'); 
$conn = getDBConnection(); 

header('Content-Type: application/json');
$response = ['success' => false, 'data' => [], 'message' => 'Invalid action.'];

$action = $_GET['action'] ?? '';

try {
    switch ($action) {
        case 'get_classes':
            // Fetch all classes
            $stmt = $conn->prepare("SELECT id, name FROM classes ORDER BY name");
            $stmt->execute();
            $result = $stmt->get_result();
            $classes = [];
            while ($row = $result->fetch_assoc()) {
                $classes[] = $row;
            }
            $stmt->close();
            $response = ['success' => true, 'data' => $classes];
            break;

        case 'get_sections':
            $class_id = (int)($_GET['class_id'] ?? 0);
            if ($class_id > 0) {
                // Fetch sections for the selected class
                // Assumption: sections table has a class_id column
                $stmt = $conn->prepare("SELECT id, name FROM sections WHERE class_id = ? ORDER BY name");
                $stmt->bind_param("i", $class_id);
                $stmt->execute();
                $result = $stmt->get_result();
                $sections = [];
                while ($row = $result->fetch_assoc()) {
                    $sections[] = $row;
                }
                $stmt->close();
                $response = ['success' => true, 'data' => $sections];
            } else {
                 $response = ['success' => false, 'message' => 'Class ID not provided.'];
            }
            break;

        case 'get_students':
            $class_id = (int)($_GET['class_id'] ?? 0);
            $section_id = (int)($_GET['section_id'] ?? 0);

            if ($class_id > 0 && $section_id > 0) {
                // Fetch student users for the selected class and section
                // IMPORTANT ASSUMPTION:
                // 1. Students ka Role ID = 3 hai.
                // 2. Ek 'students' table hai jo 'users' table se 'user_id' se linked hai.
                $stmt = $conn->prepare("
                    SELECT 
                        u.id, 
                        s.roll_no, 
                        u.name
                    FROM users u
                    JOIN students s ON u.id = s.user_id
                    WHERE u.role_id = 3 AND s.class_id = ? AND s.section_id = ?
                    ORDER BY s.roll_no
                ");
                
                $stmt->bind_param("ii", $class_id, $section_id);
                $stmt->execute();
                $result = $stmt->get_result();
                $students = [];
                while ($row = $result->fetch_assoc()) {
                    $students[] = $row;
                }
                $stmt->close();
                $response = ['success' => true, 'data' => $students];
            } else {
                 $response = ['success' => false, 'message' => 'Class ID or Section ID not provided.'];
            }
            break;

        default:
            // Invalid action, use default $response
            break;
    }

} catch (Exception $e) {
    $response = ['success' => false, 'message' => 'Database Error: ' . $e->getMessage()];
}

if (isset($conn) && is_object($conn) && method_exists($conn, 'close')) {
    $conn->close(); 
}

echo json_encode($response);
exit;
?>