<?php
// admin/attendance.php - Daily Student Attendance Management

// Header aur Sidebar files include karein.
include_once('includes/header.php'); 
// Assuming $settings, $user_role_id, aur $conn (connection function) header.php mein hain.

// Authorization Check: Agar zaroori ho, to yahan check laga sakte hain (e.g., role_id = 1 ya 2)
// if (!isset($user_role_id) || !in_array($user_role_id, [1, 2])) { /* redirect */ }

$conn = getDBConnection();
$message = $_GET['msg'] ?? '';
$error = $_GET['err'] ?? '';

// --- Data Fetching Setup ---
// Active Session ID fetch karein. Default to 1.
$current_session_id = $settings['academic.active_session_id'] ?? 1;

// Fetch current session name
$session_query = "SELECT id, name FROM academic_sessions WHERE id = ?";
$session_stmt = $conn->prepare($session_query);
$session_stmt->bind_param("i", $current_session_id);
$session_stmt->execute();
$current_session = $session_stmt->get_result()->fetch_assoc();
$session_stmt->close();

// Classes aur Sections fetch karein
$classes_result = $conn->query("SELECT id, name FROM academic_classes ORDER BY order_no ASC");
$classes = $classes_result->fetch_all(MYSQLI_ASSOC);

// Table name 'sections' se 'academic_sections' mein badal diya gaya hai.
$sections_result = $conn->query("SELECT id, name FROM academic_sections WHERE status='active' ORDER BY name ASC");
$sections = $sections_result->fetch_all(MYSQLI_ASSOC);

// Dropdown mein display ke liye ID ko key banao
$class_map = array_column($classes, 'name', 'id');
$section_map = array_column($sections, 'name', 'id');

$selected_class_id = $_GET['class_id'] ?? null;
$selected_section_id = $_GET['section_id'] ?? null;
$selected_date = $_GET['attendance_date'] ?? date('Y-m-d');

$students = [];
$attendance_data = [];
$sheet_id = null;
$is_locked = false;

// --- Load Students and Attendance Data ---
if ($selected_class_id && $selected_section_id) {
    $class_id = (int)$selected_class_id;
    $section_id = (int)$selected_section_id;

    // 1. Fetch Students of the selected Class/Section
    $students_query = "
        SELECT 
            s.id, s.first_name, s.last_name, en.roll_no
        FROM students s
        JOIN student_enrollments en ON s.id = en.student_id
        WHERE en.session_id = {$current_session_id} 
          AND en.class_id = {$class_id} 
          AND en.section_id = {$section_id}
          AND s.status = 'active'
        ORDER BY en.roll_no ASC
    ";
    $students_result = $conn->query($students_query);
    if ($students_result) {
        $students = $students_result->fetch_all(MYSQLI_ASSOC);
    } else {
        $error = "Error fetching students: " . $conn->error;
    }

    // 2. Check for existing Attendance Sheet for this date/class/section
    $sheet_query = $conn->prepare("SELECT id, is_locked FROM daily_attendance_sheets 
        WHERE session_id = ? AND class_id = ? AND section_id = ? AND attendance_date = ?");
    $sheet_query->bind_param("iiis", $current_session_id, $class_id, $section_id, $selected_date);
    $sheet_query->execute();
    $sheet_result = $sheet_query->get_result();

    if ($sheet_result && $sheet_result->num_rows > 0) {
        $sheet = $sheet_result->fetch_assoc();
        $sheet_id = $sheet['id'];
        $is_locked = (bool)$sheet['is_locked'];

        // 3. Fetch existing Attendance Records
        $records_result = $conn->query("SELECT student_id, status, note FROM daily_attendance_records WHERE sheet_id = {$sheet_id}");
        if ($records_result) {
            while ($record = $records_result->fetch_assoc()) {
                $attendance_data[$record['student_id']] = ['status' => $record['status'], 'note' => $record['note']];
            }
        }
    }
}

// Connection close karein.
if (isset($conn) && is_object($conn)) {
    $conn->close();
}
// --- End of PHP Logic ---
?>

<h1 class="mb-4">Daily Attendance <i class="fas fa-calendar-check"></i></h1>

<?php if ($message): ?>
    <div class="alert alert-success"><?php echo htmlspecialchars($message); ?></div>
<?php endif; ?>
<?php if ($error): ?>
    <div class="alert alert-danger"><?php echo htmlspecialchars($error); ?></div>
<?php endif; ?>

<div class="card p-3 mb-4 shadow-sm">
    <h5 class="card-title">Select Session, Class, Section & Date</h5>
    <form method="GET" action="attendance.php" class="form-inline">
        <div class="form-group mr-3 mb-2">
            <label class="mr-2">Session:</label>
            <?php if ($current_session): ?>
                <input type="hidden" name="session_id" value="<?php echo $current_session['id']; ?>">
                <select class="form-control" disabled>
                    <option value="<?php echo $current_session['id']; ?>" selected>
                        <?php echo htmlspecialchars($current_session['name']); ?> (Current Session)
                    </option>
                </select>
            <?php else: ?>
                <select name="session_id" class="form-control" required>
                    <option value="">-- Select Session --</option>
                    <?php foreach ($sessions as $session): ?>
                        <option value="<?php echo $session['id']; ?>" <?php echo $current_session_id == $session['id'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($session['name']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
                <div class="alert alert-warning mt-2">No active session found. Please select a session manually.</div>
            <?php endif; ?>
        </div>
        <div class="form-group mr-3 mb-2">
            <label class="mr-2">Class:</label>
            <select name="class_id" class="form-control" required>
                <option value="">-- Select Class --</option>
                <?php foreach ($classes as $class): ?>
                    <option value="<?php echo $class['id']; ?>" <?php echo $selected_class_id == $class['id'] ? 'selected' : ''; ?>><?php echo htmlspecialchars($class['name']); ?></option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="form-group mr-3 mb-2">
            <label class="mr-2">Section:</label>
            <select name="section_id" class="form-control" required>
                <option value="">-- Select Section --</option>
                <?php foreach ($sections as $section): ?>
                    <option value="<?php echo $section['id']; ?>" <?php echo $selected_section_id == $section['id'] ? 'selected' : ''; ?>><?php echo htmlspecialchars($section['name']); ?></option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="form-group mr-3 mb-2">
            <label class="mr-2">Date:</label>
            <input type="date" name="attendance_date" class="form-control" value="<?php echo htmlspecialchars($selected_date); ?>" required>
        </div>
        <button type="submit" class="btn btn-info mb-2"><i class="fas fa-search"></i> Load Students</button>
    </form>
</div>

<?php if ($selected_class_id && $selected_section_id): ?>
    <?php if ($is_locked): ?>
        <div class="alert alert-warning">Attendance sheet has been <strong>locked</strong> for this date. Changes cannot be made.</div>
    <?php endif; ?>
    
    <?php if (count($students) > 0): ?>
        <div class="card shadow-sm">
            <div class="card-header bg-primary text-white">
                <h4 class="mb-0">
                    <i class="fas fa-users"></i> 
                    Attendance for Class <strong><?php echo htmlspecialchars($class_map[$selected_class_id] ?? 'N/A'); ?></strong> / 
                    Section <strong><?php echo htmlspecialchars($section_map[$selected_section_id] ?? 'N/A'); ?></strong> on 
                    <strong><?php echo htmlspecialchars($selected_date); ?></strong>
                </h4>
            </div>
            
            <form action="attendance_process.php" method="POST">
                <input type="hidden" name="action" value="save_attendance">
                <input type="hidden" name="sheet_id" value="<?php echo $sheet_id; ?>">
                <input type="hidden" name="class_id" value="<?php echo $selected_class_id; ?>">
                <input type="hidden" name="section_id" value="<?php echo $selected_section_id; ?>">
                <input type="hidden" name="attendance_date" value="<?php echo $selected_date; ?>">
                <input type="hidden" name="session_id" value="<?php echo $current_session_id; ?>">
                
                <div class="card-body p-0">
                    <div class="table-responsive">
                        <table class="table table-bordered table-hover table-sm mb-0">
                            <thead class="thead-dark">
                                <tr>
                                    <th style="width: 60px;" class="text-center">Roll No</th>
                                    <th>Student Name</th>
                                    <th style="width: 300px;" class="text-center">Attendance Status</th>
                                    <th style="width: 200px;">Remarks</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($students as $index => $student): ?>
                                    <?php 
                                        $student_id = $student['id'];
                                        $current_status = $attendance_data[$student_id]['status'] ?? 'Present';
                                        $current_note = $attendance_data[$student_id]['note'] ?? '';
                                        $disabled_attr = $is_locked ? 'disabled' : '';
                                        $row_class = $index % 2 === 0 ? '' : 'table-row-light';
                                    ?>
                                <tr class="<?php echo $row_class; ?>">
                                    <td class="text-center align-middle">
                                        <span class="badge badge-secondary badge-pill"><?php echo htmlspecialchars($student['roll_no'] ?? '-'); ?></span>
                                    </td>
                                    <td class="align-middle">
                                        <strong><?php echo htmlspecialchars($student['first_name'] . ' ' . $student['last_name']); ?></strong>
                                    </td>
                                    <td class="text-center align-middle">
                                        <div class="btn-group btn-group-toggle attendance-radio-group" data-toggle="buttons">
                                            <label class="btn btn-outline-success btn-sm <?php echo $current_status === 'Present' ? 'active' : ''; ?>">
                                                <input type="radio" name="status[<?php echo $student_id; ?>]" value="Present" 
                                                    <?php echo $current_status === 'Present' ? 'checked' : ''; ?> <?php echo $disabled_attr; ?>> 
                                                <i class="fas fa-check-circle"></i> Present
                                            </label>
                                            <label class="btn btn-outline-danger btn-sm <?php echo $current_status === 'Absent' ? 'active' : ''; ?>">
                                                <input type="radio" name="status[<?php echo $student_id; ?>]" value="Absent" 
                                                    <?php echo $current_status === 'Absent' ? 'checked' : ''; ?> <?php echo $disabled_attr; ?>> 
                                                <i class="fas fa-times-circle"></i> Absent
                                            </label>
                                            <label class="btn btn-outline-warning btn-sm <?php echo $current_status === 'Leave' ? 'active' : ''; ?>">
                                                <input type="radio" name="status[<?php echo $student_id; ?>]" value="Leave" 
                                                    <?php echo $current_status === 'Leave' ? 'checked' : ''; ?> <?php echo $disabled_attr; ?>> 
                                                <i class="fas fa-envelope"></i> Leave
                                            </label>
                                            <label class="btn btn-outline-info btn-sm <?php echo $current_status === 'Late' ? 'active' : ''; ?>">
                                                <input type="radio" name="status[<?php echo $student_id; ?>]" value="Late" 
                                                    <?php echo $current_status === 'Late' ? 'checked' : ''; ?> <?php echo $disabled_attr; ?>> 
                                                <i class="fas fa-clock"></i> Late
                                            </label>
                                        </div>
                                    </td>
                                    <td class="align-middle">
                                        <input type="text" name="note[<?php echo $student_id; ?>]" 
                                            class="form-control form-control-sm remarks-input" 
                                            value="<?php echo htmlspecialchars($current_note); ?>" 
                                            placeholder="Optional remarks..." 
                                            <?php echo $disabled_attr; ?>>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>

                <?php if (!$is_locked): ?>
                    <div class="card-footer bg-light">
                        <div class="row align-items-center">
                            <div class="col-md-6">
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" name="lock_sheet" value="1" id="lockSheet">
                                    <label class="form-check-label text-danger" for="lockSheet">
                                        <i class="fas fa-lock"></i> <strong>Lock Attendance Sheet</strong>
                                        <small class="d-block text-muted">Once locked, attendance cannot be modified</small>
                                    </label>
                                </div>
                            </div>
                            <div class="col-md-6 text-right">
                                <button type="submit" class="btn btn-success btn-lg">
                                    <i class="fas fa-save"></i> Save Attendance
                                </button>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>
            </form>
        </div>

        <!-- Quick Action Buttons -->
        <?php if (!$is_locked): ?>
        <div class="mt-3">
            <div class="btn-group" role="group">
                <button type="button" class="btn btn-outline-success btn-sm" onclick="markAllPresent()">
                    <i class="fas fa-check-double"></i> Mark All Present
                </button>
                <button type="button" class="btn btn-outline-danger btn-sm" onclick="markAllAbsent()">
                    <i class="fas fa-times"></i> Mark All Absent
                </button>
                <button type="button" class="btn btn-outline-warning btn-sm" onclick="markAllLeave()">
                    <i class="fas fa-envelope"></i> Mark All Leave
                </button>
                <button type="button" class="btn btn-outline-info btn-sm" onclick="markAllLate()">
                    <i class="fas fa-clock"></i> Mark All Late
                </button>
            </div>
        </div>
        <?php endif; ?>

    <?php else: ?>
        <div class="alert alert-warning">
            <i class="fas fa-exclamation-triangle"></i> 
            No active students found in the selected class/section for the current session.
        </div>
    <?php endif; ?>
<?php else: ?>
    <div class="alert alert-info">
        <i class="fas fa-info-circle"></i> 
        Please select class, section and date to view students.
    </div>
<?php endif; ?>

<style>
.attendance-radio-group .btn {
    padding: 0.25rem 0.5rem;
    font-size: 0.8rem;
    margin: 1px;
}

.attendance-radio-group .btn.active {
    font-weight: bold;
}

.attendance-radio-group .btn-outline-success.active {
    background-color: #28a745;
    border-color: #28a745;
    color: white;
}

.attendance-radio-group .btn-outline-danger.active {
    background-color: #dc3545;
    border-color: #dc3545;
    color: white;
}

.attendance-radio-group .btn-outline-warning.active {
    background-color: #ffc107;
    border-color: #ffc107;
    color: #212529;
}

.attendance-radio-group .btn-outline-info.active {
    background-color: #17a2b8;
    border-color: #17a2b8;
    color: white;
}

.table-row-light {
    background-color: #f8f9fa;
}

.remarks-input {
    font-size: 0.85rem;
}

.card-header {
    font-size: 1.1rem;
}
</style>

<script>
// Quick action functions
function markAllPresent() {
    document.querySelectorAll('input[value="Present"]').forEach(radio => {
        radio.checked = true;
        radio.closest('label').classList.add('active');
        radio.closest('label').classList.remove('btn-outline-danger', 'btn-outline-warning', 'btn-outline-info');
        radio.closest('label').classList.add('btn-outline-success');
    });
}

function markAllAbsent() {
    document.querySelectorAll('input[value="Absent"]').forEach(radio => {
        radio.checked = true;
        radio.closest('label').classList.add('active');
        radio.closest('label').classList.remove('btn-outline-success', 'btn-outline-warning', 'btn-outline-info');
        radio.closest('label').classList.add('btn-outline-danger');
    });
}

function markAllLeave() {
    document.querySelectorAll('input[value="Leave"]').forEach(radio => {
        radio.checked = true;
        radio.closest('label').classList.add('active');
        radio.closest('label').classList.remove('btn-outline-success', 'btn-outline-danger', 'btn-outline-info');
        radio.closest('label').classList.add('btn-outline-warning');
    });
}

function markAllLate() {
    document.querySelectorAll('input[value="Late"]').forEach(radio => {
        radio.checked = true;
        radio.closest('label').classList.add('active');
        radio.closest('label').classList.remove('btn-outline-success', 'btn-outline-danger', 'btn-outline-warning');
        radio.closest('label').classList.add('btn-outline-info');
    });
}

// Auto update button styles when radio changes
document.addEventListener('change', function(e) {
    if (e.target.type === 'radio' && e.target.name.startsWith('status')) {
        const label = e.target.closest('label');
        const btnGroup = label.closest('.btn-group');
        
        // Remove active class from all buttons in group
        btnGroup.querySelectorAll('label').forEach(lbl => {
            lbl.classList.remove('active', 'btn-outline-success', 'btn-outline-danger', 'btn-outline-warning', 'btn-outline-info');
        });
        
        // Add appropriate class based on value
        label.classList.add('active');
        switch(e.target.value) {
            case 'Present':
                label.classList.add('btn-outline-success');
                break;
            case 'Absent':
                label.classList.add('btn-outline-danger');
                break;
            case 'Leave':
                label.classList.add('btn-outline-warning');
                break;
            case 'Late':
                label.classList.add('btn-outline-info');
                break;
        }
    }
});
</script>

<?php
include_once('includes/footer.php'); 
?>