<?php
// FILE: admin/award_list_generate.php - FINAL WORKING CODE (Using Robust Settings Logic from Salary Slip)

// Database configuration file
include_once('../config/db.php'); 

// Ensure the connection function is available
if (!function_exists('getDBConnection')) {
    die("Error: getDBConnection() function not found. Please check ../config/db.php path.");
}

$conn = getDBConnection();

$class_id = (int)($_GET['class_id'] ?? 0);
$section_id = (int)($_GET['section_id'] ?? 0);
$term_id = (int)($_GET['term_id'] ?? 0); 

if ($class_id === 0 || $section_id === 0 || $term_id === 0) {
    die("<h1>Error:</h1> Please select Class, Section, and Exam."); 
}

$report_data = []; 
$subject_headers = []; 
$class_details = [];
$settings = []; // <--- $settings array will hold all key-value settings

try {
    // --- 0. Fetch School Settings (Using the robust logic from salary_slip_print.php) ---
    $settings_data = [];
    $settings_query = $conn->query("SELECT setting_key, setting_value FROM settings");
    
    if (!$settings_query) {
         die("<h1>Database Error:</h1> Settings query failed: " . $conn->error);
    }
    
    while ($row = $settings_query->fetch_assoc()) {
        // Trim() function use kiya hai taake keys aur values mein extra space masla na kare
        $settings_data[trim($row['setting_key'])] = trim($row['setting_value']);
    }
    
    // Now map possible keys to a standardized $settings array:
    $settings['institute_name'] = $settings_data['institute.name'] ?? $settings_data['institute name'] ?? $settings_data['institute_name'] ?? "Institute Name Not Set";
    $settings['institute_address'] = $settings_data['institute.address'] ?? $settings_data['institute address'] ?? $settings_data['institute_address'] ?? "Address Not Set";
    $settings['institute_phone'] = $settings_data['institute.phone'] ?? $settings_data['institute phone'] ?? $settings_data['institute_phone'] ?? "Phone Not Set";
    $settings['institute_logo_path'] = $settings_data['institute.logo_path'] ?? $settings_data['institute logo path'] ?? $settings_data['institute_logo_path'] ?? '';
    
    // --- End of Settings Fetch ---


    // --- 1. Fetch Class, Section, and Term Details for Header (Unchanged) ---
    $details_query = $conn->prepare("
        SELECT 
            c.name as class_name, 
            s.name as section_name, 
            e.name as term_name        
        FROM academic_classes c        
        JOIN academic_sections s ON s.id = ?  
        JOIN exams e ON e.id = ?            
        WHERE c.id = ?
    ");
    $details_query->bind_param("iii", $section_id, $term_id, $class_id);
    $details_query->execute();
    $class_details = $details_query->get_result()->fetch_assoc();
    $details_query->close();
    
    if (!$class_details) {
         throw new Exception("Class, Section, or Exam details not found.");
    }

    // --- 2. Fetch All Subjects allocated to this Class/Section (Unchanged) ---
    $subjects_query = $conn->prepare("
        SELECT DISTINCT 
            t2.id, 
            t2.name 
        FROM course_allocation t1  
        JOIN subjects t2 ON t1.subject_id = t2.id  
        WHERE t1.class_id = ? 
        ORDER BY t2.id ASC
    ");
    $subjects_query->bind_param("i", $class_id); 
    $subjects_query->execute();
    $subjects_result = $subjects_query->get_result()->fetch_all(MYSQLI_ASSOC);
    $subjects_query->close();
    
    // Subject headers tayyar karein
    foreach ($subjects_result as $subject) {
        $subject_headers[$subject['id']] = $subject['name'];
    }

    // --- 3. Fetch Student List (✅ MODIFIED: Roll No. added to SELECT and ORDER BY) ---
    $students_query = $conn->prepare("
        SELECT 
            t1.id as student_id, 
            t1.first_name as student_name, 
            t1.father_name,
            t2.roll_no  
        FROM students t1                             
        JOIN student_enrollments t2 ON t1.id = t2.student_id        
        WHERE t2.class_id = ? AND t2.section_id = ? AND t1.status = 'active'
        ORDER BY t2.roll_no ASC, t1.first_name ASC
    ");
    $students_query->bind_param("ii", $class_id, $section_id);
    $students_query->execute();
    $students_result = $students_query->get_result()->fetch_all(MYSQLI_ASSOC);
    $students_query->close();
    
    // Data ko 'report_data' mein structure karein (✅ MODIFIED: roll_no shamil kiya)
    foreach ($students_result as $row) {
        $report_data[$row['student_id']] = [
            'student_name' => $row['student_name'],
            'father_name' => $row['father_name'],
            'roll_no' => $row['roll_no'] // ✅ NEW: Roll No. key added
        ];
    }
    
    // Load the Print View.
    require_once('award_list_print.php');

} catch (Exception $e) {
    die("<h1>Database Error:</h1> " . $e->getMessage());
} finally {
    if (isset($conn) && is_object($conn) && $conn->ping()) {
        $conn->close();
    }
}
?>