<?php
// admin/fees_process.php - COMPLETE CODE (FIXED: Voucher Number Issue - All Students Now Processed)
session_start();
// Include the database connection and utility functions
include_once('../config/db.php');

// Simple redirection helper function
function redirect($page, $message, $type = 'success') {
    $param = ($type === 'success') ? 'msg' : 'err';
    header('Location: ' . $page . '.php?' . $param . '=' . urlencode($message));
    exit;
}

// Function to get the next voucher number - FIXED: Transaction safe
function getNextVoucherNo($conn) {
    $conn->begin_transaction();
    try {
        // 1. Fetch last used number from settings with FOR UPDATE lock
        $q = $conn->query("SELECT setting_value FROM settings WHERE setting_key='fee.last_voucher_no' FOR UPDATE");
        $last_no = $q->fetch_assoc()['setting_value'] ?? 1000; 

        // 2. Increment and update
        $next_no = (int)$last_no + 1;
        $conn->query("UPDATE settings SET setting_value = '{$next_no}' WHERE setting_key='fee.last_voucher_no'");
        
        $conn->commit();
        return $next_no;
    } catch (Exception $e) {
        $conn->rollback();
        return 1000; // Fallback number
    }
}

$conn = getDBConnection();
$action = $_POST['action'] ?? $_GET['action'] ?? '';

// ====================================================================
// ACTION 1: RECORD FEE PAYMENT
// ====================================================================
if ($action === 'record_payment' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $invoice_id = (int)$_POST['invoice_id'];
    $amount_received = (float)$_POST['amount_received'];
    $payment_date = $_POST['payment_date'] ?? date('Y-m-d');
    $payment_method = $_POST['payment_method'] ?? 'Cash';
    
    if ($invoice_id == 0 || $amount_received <= 0) {
        redirect('fees_vouchers', 'Invalid Invoice ID or Amount.', 'danger');
    }

    $conn->begin_transaction();
    try {
        // A. Get Invoice Details
        $inv_q = $conn->query("SELECT net_payable, amount_paid FROM fee_invoices WHERE id = {$invoice_id}");
        $invoice = $inv_q->fetch_assoc();
        
        if (!$invoice) {
            throw new Exception("Invoice not found.");
        }
        
        $net_payable = (float)$invoice['net_payable'];
        $already_paid = (float)$invoice['amount_paid'];
        $remaining_due = $net_payable - $already_paid;
        
        // Check if over-payment is occurring
        if ($amount_received > $remaining_due) {
            throw new Exception("Amount exceeds remaining due: Rs. " . number_format($remaining_due));
        }
        
        // B. Calculate new status and paid amount
        $new_paid = $already_paid + $amount_received;
        $new_status = ($new_paid >= $net_payable) ? 'paid' : 'partial';

        // C. Update Invoice
        $update_q = $conn->prepare("UPDATE fee_invoices SET amount_paid = ?, status = ? WHERE id = ?");
        $update_q->bind_param("dsi", $new_paid, $new_status, $invoice_id);
        if (!$update_q->execute()) {
            throw new Exception("Invoice update failed: " . $update_q->error);
        }

        // D. Insert Payment Record
        $insert_q = $conn->prepare("INSERT INTO fee_payments (invoice_id, amount, payment_date, method) VALUES (?, ?, ?, ?)");
        $insert_q->bind_param("idss", $invoice_id, $amount_received, $payment_date, $payment_method);
        if (!$insert_q->execute()) {
            throw new Exception("Payment record insert failed: " . $insert_q->error);
        }

        $conn->commit();
        redirect('fees_vouchers', 'Payment recorded successfully! Invoice Status: ' . ucfirst($new_status));

    } catch (Exception $e) {
        $conn->rollback();
        redirect('fees_vouchers', 'Payment process failed: ' . $e->getMessage(), 'danger');
    }
}

// ====================================================================
// ACTION 2: DELETE VOUCHER
// ====================================================================
elseif ($action === 'delete_voucher' && isset($_GET['id'])) {
    $invoice_id = (int)$_GET['id'];
    
    if ($invoice_id == 0) {
        redirect('fees_vouchers', 'Invalid Voucher ID.', 'danger');
    }

    $conn->begin_transaction();
    try {
        // A. Delete associated payments
        $conn->query("DELETE FROM fee_payments WHERE invoice_id = {$invoice_id}");
        // B. Delete associated items
        $conn->query("DELETE FROM fee_invoice_items WHERE invoice_id = {$invoice_id}");
        // C. Delete the invoice itself
        $conn->query("DELETE FROM fee_invoices WHERE id = {$invoice_id}");

        $conn->commit();
        redirect('fees_vouchers', 'Voucher and all associated records deleted successfully.');

    } catch (Exception $e) {
        $conn->rollback();
        redirect('fees_vouchers', 'Voucher deletion failed: ' . $e->getMessage(), 'danger');
    }
}

// ====================================================================
// ACTION 3: GENERATE FEE VOUCHERS FROM ASSIGNED FEES (COMPLETELY FIXED)
// ====================================================================
elseif ($action === 'generate_vouchers' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    
    $session_id = (int)$_POST['session_id'];
    $month_year = $_POST['month_year'] ?? ''; // Format: YYYY-MM
    $issue_date = $_POST['issue_date'] ?? date('Y-m-d');
    $due_date = $_POST['due_date'] ?? date('Y-m-d', strtotime('+10 days')); 
    $class_filter = (int)($_POST['class_id'] ?? 0); 
    $section_filter = (int)($_POST['section_id'] ?? 0); 
    
    if ($session_id == 0 || empty($month_year)) {
        redirect('fees_generate', 'Session ID ya Month/Year missing hai.', 'danger');
    }
    
    // Convert YYYY-MM to month name for fee matching
    $month_name = date('F', strtotime($month_year . '-01'));
    $academic_year = date('Y', strtotime($month_year . '-01'));
    
    $generated_count = 0;
    $skipped_count = 0;
    $errors = [];
    
    $conn->begin_transaction();

    try {
        // 1. Student Query Filters - FIXED: Proper joins and conditions
        $class_condition = $class_filter > 0 ? " AND en.class_id = {$class_filter}" : "";
        $section_condition = $section_filter > 0 ? " AND en.section_id = {$section_filter}" : ""; 

        $students_query = "
            SELECT 
                s.id as student_id, 
                en.class_id,
                en.section_id,
                CONCAT(s.first_name, ' ', s.last_name) as student_name,
                s.father_name,
                c.name as class_name,
                sec.name as section_name,
                en.roll_no
            FROM students s
            INNER JOIN student_enrollments en ON s.id = en.student_id AND en.session_id = {$session_id}
            INNER JOIN academic_classes c ON en.class_id = c.id
            LEFT JOIN academic_sections sec ON en.section_id = sec.id
            WHERE s.status = 'active' 
            {$class_condition} {$section_condition}
            ORDER BY c.order_no ASC, en.roll_no ASC
        ";
        
        $students_result = $conn->query($students_query);
        
        if (!$students_result) {
            throw new Exception("Student query failed: " . $conn->error);
        }
        
        if ($students_result->num_rows == 0) {
            throw new Exception("No active students found for the selected criteria.");
        }

        $students = $students_result->fetch_all(MYSQLI_ASSOC);
        
        // 2. Prepared Statements for insertion
        $insert_invoice_stmt = $conn->prepare("
            INSERT INTO fee_invoices 
            (student_id, session_id, voucher_no, month_year, issue_date, due_date, total_amount, net_payable, amount_paid, status) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        
        $insert_item_stmt = $conn->prepare("
            INSERT INTO fee_invoice_items 
            (invoice_id, fee_type, description, amount) 
            VALUES (?, ?, ?, ?)
        ");
        
        foreach ($students as $student) {
            $student_id = $student['student_id'];
            $class_id = $student['class_id'];
            
            // A. Check if voucher already exists for this student and month - FIXED QUERY
            $check_voucher_sql = "
                SELECT id 
                FROM fee_invoices 
                WHERE student_id = ? 
                AND month_year = ?
                AND session_id = ?
            ";
            $check_stmt = $conn->prepare($check_voucher_sql);
            $check_stmt->bind_param("isi", $student_id, $month_year, $session_id);
            $check_stmt->execute();
            $check_result = $check_stmt->get_result();
            
            if ($check_result->num_rows > 0) {
                $skipped_count++;
                $check_stmt->close();
                continue; // Skip if already exists
            }
            $check_stmt->close();
            
            // B. Get assigned fees for this student and SPECIFIC MONTH ONLY - FIXED QUERY
            $fees_query = "
                SELECT DISTINCT fee_type, amount, examination_type
                FROM student_fees 
                WHERE student_id = ? 
                AND academic_year = ?
                AND (month = ? OR is_recurring = 1)
                AND amount > 0
            ";
            $fees_stmt = $conn->prepare($fees_query);
            $fees_stmt->bind_param("iis", $student_id, $academic_year, $month_name);
            $fees_stmt->execute();
            $fees_result = $fees_stmt->get_result();
            $assigned_fees = $fees_result->fetch_all(MYSQLI_ASSOC);
            $fees_stmt->close();
            
            if (empty($assigned_fees)) {
                $skipped_count++;
                continue; // Skip if no fees assigned for this specific month
            }
            
            // C. Calculate total amount and prepare items - FIXED: Remove duplicates
            $total_amount = 0;
            $items_to_insert = [];
            $fee_types_used = [];
            
            foreach ($assigned_fees as $fee) {
                // Avoid duplicate fee types in same voucher
                $fee_key = $fee['fee_type'] . '_' . ($fee['examination_type'] ?? '');
                if (in_array($fee_key, $fee_types_used)) {
                    continue; // Skip duplicate fee type
                }
                $fee_types_used[] = $fee_key;
                
                $description = ucfirst($fee['fee_type']) . " Fee";
                if ($fee['fee_type'] == 'examination' && !empty($fee['examination_type'])) {
                    $description .= " - " . ucfirst(str_replace('_', ' ', $fee['examination_type']));
                }
                
                $total_amount += $fee['amount'];
                $items_to_insert[] = [
                    'fee_type' => $fee['fee_type'],
                    'description' => $description,
                    'amount' => $fee['amount']
                ];
            }
            
            if ($total_amount <= 0) {
                $skipped_count++;
                continue; // Skip if no payable amount
            }

            // D. Insert Invoice - FIXED: Get unique voucher number for each student
            $voucher_no = 'VCHR-' . getNextVoucherNo($conn); 
            $amount_paid = 0.00;
            $status = 'unpaid';
            
            $insert_invoice_stmt->bind_param("iissssddds", 
                $student_id, 
                $session_id, 
                $voucher_no, 
                $month_year, 
                $issue_date, 
                $due_date, 
                $total_amount, // total_amount
                $total_amount, // net_payable (same as total for assigned fees)
                $amount_paid, 
                $status
            );
            
            if (!$insert_invoice_stmt->execute()) {
                $errors[] = "Invoice insert failed for student {$student['student_name']}: " . $insert_invoice_stmt->error;
                continue;
            }
            $invoice_id = $insert_invoice_stmt->insert_id;
            
            // E. Insert Invoice Items from assigned fees
            foreach ($items_to_insert as $item) {
                $insert_item_stmt->bind_param("issd", $invoice_id, $item['fee_type'], $item['description'], $item['amount']);
                if (!$insert_item_stmt->execute()) {
                    $errors[] = "Invoice item insert failed for voucher {$voucher_no}: " . $insert_item_stmt->error;
                }
            }
            
            $generated_count++;
        }
        
        $insert_invoice_stmt->close();
        $insert_item_stmt->close();

        $conn->commit();
        
        // Final Message with details
        $final_message = "✅ <strong>$generated_count vouchers</strong> for <strong>$month_name $academic_year</strong> successfully generated! ";
        $final_message .= "($skipped_count students skipped)";
        
        if (!empty($errors)) {
            $final_message .= "<br>⚠️ Some errors occurred: " . implode(', ', array_slice($errors, 0, 3));
        }
        
        redirect('fees_generate', $final_message, 'success');
        
    } catch (Exception $e) {
        $conn->rollback();
        redirect('fees_generate', 'Voucher generation failed: ' . $e->getMessage(), 'danger');
    }
}

// ====================================================================
// FALLBACK / ERROR
// ====================================================================
else {
    redirect('fees_vouchers', 'Invalid action or request method.', 'danger');
}

$conn->close();
?>
?>