<?php
session_start();
// --- Configuration and Database Connection ---
include_once('../config/db.php'); 

$conn = getDBConnection();
$settings_data = [];
$result = $conn->query("SELECT setting_key, setting_value FROM settings");
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $settings_data[$row['setting_key']] = $row['setting_value'];
    }
    $result->free(); 
}
$settings = array_merge($settings ?? [], $settings_data);


// --- School/Report Details (Dynamic Fetching from settings table) ---
$school_name = $settings['institute.name'] ?? "Mubeshir Computer Academy"; 
$school_address = $settings['institute.address'] ?? "Address Not Set"; 
$school_phone = $settings['institute.phone'] ?? "Phone Not Set"; 
$logo_db_path = $settings['institute.logo_path'] ?? ''; 
$logo_display_url = !empty($logo_db_path) ? '../' . $logo_db_path : '../assets/img/logo.png'; 

$report_heading = "MONTHLY FINANCIAL SUMMARY REPORT"; 

// --- Data Fetching Setup ---
$selected_month_year = $_GET['filter_month_year'] ?? date('Y-m'); 

// *** DATA FETCHING LOGIC GOES HERE ***
// Function to fetch all financial data (Fee, Salary, Expenses, P&L) for the report
function fetch_financial_report_data($conn, $selected_month_year) {
    $data = [
        'period' => date('F Y', strtotime($selected_month_year)),
        'fee_status' => [],
        'salary_status' => [],
        'expenses' => [],
        'total_income' => 0,
        'total_outflow' => 0,
        'net_profit' => 0,
    ];
    
    $month_start = $selected_month_year . '-01';
    $month_end = date('Y-m-t', strtotime($month_start));
    
    // 1. Fetch Fee Collection Status (Class Wise) - CORRECTED QUERY
    $fee_query = "
        SELECT 
            ac.name as Class,
            COUNT(fi.id) as invoice_count,
            SUM(fi.total_amount) as Expected,
            SUM(fi.amount_paid) as Paid,
            SUM(fi.total_amount - fi.amount_paid) as Pending
        FROM fee_invoices fi
        INNER JOIN student_enrollments se ON fi.student_id = se.student_id
        INNER JOIN academic_classes ac ON se.class_id = ac.id
        WHERE fi.month_year = ?
        GROUP BY ac.name
        ORDER BY ac.name
    ";
    
    $stmt = $conn->prepare($fee_query);
    $stmt->bind_param("s", $selected_month_year);
    $stmt->execute();
    $fee_result = $stmt->get_result();
    
    while ($row = $fee_result->fetch_assoc()) {
        $data['fee_status'][] = [
            'Class' => $row['Class'],
            'Expected' => number_format($row['Expected'], 2),
            'Paid' => number_format($row['Paid'], 2),
            'Pending' => number_format($row['Pending'], 2)
        ];
    }
    $stmt->close();
    
    // If no fee data found, show placeholder
    if (empty($data['fee_status'])) {
        $data['fee_status'][] = [
            'Class' => 'No fee data found',
            'Expected' => '0.00',
            'Paid' => '0.00',
            'Pending' => '0.00'
        ];
    }
    
    // 2. Fetch Staff Salary Status
    $salary_query = "
        SELECT 
            t.name as Teacher,
            ts.gross_salary as Expected,
            ts.net_paid as Paid,
            (ts.gross_salary - ts.net_paid) as Pending
        FROM teacher_salary_slips ts
        INNER JOIN teachers t ON ts.teacher_id = t.id
        WHERE DATE_FORMAT(ts.month_year, '%Y-%m') = ?
        ORDER BY t.name
    ";
    
    $stmt = $conn->prepare($salary_query);
    $stmt->bind_param("s", $selected_month_year);
    $stmt->execute();
    $salary_result = $stmt->get_result();
    
    while ($row = $salary_result->fetch_assoc()) {
        $data['salary_status'][] = [
            'Teacher' => $row['Teacher'],
            'Expected' => number_format($row['Expected'], 2),
            'Paid' => number_format($row['Paid'], 2),
            'Pending' => number_format($row['Pending'], 2)
        ];
    }
    $stmt->close();
    
    // If no salary data found, show placeholder
    if (empty($data['salary_status'])) {
        $data['salary_status'][] = [
            'Teacher' => 'No salary data found',
            'Expected' => '0.00',
            'Paid' => '0.00',
            'Pending' => '0.00'
        ];
    }
    
    // 3. Fetch Other Operating Expenses
    $expense_query = "
        SELECT 
            date as Date,
            expense_head as Head,
            description as Description,
            amount as Amount
        FROM other_expenses 
        WHERE date BETWEEN ? AND ?
        ORDER BY date
    ";
    
    $stmt = $conn->prepare($expense_query);
    $stmt->bind_param("ss", $month_start, $month_end);
    $stmt->execute();
    $expense_result = $stmt->get_result();
    
    while ($row = $expense_result->fetch_assoc()) {
        $data['expenses'][] = [
            'Date' => $row['Date'],
            'Head' => $row['Head'],
            'Description' => $row['Description'],
            'Amount' => number_format($row['Amount'], 2)
        ];
    }
    $stmt->close();
    
    // If no expense data found, show placeholder
    if (empty($data['expenses'])) {
        $data['expenses'][] = [
            'Date' => date('Y-m-d'),
            'Head' => 'No expenses found',
            'Description' => '',
            'Amount' => '0.00'
        ];
    }
    
    // 4. Calculate Totals for Profit & Loss
    // Total Income (Fees Paid in the month)
    $income_query = "
        SELECT COALESCE(SUM(amount_paid), 0) as total_income 
        FROM fee_invoices 
        WHERE month_year = ?
    ";
    $stmt = $conn->prepare($income_query);
    $stmt->bind_param("s", $selected_month_year);
    $stmt->execute();
    $income_result = $stmt->get_result();
    $income_row = $income_result->fetch_assoc();
    $data['total_income'] = number_format($income_row['total_income'], 2);
    $stmt->close();
    
    // Total Salary Paid
    $salary_paid_query = "
        SELECT COALESCE(SUM(net_paid), 0) as total_salary 
        FROM teacher_salary_slips 
        WHERE DATE_FORMAT(month_year, '%Y-%m') = ?
    ";
    $stmt = $conn->prepare($salary_paid_query);
    $stmt->bind_param("s", $selected_month_year);
    $stmt->execute();
    $salary_paid_result = $stmt->get_result();
    $salary_paid_row = $salary_paid_result->fetch_assoc();
    $total_salary_paid = $salary_paid_row['total_salary'];
    $stmt->close();
    
    // Total Expenses
    $total_expense_query = "
        SELECT COALESCE(SUM(amount), 0) as total_expenses 
        FROM other_expenses 
        WHERE date BETWEEN ? AND ?
    ";
    $stmt = $conn->prepare($total_expense_query);
    $stmt->bind_param("ss", $month_start, $month_end);
    $stmt->execute();
    $total_expense_result = $stmt->get_result();
    $total_expense_row = $total_expense_result->fetch_assoc();
    $total_expenses = $total_expense_row['total_expenses'];
    $stmt->close();
    
    // Calculate totals
    $total_income_value = $income_row['total_income'];
    $total_outflow_value = $total_salary_paid + $total_expenses;
    $net_profit_value = $total_income_value - $total_outflow_value;
    
    $data['total_outflow'] = number_format($total_outflow_value, 2);
    $data['net_profit'] = number_format($net_profit_value, 2);
    
    return $data;
}

$report_data = fetch_financial_report_data($conn, $selected_month_year);


// Function to generate the full HTML content of the financial report body
function generate_financial_report_html($report_data) {
    // Calculate grand totals for fees
    $fee_expected_total = 0;
    $fee_paid_total = 0;
    $fee_pending_total = 0;
    
    foreach ($report_data['fee_status'] as $fee) {
        if ($fee['Class'] !== 'No fee data found') {
            $fee_expected_total += (float)str_replace(',', '', $fee['Expected']);
            $fee_paid_total += (float)str_replace(',', '', $fee['Paid']);
            $fee_pending_total += (float)str_replace(',', '', $fee['Pending']);
        }
    }
    
    // Calculate grand totals for salary
    $salary_expected_total = 0;
    $salary_paid_total = 0;
    $salary_pending_total = 0;
    
    foreach ($report_data['salary_status'] as $salary) {
        if ($salary['Teacher'] !== 'No salary data found') {
            $salary_expected_total += (float)str_replace(',', '', $salary['Expected']);
            $salary_paid_total += (float)str_replace(',', '', $salary['Paid']);
            $salary_pending_total += (float)str_replace(',', '', $salary['Pending']);
        }
    }
    
    // Calculate total expenses
    $expense_total = 0;
    foreach ($report_data['expenses'] as $expense) {
        if ($expense['Head'] !== 'No expenses found') {
            $expense_total += (float)str_replace(',', '', $expense['Amount']);
        }
    }
    
    ob_start();
    ?>
    <div class="report-content-body">
        
        <h4 style="text-align: center; margin-top: 5px; margin-bottom: 15px; font-size: 11pt;">
            Report Period: <strong><?php echo $report_data['period']; ?></strong>
        </h4>
        
        <h5>1. Fee Collection Status (Class Wise)</h5>
        <table class="table table-bordered table-sm fee-table">
            <thead>
                <tr>
                    <th>Class</th>
                    <th>Total Expected (All-Time)</th>
                    <th>Total Paid Fee (In Range)</th>
                    <th>Total Pending (All-Time)</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($report_data['fee_status'] as $row): ?>
                <tr>
                    <td><?php echo htmlspecialchars($row['Class']); ?></td>
                    <td class="text-center"><?php echo $row['Expected']; ?></td>
                    <td class="text-center"><?php echo $row['Paid']; ?></td>
                    <td class="text-center"><?php echo $row['Pending']; ?></td>
                </tr>
                <?php endforeach; ?>
                <tr>
                    <th>GRAND TOTAL</th>
                    <th class="text-center"><?php echo number_format($fee_expected_total, 2); ?></th>
                    <th class="text-center"><?php echo number_format($fee_paid_total, 2); ?></th>
                    <th class="text-center"><?php echo number_format($fee_pending_total, 2); ?></th>
                </tr>
            </tbody>
        </table>
        
        <h5 style="margin-top: 15px;">2. Staff Salary Status (Teacher Wise)</h5>
        <table class="table table-bordered table-sm salary-table">
             <thead>
                <tr>
                    <th>Teacher Name</th>
                    <th>Total Expected Salary (In Range)</th>
                    <th>Total Paid Salary (In Range)</th>
                    <th>Total Pending Salary (In Range)</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($report_data['salary_status'] as $row): ?>
                <tr>
                    <td><?php echo htmlspecialchars($row['Teacher']); ?></td>
                    <td class="text-center"><?php echo $row['Expected']; ?></td>
                    <td class="text-center"><?php echo $row['Paid']; ?></td>
                    <td class="text-center"><?php echo $row['Pending']; ?></td>
                </tr>
                <?php endforeach; ?>
                <tr>
                    <th>GRAND TOTAL</th>
                    <th class="text-center"><?php echo number_format($salary_expected_total, 2); ?></th>
                    <th class="text-center"><?php echo number_format($salary_paid_total, 2); ?></th>
                    <th class="text-center"><?php echo number_format($salary_pending_total, 2); ?></th>
                </tr>
            </tbody>
        </table>

        <h5 style="margin-top: 15px;">3. Other Operating Expenses (Breakdown)</h5>
        <table class="table table-bordered table-sm expense-table">
            <thead>
                <tr>
                    <th>Date</th>
                    <th>Expense Head</th>
                    <th>Description</th>
                    <th>Amount Paid</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($report_data['expenses'] as $row): ?>
                <tr>
                    <td><?php echo $row['Date']; ?></td>
                    <td><?php echo htmlspecialchars($row['Head']); ?></td>
                    <td><?php echo htmlspecialchars($row['Description']); ?></td>
                    <td class="text-right"><?php echo $row['Amount']; ?></td>
                </tr>
                <?php endforeach; ?>
                <tr>
                    <th colspan="3" class="text-right">TOTAL OPERATING EXPENSES</th>
                    <th class="text-right"><?php echo number_format($expense_total, 2); ?></th>
                </tr>
            </tbody>
        </table>

        <h5 style="margin-top: 15px;">4. Profit & Loss Statement Summary</h5>
        <div class="p-l-summary">
            <p>Total Fees Collected (INCOME): <span>Rs. <?php echo $report_data['total_income']; ?></span></p>
            <p>Total Outflow (Salaries Paid + Expenses): <span>Rs. <?php echo $report_data['total_outflow']; ?></span></p>
            <hr>
            <p>NET PROFIT/LOSS: 
                <span>
                    <?php 
                    $net_profit_value = (float)str_replace(',', '', $report_data['net_profit']);
                    if ($net_profit_value >= 0) {
                        echo "<strong style='color: green;'>PROFIT: Rs. " . $report_data['net_profit'] . "</strong>";
                    } else {
                        echo "<strong style='color: red;'>LOSS: Rs. " . str_replace('-', '', $report_data['net_profit']) . "</strong>";
                    }
                    ?>
                </span>
            </p>
        </div>
        
    </div>
    <?php
    return ob_get_clean();
}


// --- Check for Print Mode ---
$is_print_mode = isset($_GET['mode']) && $_GET['mode'] === 'print';


// =========================================================
//            START: Print Mode Rendering
// =========================================================

if ($is_print_mode) {
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Financial Report - <?php echo $report_data['period']; ?></title>
    <link rel="stylesheet" href="../assets/css/bootstrap.min.css"> 
    <style>
        /* === FINAL CSS FIX FOR PRINT FIT & HEADER STYLES === */
        @media print {
            /* Browser's default header/footer (Date, Time, Link, Page 1/1) remove karne ki koshish */
            @page {
                size: portrait; /* Financial reports ke liye Portrait rakha hai */
                /* Margin bilkul zero kar diya taaki browser header/footer ki jagah khatam ho */
                margin: 0; 
            }
            body, .report-container {
                padding: 0; 
                margin: 0; 
                width: 100%;
                font-size: 10pt; 
            }
            /* Table break settings */
            thead { display: table-header-group; }
            table { page-break-inside: auto; }
            tr { page-break-inside: avoid; page-break-after: auto; }
        }
        .report-container { 
            /* Content ko thoda sa padding denge, taaki edge se na chipke */
            padding: 15px 15px; 
            width: 100%;
            box-sizing: border-box; 
        }
        
        /* === LOGO CENTER aur HEADER Styles (Aapki request ke mutabiq) === */
        .print-header { 
            max-width: 100%; 
            margin: 0 auto 10px auto; 
            padding: 5px 0 10px 0; 
            text-align: center; /* Saari cheezein center hongi */
            border-bottom: 2px solid #333; /* Bottom line add kiya */
        }
        /* Logo aur Info containers block elements hain jo center mein aayenge */
        .print-header .logo-container,
        .print-header .info-container {
            display: block; 
            margin: 0 auto; 
            width: auto;
            float: none; 
        }
        .print-header img { 
            height: 75px; /* Logo height */
            width: auto;
            margin-bottom: 5px;
        }
        .print-header h3 { 
            margin: 0; 
            font-size: 14pt; 
            font-weight: bold; 
        }
        .print-header .address-line { 
            margin: 0; 
            font-size: 9pt; 
            line-height: 1.2; 
        }
        
        /* Table and Report Body Styles */
        .report-content-body {
            max-width: 800px;
            margin: 0 auto;
        }
        table { width: 100%; margin: 0; border-collapse: collapse; margin-top: 5px; }
        th, td { 
            border: 1px solid #ddd; padding: 4px 6px; 
            text-align: left; font-size: 9pt; line-height: 1.3; 
        }
        th { background-color: #f2f2f2; font-weight: bold; text-align: center; }
        h5 { font-size: 10pt; font-weight: bold; margin-top: 15px; margin-bottom: 5px; }
        .p-l-summary p { margin: 0; font-size: 10pt; }
        .p-l-summary span { float: right; font-weight: bold; }

    </style>
</head>
<body>
    <div class="report-container">
        <div class="print-header">
            <div class="logo-container">
                <img src="<?php echo htmlspecialchars($logo_display_url); ?>" alt="School Logo" onerror="this.style.display='none'">
            </div>

            <div class="info-container">
                <h3><?php echo htmlspecialchars($school_name); ?></h3>
                
                <p class="address-line"><?php echo htmlspecialchars($school_address); ?></p>
                
                <p class="address-line">Phone: <?php echo htmlspecialchars($school_phone); ?></p>
                
                <h4 style="margin-top: 5px; font-size: 12pt;"><?php echo $report_heading; ?></h4>
            </div>
            
            <div style="clear: both;"></div> 
        </div>
        
        <?php echo generate_financial_report_html($report_data); ?>
        
        <div style="margin-top: 40px; text-align: center; font-size: 9pt; width: 100%;">
            <table style="width: 100%; margin: 0; border: none; border-collapse: separate;">
                <tr style="border: none;">
                    <td style="width: 50%; border: none; padding: 0; text-align: center;">
                        <div style="border-top: 1px solid #000; margin-top: 10px; padding-top: 2px; display: inline-block; width: 60%;">
                            Accountant Signature
                        </div>
                    </td>
                    <td style="width: 50%; border: none; padding: 0; text-align: center;">
                        <div style="border-top: 1px solid #000; margin-top: 10px; padding-top: 2px; display: inline-block; width: 60%;">
                            Principal Signature
                        </div>
                    </td>
                </tr>
            </table>
        </div>
        
    </div>
    <script>
        window.onload = function() {
            // Print dialog automatically open ho jayega
            window.print();
        };
    </script>
</body>
</html>
<?php
    exit; // Print mode mein iske aage ka code nahi chalega
}


// =========================================================
//            START: Screen Display Mode Rendering
// =========================================================

// Is hisse mein sidebar aur main navigation header/footer include honge.
// Sidebar ko remove karne ke liye bas yahaan 'includes/header.php' aur 'includes/footer.php' 
// ko include na karein, ya phir aapka header.php file sidebar dikhane ka zimmedaar hai.

// Assuming includes/header.php exists and includes the sidebar/menu
include_once('includes/header.php'); 
?>

<div class="container-fluid">
    <div class="row">
        <main role="main" class="col-md-9 ml-sm-auto col-lg-10 px-4">
            
            <h1 class="h2 mb-4">Financial Summary Report</h1>

            <?php // Delete successful message (from screenshot image_da4abf.png)
            if (isset($_GET['status']) && $_GET['status'] == 'deleted' && isset($_GET['id'])): ?>
                <div class="alert alert-success">Expense with ID: <?php echo htmlspecialchars($_GET['id']); ?> has been deleted successfully.</div>
            <?php endif; ?>

            <div class="card p-4 shadow-sm">
                <div class="d-flex justify-content-between align-items-center mb-3">
                    <h4 class="mb-0"><i class="fas fa-chart-line"></i> Financial Summary Report</h4>
                    <div>
                        <a href="add_expense.php" class="btn btn-warning btn-sm"><i class="fas fa-plus"></i> Add Extra Expense</a>
                        <?php 
                            // Print URL for the print mode
                            $print_url = "financial_summary_report.php?mode=print&filter_month_year={$selected_month_year}";
                        ?>
                        <a href="<?php echo $print_url; ?>" target="_blank" class="btn btn-primary btn-sm ml-2"><i class="fas fa-print"></i> Print Report</a>
                    </div>
                </div>

                <form method="GET" action="financial_summary_report.php" class="form-inline mb-4">
                    <div class="form-group mr-3">
                        <label class="mr-2">Select Month:</label>
                        <input type="month" name="filter_month_year" class="form-control" value="<?php echo htmlspecialchars($selected_month_year); ?>" required>
                    </div>
                    <button type="submit" class="btn btn-success"><i class="fas fa-search"></i> Generate Report</button>
                </form>

                <?php echo generate_financial_report_html($report_data); ?>

            </div>
        </main>
    </div>
</div>

<?php
// Assuming includes/footer.php exists
include_once('includes/footer.php'); 
?>