<?php
// admin/login.php - Finalized Login Page with Hashing Verification

// Ensure session is started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Redirect if already logged in
if (isset($_SESSION['is_admin_logged_in']) && $_SESSION['is_admin_logged_in'] === true) {
    header("Location: index.php");
    exit();
}

// Include database connection function
// Checks if installation is complete by looking for db.php
if (!file_exists('../config/db.php')) {
    // If config is missing, redirect to installation
    header("Location: ../install/index.php"); 
    exit();
}
include_once('../config/db.php');

$error = '';
$username_or_email = '';

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    
    // Sanitize user inputs
    $username_or_email = trim($_POST['username'] ?? '');
    $password_input = $_POST['password'] ?? '';

    if (empty($username_or_email) || empty($password_input)) {
        $error = "Please enter both username/email and password.";
    } else {
        
        // 1. Database Connection
        $conn = getDBConnection();

        // 2. Prepare SQL Statement: Check for user by 'username' OR 'email'
        // 'password_hash' column is used to retrieve the stored hash
        $stmt = $conn->prepare("SELECT id, name, username, password_hash, role_id FROM users WHERE (username = ? OR email = ?) AND status = 'active'");
        $stmt->bind_param("ss", $username_or_email, $username_or_email);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows === 1) {
            $user = $result->fetch_assoc();
            
            // 3. Verify Password: Checks the input password against the stored HASH
            if (password_verify($password_input, $user['password_hash'])) {
                
                // Success: Set session variables
                $_SESSION['user_id'] = $user['id'];
                $_SESSION['username'] = $user['username'];
                $_SESSION['name'] = $user['name'];
                $_SESSION['role_id'] = $user['role_id'];
                $_SESSION['is_admin_logged_in'] = true;

                $stmt->close();
                $conn->close();
                
                // Redirect to Dashboard
                header("Location: dashboard.php"); 
                exit();
            } else {
                // Password incorrect
                $error = "Invalid username/email or password.";
            }
        } else {
            // User not found
            $error = "Invalid username/email or password.";
        }

        $stmt->close();
        $conn->close();
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Admin Login - School Management</title>
    <style>
        body { font-family: Arial, sans-serif; background-color: #f4f4f4; display: flex; justify-content: center; align-items: center; min-height: 100vh; margin: 0; }
        .login-container { background-color: #fff; padding: 30px; border-radius: 8px; box-shadow: 0 4px 8px rgba(0, 0, 0, 0.1); width: 350px; }
        h2 { text-align: center; margin-bottom: 20px; color: #333; }
        .form-group { margin-bottom: 15px; }
        .form-group label { display: block; margin-bottom: 5px; font-weight: bold; }
        .form-group input { width: 100%; padding: 10px; border: 1px solid #ddd; border-radius: 4px; box-sizing: border-box; }
        .error-message { color: #dc3545; text-align: center; margin-bottom: 15px; }
        .btn-primary { width: 100%; padding: 10px; background-color: #007bff; border: none; color: white; border-radius: 4px; cursor: pointer; font-size: 16px; }
        .btn-primary:hover { background-color: #0056b3; }
    </style>
</head>
<body>

<div class="login-container">
    <h2>Admin Login</h2>

    <?php if ($error): ?>
        <p class="error-message"><?php echo htmlspecialchars($error); ?></p>
    <?php endif; ?>

    <form method="POST" action="login.php">
        <div class="form-group">
            <label for="username">Username / Email:</label>
            <input type="text" id="username" name="username" required 
                   value="<?php echo htmlspecialchars($username_or_email); ?>">
        </div>
        <div class="form-group">
            <label for="password">Password:</label>
            <input type="password" id="password" name="password" required>
        </div>
        <div class="form-group">
            <button type="submit" class="btn-primary">Login</button>
        </div>
    </form>
</div>

</body>
</html>