<?php
// admin/parent_management.php - Final Complete Code with Robust Name Handling

// 1. Start Session & DB Connection
session_start();
// Zaroori: Make sure this path is correct
include_once('../config/db.php'); 
$conn = getDBConnection(); 
$message = ''; 
$error = ''; 

if (isset($_GET['msg'])) $message = htmlspecialchars($_GET['msg']);
if (isset($_GET['err'])) $error = htmlspecialchars($_GET['err']);

// 3. Authorization Check (Admin Only)
$user_role_id = $_SESSION['role_id'] ?? 0;
if (!isset($user_role_id) || $user_role_id != 1) { 
    header('Location: dashboard.php?err=' . urlencode('Access Denied.'));
    exit;
}

// --------------------------------------------------------------------------------
// --- Identify Parent Role ID & Fetch Dropdown Data ---
// --------------------------------------------------------------------------------
$parent_role_id = 0;
$roles_result = $conn->query("SELECT id FROM user_roles WHERE LOWER(name) LIKE '%parent%' LIMIT 1");
if ($roles_result && $row = $roles_result->fetch_assoc()) {
    $parent_role_id = (int)$row['id'];
}
if ($parent_role_id === 0) { $error .= "Configuration Error: 'Parent' role not found in the database. "; }

// Fetch Classes
$classes = [];
$class_result = $conn->query("SELECT id, name FROM academic_classes ORDER BY id");
if ($class_result) { while ($row = $class_result->fetch_assoc()) { $classes[] = $row; } }

// Fetch Sections
$sections = [];
$section_result = $conn->query("SELECT id, name FROM academic_sections ORDER BY id");
if ($section_result) { while ($row = $section_result->fetch_assoc()) { $sections[] = $row; } }


// --------------------------------------------------------------------------------
// --- INITIALIZE FILTER VARIABLES ---
// --------------------------------------------------------------------------------
// Filter IDs ko URL (GET) se uthayenge
$class_filter_id = (int)($_GET['class_filter'] ?? 0);
$section_filter_id = (int)($_GET['section_filter'] ?? 0);
$all_students = [];
$show_filtered_students = false;

// Edit mode variables
$edit_mode = false;
$editing_parent_id = 0;
$editing_parent_data = [];
$currently_linked_students = [];

// --------------------------------------------------------------------------------
// --- CHECK IF WE ARE IN EDIT MODE ---
// --------------------------------------------------------------------------------
if (isset($_GET['action']) && $_GET['action'] == 'edit' && isset($_GET['id'])) {
    $edit_mode = true;
    $editing_parent_id = (int)$_GET['id'];
    
    // Fetch parent basic info
    $stmt_parent = $conn->prepare("SELECT id, username, name, email, status FROM users WHERE id = ? AND role_id = ?");
    $stmt_parent->bind_param("ii", $editing_parent_id, $parent_role_id);
    $stmt_parent->execute();
    $result_parent = $stmt_parent->get_result();
    
    if ($result_parent->num_rows > 0) {
        $editing_parent_data = $result_parent->fetch_assoc();
        
        // Fetch currently linked students
        $stmt_linked = $conn->prepare("SELECT student_id FROM parent_student_link WHERE parent_user_id = ?");
        $stmt_linked->bind_param("i", $editing_parent_id);
        $stmt_linked->execute();
        $result_linked = $stmt_linked->get_result();
        
        while ($row = $result_linked->fetch_assoc()) {
            $currently_linked_students[] = $row['student_id'];
        }
        $stmt_linked->close();
    } else {
        $edit_mode = false;
        $error = "Parent not found or invalid ID.";
    }
    $stmt_parent->close();
}

// --------------------------------------------------------------------------------
// --- PROCESS PARENT ADDITION (POST) ---
// --------------------------------------------------------------------------------
if ($_SERVER["REQUEST_METHOD"] == "POST" && ($_POST['action'] ?? '') == 'add_parent') {
    $username = trim($_POST['username'] ?? '');
    $password = $_POST['password'] ?? '';
    $user_name = trim($_POST['user_name'] ?? '');
    $email = trim($_POST['email'] ?? NULL);
    $selected_students = $_POST['student_id'] ?? []; 
    
    if (empty($username) || empty($password) || empty($user_name)) {
        $error = "Please fill in Username, Password, and Name.";
    } else if (empty($selected_students)) {
        $error = "Parent user ke liye kam az kam ek bachcha select karna zaroori hai.";
    } else {
        try {
            $password_hash = password_hash($password, PASSWORD_DEFAULT);
            $conn->begin_transaction();

            // 1. Insert into users table
            $stmt = $conn->prepare("INSERT INTO users (username, password_hash, role_id, name, email, status) VALUES (?, ?, ?, ?, ?, 'active')");
            $stmt->bind_param("ssiss", $username, $password_hash, $parent_role_id, $user_name, $email);
            
            if ($stmt->execute()) {
                $new_parent_id = $conn->insert_id;
                $stmt->close();
                
                // 2. Insert links into parent_student_link
                $link_success = true;
                $stmt_link = $conn->prepare("INSERT INTO parent_student_link (parent_user_id, student_id) VALUES (?, ?)");
                
                foreach ($selected_students as $student_id) {
                    $student_id_int = (int)$student_id;
                    $stmt_link->bind_param("ii", $new_parent_id, $student_id_int);
                    if (!$stmt_link->execute()) { $link_success = false; break; }
                }
                $stmt_link->close();
                
                if ($link_success) {
                    $conn->commit();
                    $message = "Parent User '{$username}' successfully added and linked to " . count($selected_students) . " students!";
                } else {
                    $conn->rollback();
                    $error = "Error linking students. Transaction rolled back.";
                }
            } else {
                $conn->rollback();
                $error = "Error adding parent user: " . $conn->error;
                if ($conn->errno == 1062) $error = "Error: Username ya Email already exists.";
            }
        } catch (Exception $e) {
            if (isset($conn) && $conn->in_transaction) $conn->rollback();
            $error = "Error: " . $e->getMessage();
        }
    }
    // Redirect to clear POST data and show message
    header("Location: parent_management.php?msg=" . urlencode($message) . "&err=" . urlencode($error));
    exit;
}

// --------------------------------------------------------------------------------
// --- PROCESS PARENT UPDATE (POST) ---
// --------------------------------------------------------------------------------
if ($_SERVER["REQUEST_METHOD"] == "POST" && ($_POST['action'] ?? '') == 'update_parent') {
    $parent_id = (int)($_POST['parent_id'] ?? 0);
    $username = trim($_POST['username'] ?? '');
    $password = $_POST['password'] ?? '';
    $user_name = trim($_POST['user_name'] ?? '');
    $email = trim($_POST['email'] ?? NULL);
    $selected_students = $_POST['student_id'] ?? []; 
    
    if (empty($username) || empty($user_name)) {
        $error = "Please fill in Username and Name.";
    } else if (empty($selected_students)) {
        $error = "Parent user ke liye kam az kam ek bachcha select karna zaroori hai.";
    } else {
        try {
            $conn->begin_transaction();

            // 1. Update users table
            if (!empty($password)) {
                // Password update ke saath
                $password_hash = password_hash($password, PASSWORD_DEFAULT);
                $stmt = $conn->prepare("UPDATE users SET username = ?, password_hash = ?, name = ?, email = ? WHERE id = ? AND role_id = ?");
                $stmt->bind_param("ssssii", $username, $password_hash, $user_name, $email, $parent_id, $parent_role_id);
            } else {
                // Password update ke bagair
                $stmt = $conn->prepare("UPDATE users SET username = ?, name = ?, email = ? WHERE id = ? AND role_id = ?");
                $stmt->bind_param("sssii", $username, $user_name, $email, $parent_id, $parent_role_id);
            }
            
            if ($stmt->execute()) {
                $stmt->close();
                
                // 2. Delete existing links
                $stmt_delete = $conn->prepare("DELETE FROM parent_student_link WHERE parent_user_id = ?");
                $stmt_delete->bind_param("i", $parent_id);
                $stmt_delete->execute();
                $stmt_delete->close();
                
                // 3. Insert new links
                $link_success = true;
                $stmt_link = $conn->prepare("INSERT INTO parent_student_link (parent_user_id, student_id) VALUES (?, ?)");
                
                foreach ($selected_students as $student_id) {
                    $student_id_int = (int)$student_id;
                    $stmt_link->bind_param("ii", $parent_id, $student_id_int);
                    if (!$stmt_link->execute()) { $link_success = false; break; }
                }
                $stmt_link->close();
                
                if ($link_success) {
                    $conn->commit();
                    $message = "Parent User '{$username}' successfully updated and linked to " . count($selected_students) . " students!";
                } else {
                    $conn->rollback();
                    $error = "Error updating student links. Transaction rolled back.";
                }
            } else {
                $conn->rollback();
                $error = "Error updating parent user: " . $conn->error;
                if ($conn->errno == 1062) $error = "Error: Username ya Email already exists.";
            }
        } catch (Exception $e) {
            if (isset($conn) && $conn->in_transaction) $conn->rollback();
            $error = "Error: " . $e->getMessage();
        }
    }
    // Redirect to clear POST data and show message
    header("Location: parent_management.php?msg=" . urlencode($message) . "&err=" . urlencode($error));
    exit;
}

// --------------------------------------------------------------------------------
// --- DELETE ACTION (GET) ---
// --------------------------------------------------------------------------------
if (isset($_GET['action']) && $_GET['action'] == 'delete' && isset($_GET['id'])) {
    $user_id = (int)$_GET['id'];
    // ... (Your DELETE logic here) ...
    try {
        $conn->begin_transaction();
        // 1. Delete links first
        $stmt_link = $conn->prepare("DELETE FROM parent_student_link WHERE parent_user_id = ?"); 
        $stmt_link->bind_param("i", $user_id);
        $stmt_link->execute();
        $stmt_link->close();

        // 2. Delete user 
        $stmt = $conn->prepare("DELETE FROM users WHERE id = ? AND role_id = ?"); 
        $stmt->bind_param("ii", $user_id, $parent_role_id);
        if ($stmt->execute() && $stmt->affected_rows > 0) {
            $conn->commit();
            $message = "Parent ID {$user_id} successfully deleted, aur saare links hata diye gaye!";
        } else {
            $conn->rollback();
            $error = "Deletion failed. Parent user not found or role mismatch.";
        }
        $stmt->close();
        
    } catch (Exception $e) {
        if (isset($conn) && $conn->in_transaction) {
            $conn->rollback();
        }
        $error = "Action Error: " . $e->getMessage();
    }
    
    header("Location: parent_management.php?msg=" . urlencode($message) . "&err=" . urlencode($error));
    exit;
}

// --------------------------------------------------------------------------------
// --- SERVER-SIDE STUDENT FETCH (CRITICAL FIX HERE) ---
// --------------------------------------------------------------------------------
if ($class_filter_id > 0 && $section_filter_id > 0) {
    $show_filtered_students = true;
    
    // *** FIX: Sirf first_name use kiya gaya hai aur COALESCE lagaya gaya hai ***
    $query_students = "
        SELECT 
            s.id AS user_id,                                 
            COALESCE(s.first_name, '') AS full_name,  
            se.roll_no                                 
        FROM students s
        
        JOIN student_enrollments se ON s.id = se.student_id   
        
        WHERE s.status = 'active'
          AND se.class_id = ?
          AND se.section_id = ?
        ORDER BY se.roll_no"; 
        
    $stmt_students = $conn->prepare($query_students);
    $stmt_students->bind_param("ii", $class_filter_id, $section_filter_id);
    
    if ($stmt_students->execute()) {
        $result_students = $stmt_students->get_result();
        while ($row = $result_students->fetch_assoc()) { $all_students[] = $row; }
    } else { $error .= " | Student Fetch SQL Error: " . htmlspecialchars($conn->error); }
    $stmt_students->close();
}


// --------------------------------------------------------------------------------
// --- Fetch All Parents (for the bottom table) ---
// --------------------------------------------------------------------------------
$parents_data = [];
if ($parent_role_id > 0) {
    $query_parents = "
        SELECT u.id, u.username, u.name, u.email, u.status, COUNT(psl.student_id) AS linked_students_count
        FROM users u
        LEFT JOIN parent_student_link psl ON u.id = psl.parent_user_id
        WHERE u.role_id = ? GROUP BY u.id ORDER BY u.name";
        
    $stmt_parents = $conn->prepare($query_parents);
    $stmt_parents->bind_param("i", $parent_role_id);
    $stmt_parents->execute();
    $result_parents = $stmt_parents->get_result();

    if ($result_parents) { while ($row = $result_parents->fetch_assoc()) { $parents_data[] = $row; } }
    $stmt_parents->close();
}


// --------------------------------------------------------------------------------
// --- HTML Output Starts Here ---
// --------------------------------------------------------------------------------
include('includes/header.php'); 
?>

<h1 class="mb-4">Parent Management <i class="fas fa-user-tie"></i></h1>

<?php if (!empty($message)): ?>
    <div class="alert alert-success"><?php echo $message; ?></div>
<?php endif; ?>
<?php if (!empty($error)): ?>
    <div class="alert alert-danger"><?php echo $error; ?></div>
<?php endif; ?>

<ul class="nav nav-tabs mb-3">
    <li class="nav-item">
        <a class="nav-link" href="user_management.php"><i class="fas fa-user-shield"></i> Core User Management</a>
    </li>
    <li class="nav-item">
        <a class="nav-link active" href="parent_management.php"><i class="fas fa-user-tie"></i> Parent Management</a>
    </li>
</ul>

<div class="card mb-4 shadow-sm">
    <div class="card-header <?php echo $edit_mode ? 'bg-warning' : 'bg-success'; ?> text-white">
        <h5>
            <i class="fas <?php echo $edit_mode ? 'fa-edit' : 'fa-user-plus'; ?>"></i> 
            <?php echo $edit_mode ? 'Edit Parent User' : 'Add New Parent User'; ?>
        </h5>
    </div>
    <div class="card-body">
        
        <form id="parent_form" action="parent_management.php" method="POST">
            <input type="hidden" name="action" value="<?php echo $edit_mode ? 'update_parent' : 'add_parent'; ?>">
            <?php if ($edit_mode): ?>
                <input type="hidden" name="parent_id" value="<?php echo $editing_parent_id; ?>">
            <?php endif; ?>
            
            <div class="form-row">
                <div class="form-group col-md-4">
                    <label for="username">Username (Login ID)</label>
                    <input type="text" class="form-control" id="username" name="username" required 
                           value="<?php echo $edit_mode ? htmlspecialchars($editing_parent_data['username']) : htmlspecialchars($_POST['username'] ?? $_GET['username'] ?? ''); ?>">
                </div>
                <div class="form-group col-md-4">
                    <label for="password">Password <?php if ($edit_mode): ?><small class="text-muted">(Leave blank to keep current)</small><?php endif; ?></label>
                    <input type="password" class="form-control" id="password" name="password" <?php echo !$edit_mode ? 'required' : ''; ?>>
                </div>
                <div class="form-group col-md-4">
                    <label for="user_name">Full Name</label>
                    <input type="text" class="form-control" id="user_name" name="user_name" required 
                           value="<?php echo $edit_mode ? htmlspecialchars($editing_parent_data['name']) : htmlspecialchars($_POST['user_name'] ?? $_GET['user_name'] ?? ''); ?>">
                </div>
            </div>
            <div class="form-group">
                <label for="email">Email (Optional)</label>
                <input type="email" class="form-control" id="email" name="email" 
                       value="<?php echo $edit_mode ? htmlspecialchars($editing_parent_data['email'] ?? '') : htmlspecialchars($_POST['email'] ?? $_GET['email'] ?? ''); ?>">
            </div>
            
            <input type="hidden" name="class_filter" value="<?php echo $class_filter_id; ?>">
            <input type="hidden" name="section_filter" value="<?php echo $section_filter_id; ?>">

            <div id="parent-assignment-section" style="border: 1px dashed #28a745; padding: 15px; margin-top: 15px;">
                <h6 class="text-success"><i class="fas fa-filter"></i> Step 1: Filter Students by Class/Section</h6>
                
                <div class="form-row">
                    <div class="form-group col-md-6">
                        <label for="class_filter_ui">Select Class</label>
                        <select class="form-control" id="class_filter_ui" required 
                                onchange="updateFilterUrl()">
                            <option value="">-- Select Class --</option>
                            <?php foreach ($classes as $class): ?>
                                <option value="<?php echo $class['id']; ?>" <?php echo ($class_filter_id == $class['id']) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($class['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="form-group col-md-6">
                        <label for="section_filter_ui">Select Section</label>
                        <select class="form-control" id="section_filter_ui" required 
                                onchange="updateFilterUrl()">
                            <option value="">-- Select Section --</option>
                            <?php foreach ($sections as $section): ?>
                                <option value="<?php echo $section['id']; ?>" <?php echo ($section_filter_id == $section['id']) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($section['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>

                <h6 class="text-success mt-3"><i class="fas fa-user-graduate"></i> Step 2: Select Students to Assign</h6>
                <div id="student_multiselect_container">
                    <?php if (!$show_filtered_students): ?>
                        <div class="alert alert-info">Please select both **Class** and **Section** above.</div>
                    <?php elseif (empty($all_students)): ?>
                        <select multiple class="form-control" name="student_id[]" style="min-height: 200px;" disabled>
                             <option value="" disabled selected>-- No active students found in this Class/Section. --</option>
                        </select>
                        <small class="form-text text-muted">No students found matching the selected criteria. Please check if students are enrolled and active.</small>
                    <?php else: ?>
                        <select multiple class="form-control" id="student_id_multiselect" name="student_id[]" style="min-height: 200px;" required>
                            <?php foreach ($all_students as $student): ?>
                                <?php $full_name = htmlspecialchars($student['full_name']); ?>
                                <?php $roll_no = htmlspecialchars($student['roll_no'] ?? 'N/A'); ?>
                                <option value="<?php echo $student['user_id']; ?>"
                                    <?php echo ($edit_mode && in_array($student['user_id'], $currently_linked_students)) ? 'selected' : ''; ?>>
                                    <?php echo "Roll No. {$roll_no} - {$full_name}"; ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                        <small class="form-text text-muted">Hold down Ctrl (Windows) or Command (Mac) to select multiple students.</small>
                    <?php endif; ?>
                </div>
            </div>
            
            <div class="form-group mt-3">
                <button type="submit" class="btn <?php echo $edit_mode ? 'btn-warning' : 'btn-success'; ?>" <?php echo empty($all_students) ? 'disabled' : ''; ?>>
                    <i class="fas <?php echo $edit_mode ? 'fa-sync-alt' : 'fa-save'; ?>"></i> 
                    <?php echo $edit_mode ? 'Update Parent' : 'Create Parent'; ?>
                </button>
                
                <?php if ($edit_mode): ?>
                    <a href="parent_management.php" class="btn btn-secondary">
                        <i class="fas fa-times"></i> Cancel Edit
                    </a>
                <?php endif; ?>
            </div>
        </form>
    </div>
</div>

<div class="card shadow-sm">
    <div class="card-header">
        <h5>All Parent Users</h5>
    </div>
    <div class="card-body">
        <?php if (empty($parents_data)): ?>
            <div class="alert alert-info">No parent users found.</div>
        <?php else: ?>
            <div class="table-responsive">
                <table class="table table-striped table-bordered table-hover">
                    <thead class="thead-dark">
                        <tr>
                            <th>ID</th>
                            <th>Username</th>
                            <th>Full Name</th>
                            <th>Email</th>
                            <th>Linked Students</th>
                            <th>Status</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($parents_data as $parent): ?>
                            <tr>
                                <td><?php echo $parent['id']; ?></td>
                                <td><?php echo htmlspecialchars($parent['username']); ?></td>
                                <td><?php echo htmlspecialchars($parent['name']); ?></td>
                                <td><?php echo htmlspecialchars($parent['email'] ?? 'N/A'); ?></td>
                                <td><span class="badge badge-primary"><?php echo $parent['linked_students_count']; ?></span></td>
                                <td>
                                    <?php 
                                        $status_class = ($parent['status'] == 'active') ? 'badge-success' : 'badge-danger';
                                        echo "<span class='badge {$status_class}'>" . htmlspecialchars(ucfirst($parent['status'])) . "</span>";
                                    ?>
                                </td>
                                <td>
                                    <a href="parent_management.php?action=edit&id=<?php echo $parent['id']; ?>" 
                                       class="btn btn-sm btn-warning mr-1" 
                                       title="Edit Parent">
                                        <i class="fas fa-edit"></i> Edit
                                    </a>
                                    <a href="parent_management.php?action=delete&id=<?php echo $parent['id']; ?>" 
                                       class="btn btn-sm btn-danger" 
                                       title="Delete Parent and Links"
                                       onclick="return confirm('Are you sure you want to permanently delete parent <?php echo htmlspecialchars($parent['username']); ?>? All student links will be removed.');">
                                        <i class="fas fa-trash"></i> Delete
                                    </a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>
</div>

<script>
function updateFilterUrl() {
    const classFilter = document.getElementById('class_filter_ui').value;
    const sectionFilter = document.getElementById('section_filter_ui').value;
    const username = document.getElementById('username').value;
    const userName = document.getElementById('user_name').value;
    const email = document.getElementById('email').value;
    
    let url = `parent_management.php?class_filter=${classFilter}&section_filter=${sectionFilter}`;
    
    <?php if ($edit_mode): ?>
        url += `&action=edit&id=<?php echo $editing_parent_id; ?>`;
    <?php else: ?>
        if (username) url += `&username=${encodeURIComponent(username)}`;
        if (userName) url += `&user_name=${encodeURIComponent(userName)}`;
        if (email) url += `&email=${encodeURIComponent(email)}`;
    <?php endif; ?>
    
    window.location.href = url;
}
</script>

<?php
//$conn->close();
include('includes/footer.php');
?>