<?php
// admin/print_result_card.php - Student Result Card Print View (FINAL PROFESSIONAL VERSION - V12 with Class Position and Session Filter)

session_start();
ob_start(); 
include_once('../config/db.php');
include_once('includes/header.php'); 

$conn = getDBConnection();

// --- START: Settings Fetch Logic ---
$settings = [];
$settings_result = $conn->query("SELECT setting_key, setting_value FROM settings");
if ($settings_result) {
    while ($row = $settings_result->fetch_assoc()) {
        $settings[$row['setting_key']] = $row['setting_value'];
    }
}

$school_name = $settings['institute.name'] ?? "SCHOOL NAME (UPDATE SETTINGS)";
$school_address = $settings['institute.address'] ?? "School Address (UPDATE SETTINGS)";
$school_phone = $settings['institute.phone'] ?? ""; 
$current_session_id = $settings['academic.active_session_id'] ?? 1;
// Logo URL ko theek kiya gaya hai for the admin/ folder
$logo_path_setting = $settings['institute.logo_path'] ?? '';
$logo_url = !empty($logo_path_setting) ? '../' . $logo_path_setting : null;
// --- END: Settings Fetch Logic ---

// --- Helper Function: Get Number Suffix (1st, 2nd, 3rd) ---
if (!function_exists('getNumberSuffix')) {
    function getNumberSuffix($num) {
        if (!is_numeric($num) || $num <= 0) {
            return $num; 
        }
        if ($num % 100 >= 11 && $num % 100 <= 13) {
            return $num . 'th';
        }
        switch ($num % 10) {
            case 1:  return $num . 'st';
            case 2:  return $num . 'nd';
            case 3:  return $num . 'rd';
            default: return $num . 'th';
        }
    }
}
// --- End Helper Function ---

// Fetch all sessions for dropdown
$sessions_result = $conn->query("SELECT id, name FROM academic_sessions ORDER BY id DESC");
$sessions = $sessions_result->fetch_all(MYSQLI_ASSOC);

$selected_session_id = $_GET['session_id'] ?? $current_session_id;
$selected_exam_id = $_GET['exam_id'] ?? null;
$selected_class_id = $_GET['class_id'] ?? null;
$selected_section_id = $_GET['section_id'] ?? null; 
$student_id_filter = $_GET['student_id'] ?? null;

// Fetch exams based on selected session
$exams = [];
if ($selected_session_id) {
    $exams_result = $conn->query("SELECT id, name FROM exams WHERE session_id = {$selected_session_id} ORDER BY id DESC");
    $exams = $exams_result->fetch_all(MYSQLI_ASSOC);
}

$classes = $conn->query("SELECT id, name FROM academic_classes ORDER BY order_no ASC")->fetch_all(MYSQLI_ASSOC);
$sections = $conn->query("SELECT id, name FROM academic_sections WHERE status='active' ORDER BY name ASC")->fetch_all(MYSQLI_ASSOC);

$students_to_print = [];
$current_exam_name = '';

if ($selected_exam_id && $selected_class_id) {
    $exam_id = (int)$selected_exam_id;
    $class_id = (int)$selected_class_id;
    $section_id = $selected_section_id ? (int)$selected_section_id : NULL;
    $student_id = $student_id_filter ? (int)$student_id_filter : NULL;
    
    // --- Fetch Exam Name ---
    $exam_name_q = $conn->prepare("SELECT name FROM exams WHERE id = ?");
    $exam_name_q->bind_param("i", $exam_id);
    $exam_name_q->execute();
    $exam_data = $exam_name_q->get_result()->fetch_assoc();
    $current_exam_name = $exam_data['name'] ?? 'Selected Exam';
    $exam_name_q->close();

    // --- Fetch Students ---
    $query = "
        SELECT 
            s.id, s.first_name, s.last_name, s.father_name, 
            en.roll_no, 
            ac.name as class_name, 
            asec.name as section_name
        FROM students s
        JOIN student_enrollments en ON s.id = en.student_id
        JOIN academic_classes ac ON en.class_id = ac.id
        LEFT JOIN academic_sections asec ON en.section_id = asec.id
        WHERE en.session_id = ? AND en.class_id = ? AND s.status = 'active'
    ";
    $params = [$selected_session_id, $class_id];
    $types = "ii";

    if ($section_id !== NULL) {
        $query .= " AND en.section_id = ?";
        $params[] = $section_id;
        $types .= "i";
    }
    if ($student_id !== NULL) {
        $query .= " AND s.id = ?";
        $params[] = $student_id;
        $types .= "i";
    }
    
    $query .= " ORDER BY en.roll_no ASC";
    
    $stmt = $conn->prepare($query);
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    $students_to_print = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    // ===================================================================
    // ✅ POSITION CALCULATION LOGIC (NEW)
    // ===================================================================
    $total_students = count($students_to_print);
    $student_ids = array_column($students_to_print, 'id');

    if ($total_students > 0) {
        // 1. Fetch total marks and failure count for all students 
        $totals_query = "
            SELECT 
                em.student_id,
                SUM(CASE WHEN em.is_absent = 0 AND em.marks_obtained IS NOT NULL THEN em.marks_obtained ELSE 0 END) AS total_obtained,
                SUM(ces.total_marks) AS total_max,
                COUNT(CASE WHEN em.is_absent = 1 OR (em.marks_obtained < ces.passing_marks AND em.marks_obtained IS NOT NULL) THEN 1 ELSE NULL END) AS failed_subjects_count
            FROM exam_marks em
            INNER JOIN class_exam_structure ces 
                ON em.subject_id = ces.subject_id 
                AND em.exam_id = ces.exam_id 
                AND em.class_id = ces.class_id
            WHERE em.exam_id = ? AND em.class_id = ? AND em.student_id IN (" . implode(',', array_fill(0, count($student_ids), '?')) . ")
            GROUP BY em.student_id
        ";
        
        $types = "ii" . str_repeat("i", count($student_ids));
        $params_totals = array_merge([$exam_id, $class_id], $student_ids);
        
        $totals_stmt = $conn->prepare($totals_query);
        $totals_stmt->bind_param($types, ...$params_totals);
        $totals_stmt->execute();
        $totals_data = $totals_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        $totals_stmt->close();

        $student_totals_map = [];
        foreach ($totals_data as $data) {
            $student_totals_map[$data['student_id']] = [
                'total_obtained' => (float)$data['total_obtained'],
                'total_max' => (float)$data['total_max'],
                'failed_subjects_count' => (int)$data['failed_subjects_count']
            ];
        }
        
        // 2. Prepare data for sorting (merge student details with their calculated totals)
        $student_ranks = [];
        foreach ($students_to_print as $student) {
            $id = $student['id'];
            // If totals are missing, assume failure/incomplete data (total_obtained = -1, failed_subjects_count = 999)
            $totals = $student_totals_map[$id] ?? ['total_obtained' => -1, 'total_max' => 0, 'failed_subjects_count' => 999];
            
            $student_ranks[] = [
                'id' => $id,
                'roll_no' => (int)$student['roll_no'],
                'total_obtained' => $totals['total_obtained'],
                'failed_count' => $totals['failed_subjects_count']
            ];
        }
        
        // 3. Sort students: 
        // a) Total Marks DESC
        // b) Failed Count ASC (Less failures is better)
        // c) Roll No ASC (Tie-breaker)
        usort($student_ranks, function($a, $b) {
            if ($a['total_obtained'] !== $b['total_obtained']) {
                return $b['total_obtained'] <=> $a['total_obtained'];
            }
            if ($a['failed_count'] !== $b['failed_count']) {
                return $a['failed_count'] <=> $b['failed_count'];
            }
            return $a['roll_no'] <=> $b['roll_no'];
        });
        
        // 4. Assign Positions (MODIFIED LOGIC: Rank all students based on total marks, removing 'N/A' for failed subjects)
        $last_assigned_rank = 0; // The actual position number (1, 2, 3...)
        $group_total_for_rank = -1; // Total marks that achieved the $last_assigned_rank
        
        foreach ($student_ranks as $key => &$rank_data) {
            $current_total = $rank_data['total_obtained'];
            // The previous 'N/A' condition for failed or incomplete students has been intentionally removed as requested.
            
            // Rank logic applied to ALL students:
            if ($current_total !== $group_total_for_rank) {
                // New score group found - assign the next rank number (Dense Rank)
                $last_assigned_rank++; 
                $group_total_for_rank = $current_total; // Update the total for this rank group
            }
            
            // Assign the rank (tie-sharing). This now includes all students, irrespective of pass/fail status.
            $rank_data['position'] = $last_assigned_rank; 
        }
        unset($rank_data);
        
        // 5. Merge Position back into $students_to_print
        $position_map = array_column($student_ranks, 'position', 'id');

        foreach ($students_to_print as &$student) {
            $student['class_position'] = $position_map[$student['id']] ?? 'N/A';
        }
        unset($student); 
    }
    // ===================================================================
}
?>

<h1 class="mb-4">Print Student Result Card <i class="fas fa-print"></i></h1>

<div class="card p-3 mb-4 shadow-sm">
    <h5 class="card-title">Select Filters for Result Card</h5>
    <form method="GET" action="print_result_card.php" class="form-inline">
        <div class="form-group mr-3 mb-2">
            <label class="mr-2">Session:</label>
            <select name="session_id" id="session_id" class="form-control" required onchange="this.form.submit()">
                <option value="">-- Select Session --</option>
                <?php foreach ($sessions as $session): ?>
                    <option value="<?php echo $session['id']; ?>" <?php echo $selected_session_id == $session['id'] ? 'selected' : ''; ?>>
                        <?php echo htmlspecialchars($session['name']); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="form-group mr-3 mb-2">
            <label class="mr-2">Exam:</label>
            <select name="exam_id" class="form-control" required>
                <option value="">-- Select Exam --</option>
                <?php foreach ($exams as $exam): ?>
                    <option value="<?php echo $exam['id']; ?>" <?php echo $selected_exam_id == $exam['id'] ? 'selected' : ''; ?>>
                        <?php echo htmlspecialchars($exam['name']); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="form-group mr-3 mb-2">
            <label class="mr-2">Class:</label>
            <select name="class_id" class="form-control" required>
                <option value="">-- Select Class --</option>
                <?php foreach ($classes as $class): ?>
                    <option value="<?php echo $class['id']; ?>" <?php echo $selected_class_id == $class['id'] ? 'selected' : ''; ?>>
                        <?php echo htmlspecialchars($class['name']); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="form-group mr-3 mb-2">
            <label class="mr-2">Section (Optional):</label>
            <select name="section_id" class="form-control">
                <option value="">-- All Sections --</option>
                <?php foreach ($sections as $section): ?>
                    <option value="<?php echo $section['id']; ?>" <?php echo $selected_section_id == $section['id'] ? 'selected' : ''; ?>>
                        <?php echo htmlspecialchars($section['name']); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="form-group mr-3 mb-2">
            <label class="mr-2">Student ID (Optional):</label>
            <input type="number" name="student_id" class="form-control" value="<?php echo htmlspecialchars($student_id_filter); ?>" placeholder="Enter Student ID">
        </div>
        
        <button type="submit" class="btn btn-primary mb-2"><i class="fas fa-search"></i> Show Results</button>
    </form>
</div>

<?php if (count($students_to_print) > 0): ?>
    
    <div class="alert alert-info">Found <?php echo count($students_to_print); ?> student(s). Use **Print** button below to generate result cards.</div>
    <button onclick="window.print()" class="btn btn-warning mb-4"><i class="fas fa-print"></i> Print All Result Cards</button>

<div class="result-cards-container">
    
        <?php foreach ($students_to_print as $student): ?>
            
            <div class="result-card-page">
                <div class="card p-4 result-card-print">
                    
                    <div class="result-header" style="display: flex; align-items: center; justify-content: space-between; text-align: center; border-bottom: 2px solid #000; padding-bottom: 8px; margin-bottom: 10px;">
                        <div style="flex: 0 0 15%; text-align: left;">
                            <?php if ($logo_url && file_exists('../' . $settings['institute.logo_path'] ?? '')): ?>
                                <img src="<?php echo htmlspecialchars($logo_url); ?>" alt="School Logo" style="height: 80px;">
                            <?php endif; ?>
                        </div>

                        <div style="flex: 1; text-align: center;">
                            <h2 style="margin: 0; font-weight: bold;"><?php echo strtoupper(htmlspecialchars($school_name)); ?></h2>
                            <p style="margin: 4px 0; font-size: 12px;"><?php echo htmlspecialchars($school_address); ?></p>
                            
                            <?php if (!empty($school_phone)): ?>
                                <p style="margin: 0; font-size: 11px;">Phone: <?php echo htmlspecialchars($school_phone); ?></p> <?php endif; ?>
                            
                            <h4 style="margin-top: 8px; font-size: 14px;"><?php echo strtoupper(htmlspecialchars($current_exam_name)); ?> RESULT CARD</h4>
                        </div>

                        <div style="flex: 0 0 15%;"></div>
                    </div>
                    <div class="student-details-box">
                        <div class="detail-row">
                            <div class="detail-item">
                                <span class="detail-label">Student Name:</span> 
                                <span class="detail-value"><?php echo htmlspecialchars($student['first_name'] . ' ' . $student['last_name']); ?></span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">Father's Name:</span> 
                                <span class="detail-value"><?php echo htmlspecialchars($student['father_name'] ?? 'N/A'); ?></span>
                            </div>
                        </div>
                        <div class="detail-row">
                            <div class="detail-item">
                                <span class="detail-label">Class:</span> 
                                <span class="detail-value"><?php echo htmlspecialchars($student['class_name']); ?> (<?php echo htmlspecialchars($student['section_name'] ?? 'A'); ?>)</span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">Roll No:</span> 
                                <span class="detail-value"><?php echo htmlspecialchars($student['roll_no']); ?></span>
                            </div>
                        </div>
                        <div class="detail-row">
                            <div class="detail-item">
                                <span class="detail-label">Class Position:</span> 
                                <span class="detail-value">
                                    <?php 
                                            $pos = $student['class_position'] ?? 'N/A';
                                            echo is_numeric($pos) ? getNumberSuffix($pos) : $pos; 
                                        ?>
                                </span>
                            </div>
                            <?php /* START: Total Students line removed as per user request */ ?>
                            <?php /* END: Total Students line removed */ ?>
                        </div>
                        </div>
                    <hr class="hr-details">
                    
                    <?php
                    // --- ON-THE-FLY RESULT CALCULATION (Same as before) ---
                    $student_id = $student['id'];
                    $total_max_marks = 0;
                    $total_obtained_marks = 0;
                    $failed_subjects_array = [];
                    $max_marks_defined = true;
                    
                    // Fetch Subject Structure and Marks data 
                    $marks_query = "
                        SELECT 
                            acs.id AS subject_id, acs.name AS subject_name,
                            ces.total_marks, ces.passing_marks,
                            em.marks_obtained, em.is_absent
                        FROM subjects acs 
                        INNER JOIN course_allocation ca ON ca.subject_id = acs.id 
                        LEFT JOIN class_exam_structure ces ON ces.subject_id = acs.id AND ces.exam_id = ? AND ces.class_id = ?
                        LEFT JOIN exam_marks em 
                            ON em.subject_id = acs.id 
                            AND em.student_id = ? 
                            AND em.exam_id = ?
                            AND em.class_id = ?
                        WHERE ca.class_id = ? 
                        ORDER BY acs.name ASC
                    ";
                    
                    $marks_stmt = $conn->prepare($marks_query);
                    $marks_stmt->bind_param("iiiiii", $exam_id, $class_id, $student_id, $exam_id, $class_id, $class_id);
                    $marks_stmt->execute();
                    $marks_data = $marks_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
                    $marks_stmt->close();
                    
                    ?>
                    
                    <table class="result-table">
                        <thead>
                            <tr>
                                <th style="width: 50%;">Subject</th>
                                <th style="text-align:center" style="width: 15%;">Max Marks</th>
                                <th style="text-align:center" style="width: 15%;">Passing Marks</th>
                                <th style="text-align:center" style="width: 20%;">Marks Obtained</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($marks_data as $mark): 
                                $max_marks = (int)($mark['total_marks'] ?? 0); 
                                $passing_marks = (int)($mark['passing_marks'] ?? 0);
                                $obtained = $mark['marks_obtained'] !== NULL ? (float)$mark['marks_obtained'] : NULL;
                                $is_absent = (int)($mark['is_absent'] ?? 0);
                                
                                $obtained_display = '-';
                                $row_class = '';

                                if ($max_marks === 0) {
                                    $max_marks_defined = false;
                                } else {
                                    $total_max_marks += $max_marks;
                                    
                                    if ($is_absent) {
                                        $obtained_display = 'ABSENT';
                                        $failed_subjects_array[] = htmlspecialchars($mark['subject_name']);
                                        $row_class = 'row-fail';
                                    } elseif ($obtained === NULL) {
                                        $obtained_display = 'PENDING';
                                    } else {
                                        $total_obtained_marks += $obtained; 
                                        $obtained_display = number_format($obtained, 2);
                                        
                                        if ($obtained < $passing_marks) {
                                            $failed_subjects_array[] = htmlspecialchars($mark['subject_name']);
                                            $row_class = 'row-fail';
                                        }
                                    }
                                }
                            ?>
                            <tr class="<?php echo $row_class; ?>">
                                <td><?php echo htmlspecialchars($mark['subject_name']); ?></td>
                                <td class="text-center"><?php echo $max_marks > 0 ? $max_marks : 'N/A'; ?></td>
                                <td class="text-center"><?php echo $passing_marks > 0 ? $passing_marks : 'N/A'; ?></td>
                                <td class="text-center <?php echo $row_class === 'row-fail' ? 'text-danger' : ''; ?>"><?php echo $obtained_display; ?></td>
                            </tr>
                            <?php endforeach; ?>
                            
                            <?php 
                            $failed_subjects_count = count($failed_subjects_array);
                            
                            if (!$max_marks_defined) {
                                $total_display = 'STRUCTURE INCOMPLETE';
                            } elseif ($failed_subjects_count > 0) {
                                // Fail aur Absent subjects ka naam dikhega
                                $total_display = 'Fail in: ' . implode(', ', $failed_subjects_array);
                            } elseif ($total_max_marks > 0) {
                                $total_display = "**" . number_format($total_obtained_marks, 2) . " / {$total_max_marks}**";
                            } else {
                                $total_display = 'N/A';
                            }
                            ?>

                            <tr class="total-row">
                                <td colspan="3" class="text-right total-label">
                                    <?php 
                                    echo '**TOTAL**'; 
                                    ?>
                                </td>
                                <td class="text-center total-value <?php echo $failed_subjects_count > 0 ? 'total-fail-message' : ''; ?>">
                                    <?php echo $total_display; ?>
                                </td>
                            </tr>
                        </tbody>
                    </table>

                    <?php 
                    $percentage = $total_max_marks > 0 ? round(($total_obtained_marks / $total_max_marks) * 100, 2) : 0;
                    $final_status_text = '';
                    
                    if (!$max_marks_defined) {
                        $final_status_text = 'STRUCTURE INCOMPLETE';
                    } elseif ($failed_subjects_count > 0) {
                        $final_status_text = 'FAIL'; // Fail status ab Absent subjects ko bhi cover karega
                    } elseif ($total_max_marks > 0 && $total_obtained_marks >= 0) {
                        $final_status_text = 'PASS';
                    } else {
                        $final_status_text = 'N/A';
                    }
                    ?>
                    
                    <div class="result-summary mt-3">
                        <p><strong>Percentage:</strong> **<?php echo $percentage; ?>%**</p>
                        <p><strong>Result:</strong> **<?php echo htmlspecialchars($final_status_text); ?>**</p>
                    </div>
                    
                    <div class="signature-block">
                        <div class="sig-item">
                            <hr>
                            <span class="sig-label">Parents Signature</span>
                        </div>
                        <div class="sig-item">
                            <hr>
                            <span class="sig-label">Class Teacher Signature</span>
                        </div>
                        <div class="sig-item">
                            <hr>
                            <span class="sig-label">Principal Signature</span>
                        </div>
                    </div>
                </div>
            </div>
            
        <?php endforeach; ?>
    </div>
    
<?php else: ?>
    <div class="alert alert-info">Please select the filters and click 'Show Results' to view the result cards.</div>
<?php endif; ?>

<style>
/* @page rule to remove browser headers/footers (Date, URL, Page Number) */
@page {
    size: A4; 
    margin: 0; 
}

@media print {
    /* --- Hiding UI Elements --- */
    body { 
        margin: 0; 
        padding: 0; 
        -webkit-print-color-adjust: exact;
        print-color-adjust: exact;
        font-family: Arial, sans-serif;
        font-size: 11pt;
    }
    .navbar, .sidebar, .card:not(.result-card-print), .alert, .btn, form, h1, #sidebar, #header, #footer {
        display: none !important; 
    }
    
    /* --- Card Structure & Positioning --- */
    .result-card-page {
        margin: 0;
        padding: 0;
        width: 100%;
        height: 297mm; /* Full A4 height */
        position: relative; 
        page-break-after: always;
    }

    .result-card-print {
        box-shadow: none;
        padding: 15mm; /* 15mm margin inside the page area */
        margin: 0;
        height: 100%; 
        position: relative;
    }
    
    /* --- Header & Details --- */
    .school-name { font-size: 16pt; font-weight: bold; margin-bottom: 0; }
    .school-address { font-size: 10pt; margin: 5px 0 15px 0; }
    .exam-title { font-size: 14pt; margin-top: 10px; }
    .hr-header { border-top: 2px solid #000; margin: 15px 0; }
    .hr-details { border-top: 1px solid #000; margin: 15px 0; }
    
    .student-details-box {
        border: 1px solid #000;
        padding: 10px;
        margin-bottom: 10px;
    }
    .detail-row {
        display: flex;
        justify-content: space-between;
        margin-bottom: 5px;
    }
    .detail-item {
        width: 50%;
        font-size: 11pt;
    }
    .detail-label {
        font-weight: bold;
        display: inline-block;
        width: 120px; 
        padding-right: 5px;
    }
    .detail-value {
        font-weight: normal;
    }

    /* --- Table Styling --- */
    .result-table {
        width: 100%;
        border-collapse: collapse; 
        margin-top: 15px;
    }
    .result-table th, .result-table td {
        border: 1px solid #000;
        padding: 6px 8px;
        font-size: 10pt;
    }
    .result-table th {
        background-color: #f0f0f0 !important;
        font-weight: bold;
        text-align: center;
        vertical-align: middle;
    }
    
    /* Total Row Fixes */
    .total-row td {
        border-top: 2px solid #000 !important;
        font-weight: bold;
    }
    .total-row .total-label {
        text-align: right !important;
        background-color: #e0e0e0 !important;
        padding-right: 15px; 
    }
    .total-row .total-value {
        text-align: center !important; 
        width: 20%; 
        white-space: normal; 
    }
    
    .total-fail-message {
        font-size: 9pt; 
        color: #d9534f !important; /* Force red color for print */
    }

    /* --- Signatures (Fixed to the bottom of the page) --- */
    .signature-block {
        position: absolute;
        bottom: 1mm; /* Page ke bilkul bottom se 5mm upar */
        left: 15mm; 
        right: 15mm; 
        width: calc(100% - 30mm); 
        
        padding: 0 10px;
        font-size: 10pt;
        display: flex; 
        justify-content: space-around;
    }
    .sig-item {
        width: 25%;
        text-align: center;
    }
    .sig-item hr {
        border: none; 
        border-top: 1px solid #000; 
        margin-bottom: 2px;
    }
    .sig-label {
        font-style: italic;
        display: block;
        margin-top: 5px;
    }
    
    .result-card-page:last-child {
        page-break-after: avoid;
    }
}

/* --------------------------------------------------------------------------------
    Display CSS (Screen View) - For clarity only
-------------------------------------------------------------------------------- */
.result-card-page { padding: 15px; margin-bottom: 25px; }
.result-card-print { border: none; } 
.text-danger { color: #d9534f; }
</style>


<?php
// Close connection if open
if (isset($conn) && is_object($conn)) {
    $conn->close();
}
include_once('includes/footer.php'); 
ob_end_flush();
?>