 <?php 
// FILE: admin/print_session_result_template.php - Template for Combined Session Result Card

// Ensure all required variables are set by the main file (print_session_result.php)
if (!isset($settings) || !isset($student_data) || !isset($all_terms_results)) {
    die("<h1>Error:</h1> Data variables are missing in the template.");
}

$school_name = htmlspecialchars($settings['institute_name']);
$school_address = htmlspecialchars($settings['institute_address']);
$school_phone = htmlspecialchars($settings['institute_phone']);
$logo_url = $settings['logo_url'] ?? null;

$student_name = htmlspecialchars($student_data['first_name'] . ' ' . $student_data['last_name']);
$father_name = htmlspecialchars($student_data['father_name'] ?? 'N/A');
$class_name = htmlspecialchars($student_data['class_name']);
$section_name = htmlspecialchars($student_data['section_name'] ?? 'A');
$roll_no = htmlspecialchars($student_data['roll_no']);

// Function to process marks and calculate totals/status for ONE term
function processTermResults($marks_data) {
    $total_max_marks = 0;
    $total_obtained_marks = 0;
    $failed_subjects_array = [];
    $max_marks_defined = true;

    foreach ($marks_data as &$mark) {
        $max_marks = (int)($mark['total_marks'] ?? 0);
        $passing_marks = (int)($mark['passing_marks'] ?? 0);
        $obtained = $mark['marks_obtained'] !== NULL ? (float)$mark['marks_obtained'] : NULL;
        $is_absent = (int)($mark['is_absent'] ?? 0);

        $mark['obtained_display'] = '-';
        $mark['row_class'] = '';

        if ($max_marks === 0) {
            $max_marks_defined = false;
        } else {
            if (!$is_absent && $obtained !== NULL) {
                $total_max_marks += $max_marks;
                $total_obtained_marks += $obtained;
                $mark['obtained_display'] = number_format($obtained, 2);

                if ($obtained < $passing_marks) {
                    $failed_subjects_array[] = htmlspecialchars($mark['subject_name']);
                    $mark['row_class'] = 'row-fail';
                }
            } elseif ($is_absent) {
                $total_max_marks += $max_marks; // Max marks should be counted for total
                $failed_subjects_array[] = htmlspecialchars($mark['subject_name']) . " (ABS)";
                $mark['obtained_display'] = 'ABSENT';
                $mark['row_class'] = 'row-fail';
            } elseif ($obtained === NULL) {
                $mark['obtained_display'] = 'PENDING';
            }
        }
    }
    
    $failed_subjects_count = count($failed_subjects_array);
    $percentage = $total_max_marks > 0 ? round(($total_obtained_marks / $total_max_marks) * 100, 2) : 0;
    
    if (!$max_marks_defined) {
        $final_status_text = 'STRUCTURE INCOMPLETE';
        $total_display = 'N/A';
    } elseif ($failed_subjects_count > 0) {
        $final_status_text = 'FAIL';
        $total_display = 'Fail in: ' . implode(', ', $failed_subjects_array);
    } elseif ($total_max_marks > 0) {
        $final_status_text = 'PASS';
        $total_display = "**" . number_format($total_obtained_marks, 2) . " / {$total_max_marks}**";
    } else {
        $final_status_text = 'N/A';
        $total_display = 'N/A';
    }

    return [
        'marks_data' => $marks_data,
        'total_max_marks' => $total_max_marks,
        'total_obtained_marks' => $total_obtained_marks,
        'percentage' => $percentage,
        'final_status_text' => $final_status_text,
        'total_display' => $total_display,
        'failed_subjects_count' => $failed_subjects_count
    ];
}
?>

<div class="result-cards-container">
    <div class="result-card-page">
        <div class="card p-4 result-card-print">
            
            <div class="result-header" style="display: flex; align-items: center; justify-content: space-between; text-align: center; border-bottom: 2px solid #000; padding-bottom: 8px; margin-bottom: 10px;">
                <div style="flex: 0 0 15%; text-align: left;">
                    <?php if ($logo_url): ?>
                        <img src="<?php echo htmlspecialchars($logo_url); ?>" alt="School Logo" style="height: 80px;">
                    <?php endif; ?>
                </div>

                <div style="flex: 1; text-align: center;">
                    <h2 style="margin: 0; font-weight: bold;"><?php echo strtoupper($school_name); ?></h2>
                    <p style="margin: 4px 0; font-size: 12px;"><?php echo $school_address; ?></p>
                    
                    <?php if (!empty($school_phone)): ?>
                        <p style="margin: 0; font-size: 11px;">Phone: <?php echo $school_phone; ?></p> 
                    <?php endif; ?>
                    
                    <h4 style="margin-top: 8px; font-size: 14px;">ACADEMIC SESSION RESULT REPORT</h4>
                </div>
                <div style="flex: 0 0 15%;"></div>
            </div>

            <div class="student-details-box">
                <div class="detail-row">
                    <div class="detail-item">
                        <span class="detail-label">Student Name:</span> 
                        <span class="detail-value"><?php echo $student_name; ?></span>
                    </div>
                    <div class="detail-item">
                        <span class="detail-label">Father's Name:</span> 
                        <span class="detail-value"><?php echo $father_name; ?></span>
                    </div>
                </div>
                <div class="detail-row">
                    <div class="detail-item">
                        <span class="detail-label">Class:</span> 
                        <span class="detail-value"><?php echo $class_name; ?> (<?php echo $section_name; ?>)</span>
                    </div>
                    <div class="detail-item">
                        <span class="detail-label">Roll No:</span> 
                        <span class="detail-value"><?php echo $roll_no; ?></span>
                    </div>
                </div>
            </div>
            <hr class="hr-details">
            
            <?php 
            $overall_max_marks = 0;
            $overall_obtained_marks = 0;
            $overall_term_count = 0;

            foreach ($all_terms_results as $index => $term_result): 
                $processed_data = processTermResults($term_result['marks_data']);
                
                // Overall Calculation
                $overall_max_marks += $processed_data['total_max_marks'];
                $overall_obtained_marks += $processed_data['total_obtained_marks'];
                if ($processed_data['total_max_marks'] > 0) {
                    $overall_term_count++;
                }

                // Term Heading Logic
                $term_label = '';
                if ($index == 0) $term_label = 'FIRST TERM';
                elseif ($index == 1) $term_label = 'SECOND TERM';
                elseif ($index == 2) $term_label = 'FINAL TERM';
                else $term_label = strtoupper($term_result['term_name']);
            ?>

                <h5 class="term-heading">**<?php echo $term_label; ?>** (<?php echo $term_result['term_name']; ?>)</h5>
                
                <table class="result-table term-table">
                    <thead>
                        <tr>
                            <th style="width: 50%;">Subject</th>
                            <th style="width: 15%;">Max Marks</th>
                            <th style="width: 15%;">Passing Marks</th>
                            <th style="width: 20%;">Marks Obtained</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($processed_data['marks_data'] as $mark): ?>
                            <tr class="<?php echo $mark['row_class']; ?>">
                                <td><?php echo htmlspecialchars($mark['subject_name']); ?></td>
                                <td class="text-center"><?php echo (int)($mark['total_marks'] ?? 0) > 0 ? (int)$mark['total_marks'] : 'N/A'; ?></td>
                                <td class="text-center"><?php echo (int)($mark['passing_marks'] ?? 0) > 0 ? (int)$mark['passing_marks'] : 'N/A'; ?></td>
                                <td class="text-center <?php echo $mark['row_class'] === 'row-fail' ? 'text-danger' : ''; ?>">
                                    <?php echo $mark['obtained_display']; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                        
                        <tr class="total-row">
                            <td colspan="3" class="text-right total-label">**TOTAL (<?php echo $term_result['term_name']; ?>)**</td>
                            <td class="text-center total-value <?php echo $processed_data['failed_subjects_count'] > 0 ? 'total-fail-message' : ''; ?>">
                                <?php echo $processed_data['total_display']; ?>
                            </td>
                        </tr>
                    </tbody>
                </table>
                <div class="result-summary term-summary">
                    <p><strong>Percentage:</strong> **<?php echo $processed_data['percentage']; ?>%**</p>
                    <p><strong>Result:</strong> **<?php echo $processed_data['final_status_text']; ?>**</p>
                </div>
                <hr class="hr-term">
            <?php endforeach; ?>

            <?php 
            $overall_percentage = $overall_max_marks > 0 ? round(($overall_obtained_marks / $overall_max_marks) * 100, 2) : 0;
            ?>
            <div class="result-summary final-summary mt-4">
                <h4 style="text-align: center; border-bottom: 1px dashed #000; padding-bottom: 5px;">**OVERALL SESSION PERFORMANCE**</h4>
                <div class="detail-row">
                    <div class="detail-item" style="width: 33%;">
                        <span class="detail-label">Terms Evaluated:</span> 
                        <span class="detail-value">**<?php echo $overall_term_count; ?>**</span>
                    </div>
                    <div class="detail-item" style="width: 33%;">
                        <span class="detail-label">Total Marks:</span> 
                        <span class="detail-value">**<?php echo number_format($overall_obtained_marks, 2); ?>** / **<?php echo $overall_max_marks; ?>**</span>
                    </div>
                    <div class="detail-item" style="width: 33%;">
                        <span class="detail-label">Overall Percentage:</span> 
                        <span class="detail-value" style="font-size: 14pt;">**<?php echo $overall_percentage; ?>%**</span>
                    </div>
                </div>
            </div>
            
            <div class="signature-block">
                <div class="sig-item"><hr><span class="sig-label">Parents Signature</span></div>
                <div class="sig-item"><hr><span class="sig-label">Class Teacher Signature</span></div>
                <div class="sig-item"><hr><span class="sig-label">Principal Signature</span></div>
            </div>
        </div>
    </div>
</div>

<style>
/* --------------------------------------------------------------------------------
   PRINT CSS (Same as print_result_card.php for consistency)
-------------------------------------------------------------------------------- */
@page {
    size: A4; 
    margin: 0; 
}

@media print {
    body { 
        margin: 0; 
        padding: 0; 
        -webkit-print-color-adjust: exact;
        print-color-adjust: exact;
        font-family: Arial, sans-serif;
        font-size: 11pt;
    }
    .navbar, .sidebar, .card:not(.result-card-print), .alert, .btn, form, h1, #sidebar, #header, #footer {
        display: none !important; 
    }
    
    .result-card-page {
        margin: 0;
        padding: 0;
        width: 100%;
        height: 297mm;
        position: relative; 
        page-break-after: always;
    }

    .result-card-print {
        box-shadow: none;
        padding: 15mm; 
        margin: 0;
        height: 100%; 
        position: relative;
    }
    
    .result-header h4 { font-size: 14pt !important; }

    .student-details-box {
        border: 1px solid #000;
        padding: 10px;
        margin-bottom: 10px;
    }
    .detail-row {
        display: flex;
        justify-content: space-between;
        margin-bottom: 5px;
    }
    .detail-item {
        width: 50%;
        font-size: 11pt;
    }
    .detail-label {
        font-weight: bold;
        display: inline-block;
        width: 120px; 
        padding-right: 5px;
    }
    
    /* Term Headings */
    .term-heading { 
        font-size: 13pt; 
        text-align: center; 
        margin: 15px 0 5px 0; 
        background-color: #f2f2f2 !important; 
        padding: 3px;
        border: 1px solid #000;
    }

    /* Table Styling */
    .result-table {
        width: 100%;
        border-collapse: collapse; 
        margin-top: 5px;
        page-break-inside: auto; /* Allow tables to break */
    }
    .result-table th, .result-table td {
        border: 1px solid #000;
        padding: 4px 8px;
        font-size: 9pt;
        line-height: 1.2;
    }
    .result-table th {
        background-color: #f0f0f0 !important;
        font-weight: bold;
        text-align: center;
        vertical-align: middle;
    }
    
    /* Total Row Fixes */
    .total-row td {
        border-top: 2px solid #000 !important;
        font-weight: bold;
    }
    .total-row .total-label {
        text-align: right !important;
        background-color: #e0e0e0 !important;
        padding-right: 15px; 
    }
    .total-fail-message {
        font-size: 9pt; 
        color: #d9534f !important; /* Force red color for print */
    }

    /* Summary Block */
    .result-summary {
        display: flex;
        justify-content: flex-start;
        align-items: flex-start;
        font-size: 10pt;
        margin-top: 5px;
        gap: 20px;
        padding-left: 10px;
    }
    
    /* Hr Divider between terms */
    .hr-term {
        border: none;
        border-top: 1px dashed #ccc;
        margin: 10px 0;
    }
    
    .final-summary {
        border: 2px solid #333;
        padding: 10px;
        background-color: #f9f9f9 !important;
        break-before: avoid; /* Don't break right before the final summary */
    }
    .final-summary .detail-item { width: 33.33%; }


    /* Signatures (Fixed to the bottom of the page) */
    .signature-block {
        position: absolute;
        bottom: 1mm; 
        left: 15mm; 
        right: 15mm; 
        width: calc(100% - 30mm); 
        padding: 0 10px;
        font-size: 10pt;
        display: flex; 
        justify-content: space-around;
        page-break-before: avoid;
    }
    .sig-item {
        width: 25%;
        text-align: center;
    }
    .sig-item hr {
        border: none; 
        border-top: 1px solid #000; 
        margin-bottom: 2px;
    }
    .sig-label {
        font-style: italic;
        display: block;
        margin-top: 5px;
    }
    
    .result-card-page:last-child {
        page-break-after: avoid;
    }
}
/* -------------------------------------------------------------------------------- */
</style>