<?php
// FILE: admin/salary_process.php - FINAL FIXED CODE (Ready for Separate Deductions)
include_once('../config/db.php'); // Ensure this path is correct

$conn = getDBConnection();

// Check if the request is POST for generating slips, otherwise use GET for actions
$action = $_REQUEST['action'] ?? '';

if (empty($action)) {
    header("Location: salary_generate.php?err=" . urlencode("Invalid action specified."));
    exit();
}

try {
    if ($action === 'generate_slips') {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            throw new Exception("Invalid request method.");
        }
        
        $month_year_post = $_POST['month_year'] ?? ''; // YYYY-MM format from form
        $teacher_id_filter = $_POST['teacher_id'] ?? 'all';
        
        if (empty($month_year_post)) {
            throw new Exception("Month is required for slip generation.");
        }
        
        // FIX: Convert YYYY-MM (from form) to YYYY-MM-01 (for correct database DATE type)
        $month_year_db = $month_year_post . '-01'; 
        
        // 1. Fetch Staff Members based on filter
        $staff_query = "SELECT id, salary FROM teachers WHERE status='active'";
        $params = [];
        $types = "";

        if ($teacher_id_filter !== 'all' && (int)$teacher_id_filter > 0) {
            $staff_query .= " AND id = ?";
            $params[] = (int)$teacher_id_filter;
            $types .= "i";
        }
        
        $stmt_staff = $conn->prepare($staff_query);
        if (!empty($types)) {
            $stmt_staff->bind_param($types, ...$params);
        }
        $stmt_staff->execute();
        $staff_to_process = $stmt_staff->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmt_staff->close();

        if (empty($staff_to_process)) {
            throw new Exception("No staff members found to process.");
        }
        
        $slips_generated_count = 0;
        $slips_skipped_count = 0; 
        
        // 2. Loop through each staff member to create the slip
        foreach ($staff_to_process as $staff) {
            $teacher_id = $staff['id'];
            $base_salary = $staff['salary']; 
            
            // Deductions are now handled via deduction_process.php after slip generation.
            $total_deduction = 0; 
            
            $gross_salary = $base_salary;
            $net_paid = $gross_salary - $total_deduction;
            
            // 3. FIX: Check if slip already exists for this staff/month before inserting
            $check_stmt = $conn->prepare("SELECT id FROM teacher_salary_slips WHERE teacher_id = ? AND DATE_FORMAT(month_year, '%Y-%m') = ?");
            // Use YYYY-MM for comparison
            $check_stmt->bind_param("is", $teacher_id, $month_year_post); 
            $check_stmt->execute();
            $check_stmt->store_result(); 

            if ($check_stmt->num_rows > 0) {
                // Slip already exists (Duplicate), so skip this teacher
                $check_stmt->close();
                $slips_skipped_count++;
                continue; 
            }
            $check_stmt->close(); 

            // 4. Insert the main Salary Slip record
            $insert_slip_query = $conn->prepare("
                INSERT INTO teacher_salary_slips 
                    (teacher_id, month_year, gross_salary, total_deduction, net_paid, status, created_at) 
                VALUES (?, ?, ?, ?, ?, 'generated', NOW())
            ");
            $insert_slip_query->bind_param("isddd", 
                $teacher_id, 
                $month_year_db, // FIX: Inserting YYYY-MM-01
                $gross_salary, 
                $total_deduction, // This will be 0 initially
                $net_paid // This will be equal to gross_salary initially
            );
            $insert_slip_query->execute();
            $new_slip_id = $conn->insert_id;
            $insert_slip_query->close();

            // 5. Item saving logic (Only insert Base Salary item initially)
            $insert_items_query = $conn->prepare("
                INSERT INTO teacher_salary_items 
                    (slip_id, item_type, item_description, amount) 
                VALUES (?, ?, ?, ?)
            ");

            // --- 5.1. INSERTING BASE SALARY (Allowance) ---
            if ($base_salary > 0) {
                $item_type_allowance = 'allowance';
                $base_salary_description = 'Base Monthly Salary'; 
                
                $insert_items_query->bind_param("issd", 
                    $new_slip_id, 
                    $item_type_allowance, 
                    $base_salary_description, 
                    $base_salary
                );
                $insert_items_query->execute();
            }
            
            // Deductions are applied later, so no deduction items are inserted here.

            $insert_items_query->close();
            $slips_generated_count++;
        }
        
        $msg = "Successfully generated " . $slips_generated_count . " salary slips.";
        if ($slips_skipped_count > 0) {
            $msg .= " (" . $slips_skipped_count . " slips skipped as they already exist).";
        }
        header("Location: salary_generate.php?msg=" . urlencode($msg));
        exit();

    } elseif ($action === 'mark_paid') {
        // --- Mark Paid Logic ---
        $slip_id = (int)($_GET['id'] ?? 0);
        if ($slip_id === 0) {
             throw new Exception("Invalid slip ID for Mark Paid action.");
        }
        
        $stmt = $conn->prepare("UPDATE teacher_salary_slips SET status = 'paid', paid_date = NOW() WHERE id = ? AND status = 'generated'");
        $stmt->bind_param("i", $slip_id);
        $stmt->execute();
        
        if ($stmt->affected_rows > 0) {
            $msg = "Salary slip #{$slip_id} marked as PAID.";
            header("Location: salary_generate.php?msg=" . urlencode($msg));
        } else {
            $err = "Could not mark slip #{$slip_id} as paid (might be already paid/cancelled).";
            header("Location: salary_generate.php?err=" . urlencode($err));
        }
        $stmt->close();
        exit();

    } elseif ($action === 'delete_slip') {
        // --- Delete Slip Logic (Handles Items and Deductions) ---
        $slip_id = (int)($_GET['id'] ?? 0);
        if ($slip_id === 0) {
            throw new Exception("Invalid slip ID for Delete action.");
        }

        // Use transaction to ensure slip, items, and deductions are deleted
        $conn->begin_transaction();
        
        // 1. Delete deductions first
        $delete_deductions_stmt = $conn->prepare("DELETE FROM teacher_deductions WHERE slip_id = ?");
        $delete_deductions_stmt->bind_param("i", $slip_id);
        $delete_deductions_stmt->execute();
        $delete_deductions_stmt->close();

        // 2. Delete items 
        $delete_items_stmt = $conn->prepare("DELETE FROM teacher_salary_items WHERE slip_id = ?");
        $delete_items_stmt->bind_param("i", $slip_id);
        $delete_items_stmt->execute();
        $delete_items_stmt->close();
        
        // 3. Delete the main slip
        $delete_slip_stmt = $conn->prepare("DELETE FROM teacher_salary_slips WHERE id = ?");
        $delete_slip_stmt->bind_param("i", $slip_id);
        $delete_slip_stmt->execute();
        
        if ($delete_slip_stmt->affected_rows > 0) {
            $conn->commit();
            $msg = "Salary slip #{$slip_id} and its records successfully DELETED.";
            header("Location: salary_generate.php?msg=" . urlencode($msg));
        } else {
            $conn->rollback();
            $err = "Could not delete slip #{$slip_id} (Slip not found).";
            header("Location: salary_generate.php?err=" . urlencode($err));
        }
        $delete_slip_stmt->close();
        exit();
    } else {
        throw new Exception("Unknown action: " . $action);
    }
} catch (Exception $e) {
    // Agar koi ghalti ho to usko wapis generate page par bhej dein
    header("Location: salary_generate.php?err=" . urlencode("Error: " . $e->getMessage()));
    exit();
} finally {
    if ($conn && $conn->ping()) {
        $conn->close();
    }
}
?>