<?php
// FILE: admin/setup_roles.php - Use this file ONLY ONCE for setup/cleanup.
// CRITICAL: Ensure correct path to your DB configuration.
include_once('../config/db.php'); 

// Check if the script is run directly (optional security measure)
if (basename($_SERVER['PHP_SELF']) !== 'setup_roles.php') {
    die("Access Denied: This script is for one-time setup.");
}

$conn = getDBConnection();
$message = '';
$error = '';

try {
    // 1. Transaction start
    $conn->begin_transaction();

    // --- Cleanup Existing Roles (except Admin) ---
    // Assuming 'Admin' role always has ID 1 (Standard practice)
    $conn->query("DELETE FROM user_roles WHERE id != 1");
    $message .= "Old roles (excluding Admin) successfully removed. ";

    // --- Ensure Admin Role Exists (ID 1) ---
    // Update or insert Admin role
    $admin_role_name = 'Admin';
    $admin_check = $conn->query("SELECT id FROM user_roles WHERE id = 1");
    if ($admin_check && $admin_check->num_rows == 0) {
        // If ID 1 is missing, insert it. (Assuming auto-increment is reset or managed)
        // Note: For safer operation, we usually check by name, but for cleanup, ID 1 is critical.
        $stmt_admin = $conn->prepare("INSERT INTO user_roles (id, name) VALUES (1, ?)");
        $stmt_admin->bind_param("s", $admin_role_name);
        $stmt_admin->execute();
        $stmt_admin->close();
        $message .= "Admin role (ID 1) ensured. ";
    } else if ($admin_check && $admin_check->num_rows > 0) {
        // Ensure the name is 'Admin'
        $conn->query("UPDATE user_roles SET name = 'Admin' WHERE id = 1");
        $message .= "Admin role (ID 1) confirmed. ";
    }

    // --- Add Parents Role (If it doesn't exist) ---
    $parent_role_name = 'Parents';
    $stmt_parent_check = $conn->prepare("SELECT id FROM user_roles WHERE name = ?");
    $stmt_parent_check->bind_param("s", $parent_role_name);
    $stmt_parent_check->execute();
    $result_parent = $stmt_parent_check->get_result();
    
    if ($result_parent->num_rows == 0) {
        // Insert the new Parents role (let MySQL assign the ID, which will be > 1)
        $stmt_parent_insert = $conn->prepare("INSERT INTO user_roles (name) VALUES (?)");
        $stmt_parent_insert->bind_param("s", $parent_role_name);
        $stmt_parent_insert->execute();
        $new_id = $conn->insert_id;
        $stmt_parent_insert->close();
        $message .= "Parents role successfully added with ID: {$new_id}.";
    } else {
        $message .= "Parents role already exists. No change made.";
    }
    $stmt_parent_check->close();


    // 2. Commit transaction
    $conn->commit();
    $final_message = "✅ **SETUP COMPLETE:** " . $message;

} catch (Exception $e) {
    // 3. Rollback on error
    $conn->rollback();
    $final_message = "❌ **SETUP FAILED:** " . $e->getMessage() . " Database changes rolled back.";
    $error = true;
}

$conn->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>User Role Setup</title>
    <style>
        body { font-family: Arial, sans-serif; padding: 20px; background-color: #f4f4f4; }
        .container { max-width: 600px; margin: 0 auto; background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 0 10px rgba(0,0,0,0.1); }
        .alert { padding: 15px; margin-bottom: 20px; border: 1px solid transparent; border-radius: 4px; }
        .alert-success { color: #3c763d; background-color: #dff0d8; border-color: #d6e9c6; }
        .alert-danger { color: #a94442; background-color: #f2dede; border-color: #ebccd1; }
    </style>
</head>
<body>
    <div class="container">
        <h2>User Role Setup (One-Time Execution)</h2>
        <?php if (!empty($final_message)): ?>
            <div class="alert <?php echo $error ? 'alert-danger' : 'alert-success'; ?>">
                <?php echo nl2br(htmlspecialchars($final_message)); ?>
            </div>
        <?php endif; ?>
        <p>⚠️ **IMPORTANT:** Setup complete hone ke baad, security ke liye is `setup_roles.php` file ko server se **delete** kar dein ya iska access block kar dein.</p>
    </div>
</body>
</html>