<?php
// admin/student_marks_process.php - Marks Saving Logic (FINAL FIX for NOT NULL constraint)
session_start();
include_once('../config/db.php');

// Simple redirection helper function
function redirect($page, $params, $message, $type = 'success') {
    $param_string = http_build_query($params);
    $msg_type = ($type === 'success') ? 'msg' : 'err';
    
    header('Location: ' . $page . '.php?' . $param_string . '&' . $msg_type . '=' . urlencode($message));
    exit;
}

$conn = getDBConnection();
$action = $_POST['action'] ?? '';

// ====================================================================
// ACTION: SAVE STUDENT MARKS BATCH
// ====================================================================
if ($action === 'save_student_marks_batch' && $_SERVER['REQUEST_METHOD'] === 'POST') {

    // Required Form Data
    $exam_id = (int)($_POST['exam_id'] ?? 0);
    $class_id = (int)($_POST['class_id'] ?? 0);
    $student_id = (int)($_POST['student_id'] ?? 0);
    $subjects = $_POST['subjects'] ?? [];

    $redirect_params = [
        'exam_id' => $exam_id,
        'class_id' => $class_id,
        'student_id' => $student_id
    ];

    if ($exam_id === 0 || $class_id === 0 || $student_id === 0) {
        redirect('student_marks_entry', $redirect_params, "Error: Missing required parameters.", 'danger');
    }

    $conn->begin_transaction();
    $saved_count = 0;
    $error_occurred = false;

    // session_id table mein nahi hai, isliye isay query se hata diya gaya hai.
    $insert_query = "
        INSERT INTO exam_marks 
        (exam_id, class_id, student_id, subject_id, marks_obtained, is_absent) 
        VALUES (?, ?, ?, ?, ?, ?)
        ON DUPLICATE KEY UPDATE 
        marks_obtained = VALUES(marks_obtained), 
        is_absent = VALUES(is_absent)
    ";

    $stmt = $conn->prepare($insert_query);
    if (!$stmt) {
         redirect('student_marks_entry', $redirect_params, "Database Prepare Error: " . $conn->error, 'danger');
    }

    // Performance ke liye variables ko bind_param se pehle set karte hain
    $b_exam_id = $exam_id; 
    $b_class_id = $class_id; 
    $b_student_id = $student_id;
    $b_subject_id = 0; 
    $b_marks_value = 0.00; 
    $b_is_absent = 0; 

    // Bind parameters using the correct types (i i i i d i)
    // 'd' for marks_obtained (decimal/float) because it CANNOT be NULL.
    $stmt->bind_param("iiiidi", 
        $b_exam_id, $b_class_id, $b_student_id, $b_subject_id, 
        $b_marks_value, $b_is_absent
    );

    foreach ($subjects as $subject_id => $data) {
        
        // 1. Bound variables ki values set karein
        $b_subject_id = (int)$subject_id;
        $marks_input = $data['marks'] ?? ''; 
        $b_is_absent = (int)($data['absent'] ?? 0); 

        // 2. FIX: marks_obtained ko hamesha float value deni hai (NOT NULL ki wajah se).
        $b_marks_value = 0.00;
        
        // Agar student absent nahi hai AUR marks dale gaye hain, toh woh marks set karein
        if ($b_is_absent == 0 && $marks_input !== '') {
            $b_marks_value = (float)$marks_input;
        } 
        // Agar absent hai ya marks empty hain, toh 0.00 hi rahega (jo $b_marks_value ki initial value hai)

        // 3. Statement execute karein (ye bound variables ki current value istemal karega)
        if (!$stmt->execute()) {
            $error_occurred = true;
            $conn->rollback();
            $stmt->close();
            redirect('student_marks_entry', $redirect_params, "Marks save nahi hue. DB Error: " . $stmt->error . " (Subject ID: {$subject_id})", 'danger');
        }
        $saved_count++;
    }

    $stmt->close();

    if (!$error_occurred) {
        $conn->commit();
        redirect('student_marks_entry', $redirect_params, "Safaltamami se {$saved_count} subjects ke marks save ho chuke hain.", 'success');
    }

}
// ====================================================================
// FALLBACK / ERROR
// ====================================================================
else {
    header("Location: marks_entry.php?err=" . urlencode("Invalid action or request method."));
    exit;
}

$conn->close();
?>