<?php
// admin/system_process.php - FINAL FIXED VERSION (Redirect Issue Solved + Success Message Stay on Same Page)

session_start();
// Security Check - Admin access only
if (!isset($_SESSION['user_id']) || $_SESSION['user_role_id'] != 1) { 
    // Redirect to root login/index page
    header('Location: ../index.php'); 
    exit;
}

include_once('../config/db.php');
$conn = getDBConnection();

// ================================================================
// UNIVERSAL REDIRECT FUNCTION (FIXED PATH)
// ================================================================
function redirect($page, $msg = '', $type = 'success', $params = []) {
    $param_name = ($type === 'success') ? 'msg' : 'err';
    $url_params = http_build_query(array_merge($params, [$param_name => $msg]));

    // ✅ Fixed base path — change only if your admin folder path is different
    $base_url = '/learning/admin/';  
    header("Location: {$base_url}{$page}.php?{$url_params}");
    exit;
}

$action = $_POST['action'] ?? $_GET['action'] ?? '';

// ====================================================================
// ACTION 1: ADD NEW USER
// ====================================================================
if ($action === 'add_user' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = trim($_POST['username'] ?? '');
    $password = $_POST['password'] ?? '';
    $role_id = (int)$_POST['role_id'];
    $teacher_id = !empty($_POST['teacher_id']) ? (int)$_POST['teacher_id'] : NULL;
    
    if (empty($username) || empty($password) || $role_id === 0) {
        redirect('users', 'Zaruri fields adhoori hain.', 'danger');
    }

    $password_hash = password_hash($password, PASSWORD_DEFAULT);

    try {
        if ($teacher_id === NULL) {
            $sql = "INSERT INTO users (username, password_hash, role_id) VALUES (?, ?, ?)";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("ssi", $username, $password_hash, $role_id);
        } else {
            $sql = "INSERT INTO users (username, password_hash, role_id, teacher_id) VALUES (?, ?, ?, ?)";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("ssii", $username, $password_hash, $role_id, $teacher_id);
        }
        
        if (!$stmt->execute()) {
            throw new Exception("Error creating user: Username already exists.");
        }
        
        $stmt->close();
        redirect('users', 'User "' . $username . '" successfully created.', 'success');
        
    } catch (Exception $e) {
        if ($conn->errno == 1062) {
             redirect('users', 'User creation failed: Username or Teacher ID already in use.', 'danger');
        } else {
             redirect('users', $e->getMessage(), 'danger');
        }
    }
}

// ====================================================================
// ACTION 2: UPDATE GENERAL SETTINGS (WITH LOGO UPLOAD)
// ====================================================================
elseif ($action === 'update_settings' && $_SERVER['REQUEST_METHOD'] === 'POST') {

    $institute_name = trim($_POST['institute_name'] ?? '');
    $institute_address = trim($_POST['institute_address'] ?? '');
    $logo_path_to_save = null; 

    if (empty($institute_name)) {
        redirect('settings', 'School ka naam zaruri hai.', 'danger');
    }
    
    $conn->begin_transaction();

    try {
        // --- 1. Fetch current logo path ---
        $current_logo_query = $conn->prepare("SELECT setting_value FROM settings WHERE setting_key = 'institute.logo_path'");
        $current_logo_query->execute();
        $current_logo_result = $current_logo_query->get_result();
        $current_logo_data = $current_logo_result->fetch_assoc();
        $current_logo_path = $current_logo_data['setting_value'] ?? '';
        $current_logo_query->close();
        
        $logo_path_to_save = $current_logo_path; 

        // --- 2. Logo Upload Handling ---
        if (isset($_FILES['logo_file']) && $_FILES['logo_file']['error'] === UPLOAD_ERR_OK) {

            // Check file size (500 KB limit)
            if ($_FILES['logo_file']['size'] > 500000) { 
                 throw new Exception('Logo file size 500 KB se kam honi chahiye.');
            }

            $upload_dir = '../uploads/logos/'; 
            
            if (!is_dir($upload_dir)) {
                if (!mkdir($upload_dir, 0777, true)) {
                    throw new Exception('Uploads folder (uploads/logos/) bana nahi paaya. Permissions check karein.');
                }
            }

            $file_name = $_FILES['logo_file']['name'];
            $file_tmp = $_FILES['logo_file']['tmp_name'];
            $file_ext = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
            $allowed_ext = array('jpg', 'jpeg', 'png', 'gif');

            if (!in_array($file_ext, $allowed_ext)) {
                throw new Exception('Sirf JPG, JPEG, PNG, ya GIF files allowed hain.');
            }

            $new_file_name = 'institute_logo.' . $file_ext;
            $destination = $upload_dir . $new_file_name;

            if (move_uploaded_file($file_tmp, $destination)) {
                $logo_path_to_save = 'uploads/logos/' . $new_file_name;
                
                if (!empty($current_logo_path) && pathinfo($current_logo_path, PATHINFO_EXTENSION) != $file_ext) {
                    $old_file_to_delete = $upload_dir . 'institute_logo.' . pathinfo($current_logo_path, PATHINFO_EXTENSION);
                    if (file_exists($old_file_to_delete)) {
                         unlink($old_file_to_delete);
                    }
                }

            } else {
                throw new Exception('Logo upload karte waqt masla hua (Check server upload_max_filesize).');
            }
        }
        
        // --- 3. Database Updates ---
        $settings_to_update = [
            'institute.name' => $institute_name,
            'institute.address' => $institute_address,
            'institute.logo_path' => $logo_path_to_save
        ];

        foreach ($settings_to_update as $key => $value) {
            $stmt = $conn->prepare("
                INSERT INTO settings (setting_key, setting_value) 
                VALUES (?, ?)
                ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)
            ");
            $stmt->bind_param("ss", $key, $value);
            if (!$stmt->execute()) {
                 throw new Exception("Database mein '{$key}' save nahi ho saka.");
            }
            $stmt->close();
        }

        $conn->commit();
        redirect('settings', 'Settings aur Logo kamyabi se save ho chuke hain.', 'success');

    } catch (Exception $e) {
        $conn->rollback();
        redirect('settings', 'Settings update failed: ' . $e->getMessage(), 'danger');
    }
}

// ====================================================================
// ACTION 3: DATABASE BACKUP 
// ====================================================================
elseif ($action === 'create_backup' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    
    $backup_dir = '../storage/backups/';
    if (!is_dir($backup_dir)) { mkdir($backup_dir, 0777, true); }
    $filename = 'backup-' . date('Y-m-d-H-i-s') . '.sql';
    $filepath = $backup_dir . $filename;
    
    $tables = [];
    $table_res = $conn->query("SHOW TABLES");
    while($row = $table_res->fetch_row()) { $tables[] = $row[0]; }

    try {
        $handle = fopen($filepath, 'w');
        
        foreach ($tables as $table) {
            $create_res = $conn->query("SHOW CREATE TABLE " . $table);
            $create_row = $create_res->fetch_row();
            fwrite($handle, "\n\n-- Table Structure for table `{$table}`\n");
            fwrite($handle, $create_row[1] . ";\n");

            $data_res = $conn->query("SELECT * FROM " . $table);
            if ($data_res->num_rows > 0) {
                fwrite($handle, "\n-- Dumping data for table `{$table}`\n");
                while ($data_row = $data_res->fetch_assoc()) {
                    $cols = array_keys($data_row);
                    $values = array_map(function($v) use ($conn) {
                        if ($v === null) return 'NULL';
                        return "'" . $conn->real_escape_string($v) . "'";
                    }, array_values($data_row));

                    fwrite($handle, "INSERT INTO `{$table}` (`" . implode("`, `", $cols) . "`) VALUES (" . implode(", ", $values) . ");\n");
                }
            }
        }
        fclose($handle);
        redirect('backup', "Database backup successfully created: $filename", 'success');

    } catch (Exception $e) {
        if (isset($handle)) fclose($handle);
        redirect('backup', 'Backup failed: ' . $e->getMessage(), 'danger');
    }
}

// ====================================================================
// DEFAULT REDIRECT (Invalid action)
// ====================================================================
else {
    redirect('dashboard', 'Invalid action.', 'danger');
}

$conn->close();
?>
