-- final_schema.sql
-- School Management System - Complete Database Schema (Final Version)
-- Modified for 3-Table Class/Section Structure and Foreign Key Fixes

-- --------------------------------------------------------
-- Table: academic_sessions
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS academic_sessions (
    id INT(11) AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(50) NOT NULL,
    start_date DATE NOT NULL,
    end_date DATE NOT NULL,
    is_active BOOLEAN DEFAULT 0,
    UNIQUE KEY (name)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Seed Data (Default Session)
INSERT IGNORE INTO academic_sessions (id, name, start_date, end_date, is_active) VALUES
(1, '2024-2025', '2024-08-01', '2025-07-31', 1);


-- --------------------------------------------------------
-- Table: user_roles
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS user_roles (
    id INT(11) AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(50) NOT NULL UNIQUE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Seed Data (Roles)
INSERT IGNORE INTO user_roles (id, name) VALUES (1, 'Admin'), (2, 'Teacher'), (3, 'Librarian'), (4, 'Accountant');


-- --------------------------------------------------------
-- Table: teachers (Detailed teacher information)
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS teachers (
    id INT(11) AUTO_INCREMENT PRIMARY KEY,
    user_id INT(11) NULL, -- Link to users table
    emp_no VARCHAR(20) UNIQUE NULL,
    name VARCHAR(100) NOT NULL,
    designation VARCHAR(50) NULL,
    qualification VARCHAR(100) NULL,
    cnic VARCHAR(15) UNIQUE NULL,
    phone VARCHAR(15) NULL,
    joining_date DATE,
    status ENUM('active', 'inactive') DEFAULT 'active',
    photo_path VARCHAR(255) NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;


-- --------------------------------------------------------
-- Table: users (Login Credentials)
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS users (
    id INT(11) AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) NOT NULL UNIQUE,
    password_hash VARCHAR(255) NOT NULL, -- Correct column name for hashed password
    role_id INT(11) NOT NULL,
    teacher_id INT(11) NULL, 
    name VARCHAR(100) NULL,
    email VARCHAR(100) UNIQUE NULL,
    status ENUM('active', 'inactive') DEFAULT 'active',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (role_id) REFERENCES user_roles(id),
    FOREIGN KEY (teacher_id) REFERENCES teachers(id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;


-- --------------------------------------------------------
-- Table: settings (Institute Info)
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS settings (
    setting_key VARCHAR(100) PRIMARY KEY,
    setting_value TEXT,
    group_name VARCHAR(50)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Seed Data (Initial Settings)
INSERT IGNORE INTO settings (setting_key, setting_value, group_name) VALUES
('institute.name', 'Default School Name', 'institute'),
('institute.address', 'Default Address', 'institute'),
('institute.phone', '0000000000', 'institute'),
('academic.active_session_id', '1', 'academic');


-- --------------------------------------------------------
-- Table: academic_classes (Classes ka structure theek hai)
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS academic_classes (
    id INT(11) AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL UNIQUE,
    code VARCHAR(10) UNIQUE,
    order_no INT(11) DEFAULT 0
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- --------------------------------------------------------
-- Table: academic_sections (Sections table ko theek kiya aur rename kiya)
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS academic_sections (
    id INT(11) AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(50) NOT NULL UNIQUE, -- Section ka naam (e.g., A, B, C)
    status ENUM('active', 'inactive') DEFAULT 'active'
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Seed Data (Default Sections)
INSERT IGNORE INTO academic_sections (id, name, status) VALUES 
(1, 'A', 'active'), (2, 'B', 'active'), (3, 'C', 'active');


-- --------------------------------------------------------
-- Table: class_sections (Class aur Section ka jor)
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS class_sections (
    id INT(11) AUTO_INCREMENT PRIMARY KEY,
    session_id INT(11) NOT NULL,
    class_id INT(11) NOT NULL,
    section_id INT(11) NOT NULL,
    class_teacher_id INT(11) NULL,
    capacity INT(11) DEFAULT 40,
    FOREIGN KEY (session_id) REFERENCES academic_sessions(id),
    FOREIGN KEY (class_id) REFERENCES academic_classes(id),
    FOREIGN KEY (section_id) REFERENCES academic_sections(id),
    FOREIGN KEY (class_teacher_id) REFERENCES teachers(id),
    UNIQUE KEY (session_id, class_id, section_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;


-- --------------------------------------------------------
-- Table: students
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS students (
    id INT(11) AUTO_INCREMENT PRIMARY KEY,
    gr_no VARCHAR(50) UNIQUE NULL,
    first_name VARCHAR(50) NOT NULL,
    last_name VARCHAR(50) NULL,
    father_name VARCHAR(100) NOT NULL,
    dob DATE,
    gender ENUM('Male', 'Female', 'Other') NOT NULL,
    cnic_bform VARCHAR(20) UNIQUE NULL,
    phone VARCHAR(15),
    email VARCHAR(100) NULL,
    address TEXT,
    admission_date DATE NOT NULL,
    photo_path VARCHAR(255) NULL,
    status ENUM('active', 'inactive', 'left') DEFAULT 'active',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;


-- --------------------------------------------------------
-- Table: student_enrollments
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS student_enrollments (
    id INT(11) AUTO_INCREMENT PRIMARY KEY,
    student_id INT(11) NOT NULL,
    session_id INT(11) NOT NULL,
    class_id INT(11) NOT NULL,
    section_id INT(11) NOT NULL,
    roll_no VARCHAR(10) NULL,
    status ENUM('active', 'promoted', 'left') DEFAULT 'active',
    FOREIGN KEY (student_id) REFERENCES students(id),
    FOREIGN KEY (session_id) REFERENCES academic_sessions(id),
    FOREIGN KEY (class_id) REFERENCES academic_classes(id),
    FOREIGN KEY (section_id) REFERENCES academic_sections(id), -- FK theek kiya
    UNIQUE KEY (student_id, session_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;


-- --------------------------------------------------------
-- Table: fee_heads
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS fee_heads (
    id INT(11) AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL UNIQUE,
    type ENUM('fee', 'discount', 'fine') NOT NULL DEFAULT 'fee',
    is_recurring BOOLEAN DEFAULT 1
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Seed Data (Initial Heads)
INSERT IGNORE INTO fee_heads (name, type, is_recurring) VALUES 
('Tuition Fee', 'fee', 1),
('Admission Fee', 'fee', 0),
('Exam Fee', 'fee', 0),
('Transport Fee', 'fee', 1),
('Scholarship/Concession', 'discount', 1),
('Late Payment Fine', 'fine', 0);


-- --------------------------------------------------------
-- Table: fee_structures
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS fee_structures (
    id INT(11) AUTO_INCREMENT PRIMARY KEY,
    session_id INT(11) NOT NULL,
    class_id INT(11) NOT NULL,
    head_id INT(11) NOT NULL,
    amount DECIMAL(10, 2) NOT NULL,
    cycle ENUM('monthly', 'term', 'yearly', 'one-time') NOT NULL,
    FOREIGN KEY (session_id) REFERENCES academic_sessions(id),
    FOREIGN KEY (class_id) REFERENCES academic_classes(id),
    FOREIGN KEY (head_id) REFERENCES fee_heads(id),
    UNIQUE KEY (session_id, class_id, head_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;


-- --------------------------------------------------------
-- Table: student_fee_discounts
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS student_fee_discounts (
    id INT(11) AUTO_INCREMENT PRIMARY KEY,
    student_id INT(11) NOT NULL,
    head_id INT(11) NULL,
    type ENUM('fixed', 'percentage') NOT NULL,
    value DECIMAL(5, 2) NOT NULL,
    reason VARCHAR(255),
    is_active BOOLEAN DEFAULT 1,
    FOREIGN KEY (student_id) REFERENCES students(id),
    FOREIGN KEY (head_id) REFERENCES fee_heads(id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;


-- --------------------------------------------------------
-- Table: fee_invoices
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS fee_invoices (
    id INT(11) AUTO_INCREMENT PRIMARY KEY,
    student_id INT(11) NOT NULL,
    session_id INT(11) NOT NULL,
    voucher_no VARCHAR(50) UNIQUE NOT NULL,
    month_year CHAR(7) NOT NULL,
    issue_date DATE NOT NULL,
    due_date DATE NOT NULL,
    total_amount DECIMAL(10, 2) NOT NULL,
    net_payable DECIMAL(10, 2) NOT NULL,
    amount_paid DECIMAL(10, 2) DEFAULT 0.00,
    fine_charged DECIMAL(10, 2) DEFAULT 0.00,
    status ENUM('unpaid', 'partial', 'paid', 'cancelled') DEFAULT 'unpaid',
    FOREIGN KEY (student_id) REFERENCES students(id),
    FOREIGN KEY (session_id) REFERENCES academic_sessions(id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;


-- --------------------------------------------------------
-- Table: fee_invoice_items
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS fee_invoice_items (
    id INT(11) AUTO_INCREMENT PRIMARY KEY,
    invoice_id INT(11) NOT NULL,
    head_id INT(11) NOT NULL,
    description VARCHAR(255),
    amount DECIMAL(10, 2) NOT NULL,
    is_discount BOOLEAN DEFAULT 0,
    FOREIGN KEY (invoice_id) REFERENCES fee_invoices(id),
    FOREIGN KEY (head_id) REFERENCES fee_heads(id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;


-- --------------------------------------------------------
-- Table: fee_payments
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS fee_payments (
    id INT(11) AUTO_INCREMENT PRIMARY KEY,
    invoice_id INT(11) NOT NULL,
    payment_date DATETIME DEFAULT CURRENT_TIMESTAMP,
    amount DECIMAL(10, 2) NOT NULL,
    method ENUM('Cash', 'Bank', 'Online') DEFAULT 'Cash',
    ref_no VARCHAR(100) NULL,
    received_by INT(11) NULL, -- User ID
    note TEXT NULL,
    FOREIGN KEY (invoice_id) REFERENCES fee_invoices(id),
    FOREIGN KEY (received_by) REFERENCES users(id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;


-- --------------------------------------------------------
-- Table: subjects
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS subjects (
    id INT(11) AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL UNIQUE,
    code VARCHAR(20) UNIQUE,
    type ENUM('core', 'elective', 'optional') DEFAULT 'core'
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;


-- --------------------------------------------------------
-- Table: course_allocation
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS course_allocation (
    id INT(11) AUTO_INCREMENT PRIMARY KEY,
    session_id INT(11) NOT NULL,
    class_id INT(11) NOT NULL,
    section_id INT(11) NULL,
    subject_id INT(11) NOT NULL,
    teacher_id INT(11) NOT NULL,
    FOREIGN KEY (session_id) REFERENCES academic_sessions(id),
    FOREIGN KEY (class_id) REFERENCES academic_classes(id),
    FOREIGN KEY (section_id) REFERENCES academic_sections(id), -- FK theek kiya
    FOREIGN KEY (subject_id) REFERENCES subjects(id),
    FOREIGN KEY (teacher_id) REFERENCES teachers(id),
    UNIQUE KEY (session_id, class_id, section_id, subject_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;


-- --------------------------------------------------------
-- Table: daily_attendance_sheets
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS daily_attendance_sheets (
    id INT(11) AUTO_INCREMENT PRIMARY KEY,
    session_id INT(11) NOT NULL,
    class_id INT(11) NOT NULL,
    section_id INT(11) NOT NULL,
    attendance_date DATE NOT NULL,
    recorded_by_teacher_id INT(11) NULL,
    is_locked BOOLEAN DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (recorded_by_teacher_id) REFERENCES teachers(id),
    FOREIGN KEY (section_id) REFERENCES academic_sections(id), -- FK theek kiya
    UNIQUE KEY (session_id, class_id, section_id, attendance_date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;


-- --------------------------------------------------------
-- Table: daily_attendance_records
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS daily_attendance_records (
    id INT(11) AUTO_INCREMENT PRIMARY KEY,
    sheet_id INT(11) NOT NULL,
    student_id INT(11) NOT NULL,
    status ENUM('Present', 'Absent', 'Leave', 'Late') NOT NULL,
    note VARCHAR(255) NULL,
    FOREIGN KEY (sheet_id) REFERENCES daily_attendance_sheets(id) ON DELETE CASCADE,
    FOREIGN KEY (student_id) REFERENCES students(id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;


-- --------------------------------------------------------
-- Table: exams
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS exams (
    id INT(11) AUTO_INCREMENT PRIMARY KEY,
    session_id INT(11) NOT NULL,
    name VARCHAR(100) NOT NULL,
    type ENUM('term', 'final') DEFAULT 'term',
    start_date DATE NULL,
    end_date DATE NULL,
    status ENUM('active', 'completed') DEFAULT 'active',
    FOREIGN KEY (session_id) REFERENCES academic_sessions(id),
    UNIQUE KEY (session_id, name)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;


-- --------------------------------------------------------
-- Table: class_exam_structure
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS class_exam_structure (
    id INT(11) AUTO_INCREMENT PRIMARY KEY,
    exam_id INT(11) NOT NULL,
    class_id INT(11) NOT NULL,
    subject_id INT(11) NOT NULL,
    total_marks INT(4) NOT NULL,
    passing_marks INT(4) NOT NULL,
    FOREIGN KEY (exam_id) REFERENCES exams(id),
    FOREIGN KEY (class_id) REFERENCES academic_classes(id),
    FOREIGN KEY (subject_id) REFERENCES subjects(id),
    UNIQUE KEY (exam_id, class_id, subject_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;


-- --------------------------------------------------------
-- Table: exam_marks
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS exam_marks (
    id INT(11) AUTO_INCREMENT PRIMARY KEY,
    exam_id INT(11) NOT NULL,
    student_id INT(11) NOT NULL,
    subject_id INT(11) NOT NULL,
    marks_obtained DECIMAL(5, 2) NOT NULL,
    FOREIGN KEY (exam_id) REFERENCES exams(id),
    FOREIGN KEY (student_id) REFERENCES students(id),
    FOREIGN KEY (subject_id) REFERENCES subjects(id),
    UNIQUE KEY (exam_id, student_id, subject_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;


-- --------------------------------------------------------
-- Table: student_results
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS student_results (
    id INT(11) AUTO_INCREMENT PRIMARY KEY,
    exam_id INT(11) NOT NULL,
    student_id INT(11) NOT NULL,
    total_marks_obtained DECIMAL(6, 2) NOT NULL,
    total_max_marks INT(6) NOT NULL,
    percentage DECIMAL(5, 2) NOT NULL,
    grade VARCHAR(10) NULL,
    result_status ENUM('Pass', 'Fail', 'Promoted') NOT NULL,
    FOREIGN KEY (exam_id) REFERENCES exams(id),
    FOREIGN KEY (student_id) REFERENCES students(id),
    UNIQUE KEY (exam_id, student_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;