<?php
// FILE: parent/attendance.php - Student Attendance View for Parent
// --- CLEAN VERSION (DEBUG INFO REMOVED) ---
session_start();

// 1. Database and Includes
include_once('../config/db.php'); 

// Login check
if (!isset($_SESSION['user_id'])) { 
    header('Location: login.php?err=' . urlencode('Please log in to view attendance.')); 
    exit;
}

$conn = getDBConnection();

$parent_user_id = (int)$_SESSION['user_id'];
$student_id = (int)($_GET['student_id'] ?? 0);
$message = $_GET['msg'] ?? '';
$error = $_GET['err'] ?? '';

// Default to current month/year
$current_year = date('Y');
$current_month = date('m');

$selected_year = (int)($_GET['year'] ?? $current_year);
$selected_month = (int)($_GET['month'] ?? $current_month);

// Calculate the number of days in the selected month
$num_days_in_month = cal_days_in_month(CAL_GREGORIAN, $selected_month, $selected_year);
$first_day_of_month = date('Y-m-01', strtotime("{$selected_year}-{$selected_month}-01"));
$last_day_of_month = date('Y-m-t', strtotime("{$selected_year}-{$selected_month}-01"));

// --- Date Limit Fix ---
$days_to_display = $num_days_in_month;
$today_day = (int)date('d');
$today_month = (int)date('m');
$today_year = (int)date('Y');

if ($selected_year == $today_year && $selected_month == $today_month) {
    // Only show dates up to today's date
    $days_to_display = $today_day;
}

// --- Security Check --- 
$is_linked = 0;
$is_linked_q = $conn->prepare("SELECT COUNT(*) FROM parent_student_link WHERE parent_user_id = ? AND student_id = ?");
$is_linked_q->bind_param("ii", $parent_user_id, $student_id);
$is_linked_q->execute();
$is_linked_q->bind_result($is_linked);
$is_linked_q->fetch();
$is_linked_q->close();

if ($is_linked === 0 && $student_id != 0) {
    header('Location: dashboard.php?err=' . urlencode('Aapke paas is student ki attendance dekhne ka access nahi hai.'));
    exit;
}
if ($student_id === 0) {
    // Agar URL mein student ID nahi hai to N/A hi rahega
}

// --- Student Details Fetch ---
$student_details = ['name' => 'N/A', 'roll_no' => 'N/A'];
if ($student_id != 0) {
    // Student name fetch
    $student_q = $conn->prepare("
        SELECT s.first_name, s.last_name
        FROM students s
        WHERE s.id = ? LIMIT 1
    ");
    $student_q->bind_param("i", $student_id);
    $student_q->execute();
    $result = $student_q->get_result();
    if ($row = $result->fetch_assoc()) {
        $student_details['name'] = $row['first_name'] . ' ' . $row['last_name'];
    }
    $student_q->close();

    // Roll number fetch
    $roll_q = $conn->prepare("
        SELECT roll_no 
        FROM student_enrollments 
        WHERE student_id = ? 
        ORDER BY id DESC LIMIT 1
    ");
    $roll_q->bind_param("i", $student_id);
    $roll_q->execute();
    $result_roll = $roll_q->get_result();
    if ($row_roll = $result_roll->fetch_assoc()) {
        $student_details['roll_no'] = $row_roll['roll_no'];
    }
    $roll_q->close();
}

// --- ATTENDANCE DATA FETCH ---
$attendance_data = [];
$data_rows_found = 0;

if ($student_id != 0) {
    $attendance_q = $conn->prepare("
        SELECT 
            DATE(das.attendance_date) AS att_date, 
            dar.status,
            DAY(das.attendance_date) as day_num
        FROM daily_attendance_records dar
        JOIN daily_attendance_sheets das ON dar.sheet_id = das.id
        WHERE dar.student_id = ? 
        AND das.attendance_date >= ? AND das.attendance_date <= ?
        ORDER BY das.attendance_date
    ");
    $attendance_q->bind_param("iss", $student_id, $first_day_of_month, $last_day_of_month);
    $attendance_q->execute();
    $result = $attendance_q->get_result();

    while ($row = $result->fetch_assoc()) {
        $day_number = (int)$row['day_num'];
        $status = $row['status'];
        $attendance_data[$day_number] = ['status' => $status];
        $data_rows_found++;
    }
    $attendance_q->close();
}

// Month/Year options
$month_options = [];
for ($i = 0; $i < 12; $i++) {
    $timestamp = strtotime("first day of -$i month");
    $month_options[date('Y', $timestamp)][(int)date('m', $timestamp)] = date('F', $timestamp);
}

include_once('includes/header.php'); 
?>

<div class="page-content"> 
    <main>
        <div class="container-fluid px-4">
            <h1 class="mt-4">Student Attendance: <?php echo htmlspecialchars($student_details['name']); ?> (Roll: <?php echo htmlspecialchars($student_details['roll_no'] ?? 'N/A'); ?>)</h1>
            <ol class="breadcrumb mb-4">
                <li class="breadcrumb-item"><a href="dashboard.php">Dashboard</a></li>
                <li class="breadcrumb-item active">Attendance</li>
            </ol>
            
            <?php if ($error): ?>
                <div class="alert alert-danger"><?php echo $error; ?></div>
            <?php endif; ?>
            <?php if ($message): ?>
                <div class="alert alert-success"><?php echo $message; ?></div>
            <?php endif; ?>
            
            <?php if ($student_id != 0 && $data_rows_found === 0): ?>
                <div class="alert alert-warning">
                    <strong>No Attendance Records Found!</strong> Selected month mein is student ke liye koi attendance record nahi mila.
                </div>
            <?php endif; ?>
            
            <div class="card mb-4">
                <div class="card-header">
                    <i class="fas fa-calendar-alt me-1"></i>
                    Attendance Record for 
                    <strong><?php echo date('F Y', strtotime("{$selected_year}-{$selected_month}-01")); ?></strong>
                </div>
                <div class="card-body">
                    
                    <form method="GET" action="attendance.php" class="mb-4">
                        <input type="hidden" name="student_id" value="<?php echo $student_id; ?>">
                        <div class="row">
                            <div class="col-md-3">
                                <label for="month" class="form-label">Select Month</label>
                                <select name="month" id="month" class="form-control" required>
                                    <?php 
                                    foreach ($month_options as $year => $months) {
                                        echo "<optgroup label=\"{$year}\">";
                                        foreach ($months as $month_num => $month_name) {
                                            $selected = ($month_num == $selected_month && $year == $selected_year) ? 'selected' : '';
                                            echo "<option value=\"{$month_num}\" data-year=\"{$year}\" {$selected}>{$month_name} {$year}</option>";
                                        }
                                        echo "</optgroup>";
                                    }
                                    ?>
                                </select>
                                <input type="hidden" name="year" id="hidden_year" value="<?php echo $selected_year; ?>">
                            </div>
                            <div class="col-md-2 d-flex align-items-end">
                                <button type="submit" class="btn btn-primary btn-block"><i class="fas fa-filter"></i> Filter</button>
                            </div>
                        </div>
                    </form>

                    <div class="table-responsive">
                        <table class="table table-bordered table-striped" style="font-size: 0.9rem;">
                            <thead class="table-dark">
                                <tr>
                                    <th width="60%">Date (Full)</th>
                                    <th width="40%" class="text-center">Status</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php for ($d = 1; $d <= $days_to_display; $d++): 
                                    $date_str = date('Y-m-d', strtotime("{$selected_year}-{$selected_month}-{$d}"));
                                    $day_name = date('l', strtotime($date_str));
                                    $day_name_short = date('D', strtotime($date_str));
                                    
                                    // Fetch status for this day
                                    $status_code = 'N/M';
                                    
                                    if (isset($attendance_data[$d])) {
                                        $status_code = $attendance_data[$d]['status'];
                                    }

                                    // Determine class and display status
                                    $is_weekend = in_array($day_name_short, ['Sun']);
                                    $status_class = '';
                                    $display_status = '';

                                    if ($is_weekend) { 
                                        $status_class = 'bg-light text-muted'; 
                                        $display_status = 'Weekend';
                                    } 
                                    elseif ($status_code == 'P' || $status_code == 'Present') { 
                                        $status_class = 'bg-success text-white'; 
                                        $display_status = 'Present';
                                    } 
                                    elseif ($status_code == 'A' || $status_code == 'Absent') { 
                                        $status_class = 'bg-danger text-white'; 
                                        $display_status = 'Absent';
                                    }
                                    elseif ($status_code == 'L' || $status_code == 'Leave') { 
                                        $status_class = 'bg-warning text-dark'; 
                                        $display_status = 'Leave';
                                    }
                                    elseif ($status_code == 'H' || $status_code == 'Holiday') { 
                                        $status_class = 'bg-secondary text-white'; 
                                        $display_status = 'Holiday';
                                    }
                                    elseif ($status_code == 'Late') { 
                                        $status_class = 'bg-info text-white'; 
                                        $display_status = 'Late';
                                    }
                                    else { 
                                        $status_class = 'table-secondary'; 
                                        $display_status = 'Not Marked';
                                    } 
                                ?>
                                <tr>
                                    <td>
                                        <strong><?php echo date('d-M-Y', strtotime($date_str)); ?></strong> 
                                        <small class="text-muted">(<?php echo $day_name; ?>)</small>
                                    </td>
                                    <td class="<?php echo $status_class; ?> font-weight-bold text-center">
                                        <?php echo $display_status; ?>
                                    </td>
                                </tr>
                                <?php endfor; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <small class="text-muted mt-3">Legend: P=Present, A=Absent, L=Leave, H=Holiday, N/M=Not Marked</small>
                </div>
            </div>
        </div>
    </main>

    <?php include_once('includes/footer.php'); ?>
</div>

<script>
$(document).ready(function() {
    $('#month').on('change', function() {
        const selectedOption = $(this).find('option:selected');
        const year = selectedOption.data('year'); 
        $('#hidden_year').val(year);
    });
});
</script>

<?php 
if (isset($conn) && is_object($conn)) {
    $conn->close();
}
?>