<?php
// FILE: parent/result_card.php - Student Result Card View for Parent
session_start();

if (!isset($_SESSION['user_id'])) {
    header('Location: ../login.php');
    exit;
}

include_once('../config/db.php');
include_once('includes/header.php');
$conn = getDBConnection();

$parent_user_id = (int)$_SESSION['user_id'];
$student_id = (int)($_GET['student_id'] ?? 0);
$error = '';
$result_data = [];

// --- 1. Security Check ---
$is_linked_q = $conn->prepare("SELECT COUNT(*) FROM parent_student_link WHERE parent_user_id = ? AND student_id = ?");
$is_linked_q->bind_param("ii", $parent_user_id, $student_id);
$is_linked_q->execute();
$is_linked_q->bind_result($is_linked);
$is_linked_q->fetch();
$is_linked_q->close();

if ($student_id == 0 || $is_linked == 0) {
    $error = "Error: Invalid Student ID or the student is not linked to your account.";
    goto display_result;
}

// --- 2. Fetch Student Details ---
$details_query = "
    SELECT 
        s.first_name, s.last_name, 
        ac.name AS class_name,
        se.roll_no
    FROM students s
    LEFT JOIN student_enrollments se ON s.id = se.student_id
    LEFT JOIN academic_classes ac ON se.class_id = ac.id
    WHERE s.id = ?
    ORDER BY se.id DESC LIMIT 1
";
$student_details = [];
if ($stmt_det = $conn->prepare($details_query)) {
    $stmt_det->bind_param("i", $student_id);
    $stmt_det->execute();
    $student_details = $stmt_det->get_result()->fetch_assoc();
    $stmt_det->close();
}

if (!$student_details) {
    $error = "Error: Student details not found.";
    goto display_result;
}

$student_full_name = htmlspecialchars($student_details['first_name'] . ' ' . $student_details['last_name']);
$student_class = htmlspecialchars($student_details['class_name'] ?? 'N/A');
$roll_no = htmlspecialchars($student_details['roll_no'] ?? 'N/A');

// --- 3. Fetch Exam Results ---
try {
    // Sabhi exams fetch karo jo is student ke liye hain
    $exams_query = "
        SELECT DISTINCT 
            em.exam_id,
            e.name AS exam_name,
            e.type,
            e.start_date,
            e.end_date,
            e.status AS exam_status
        FROM exam_marks em
        JOIN exams e ON em.exam_id = e.id
        WHERE em.student_id = ?
        ORDER BY e.start_date DESC
    ";

    $exams_data = [];
    if ($stmt_exams = $conn->prepare($exams_query)) {
        $stmt_exams->bind_param("i", $student_id);
        $stmt_exams->execute();
        $exams_result = $stmt_exams->get_result();
        $exams_data = $exams_result->fetch_all(MYSQLI_ASSOC);
        $stmt_exams->close();
    }

    // Har exam ke liye details fetch karo
    foreach ($exams_data as $exam) {
        $exam_id = $exam['exam_id'];
        
        // Exam ke subjects aur marks fetch karo
        $marks_query = "
            SELECT 
                em.subject_id,
                sub.name AS subject_name,
                em.marks_obtained,
                em.is_absent,
                ces.total_marks AS max_marks,
                ces.passing_marks
            FROM exam_marks em
            JOIN subjects sub ON em.subject_id = sub.id
            LEFT JOIN class_exam_structure ces ON 
                ces.exam_id = em.exam_id AND 
                ces.class_id = em.class_id AND 
                ces.subject_id = em.subject_id
            WHERE em.student_id = ? AND em.exam_id = ?
            ORDER BY sub.name ASC
        ";

        $subjects_data = [];
        $total_obtained = 0;
        $total_max_marks = 0;

        if ($stmt_marks = $conn->prepare($marks_query)) {
            $stmt_marks->bind_param("ii", $student_id, $exam_id);
            $stmt_marks->execute();
            $marks_result = $stmt_marks->get_result();
            $subjects_data = $marks_result->fetch_all(MYSQLI_ASSOC);
            $stmt_marks->close();
        }

        // Totals calculate karo
        foreach ($subjects_data as $subject) {
            if (!$subject['is_absent'] && $subject['max_marks'] > 0) {
                $total_obtained += $subject['marks_obtained'];
                $total_max_marks += $subject['max_marks'];
            }
        }

        // Percentage aur grade calculate karo
        $percentage = 0;
        $grade = 'N/A';
        $result_status = 'Pending';

        if ($total_max_marks > 0) {
            $percentage = round(($total_obtained / $total_max_marks) * 100, 2);
            $grade = calculateGrade($percentage);
            $result_status = ($percentage >= 33) ? 'Pass' : 'Fail';
        }

        // Result data mein add karo
        $result_data[] = [
            'exam_id' => $exam_id,
            'exam_name' => $exam['exam_name'],
            'exam_type' => $exam['type'],
            'start_date' => $exam['start_date'],
            'end_date' => $exam['end_date'],
            'exam_status' => $exam['exam_status'],
            'subjects' => $subjects_data,
            'total_marks_obtained' => $total_obtained,
            'total_max_marks' => $total_max_marks,
            'percentage' => $percentage,
            'grade' => $grade,
            'result_status' => $result_status
        ];
    }

    if (empty($result_data)) {
        $error = "No exam results found for this student.";
    }

} catch (Exception $e) {
    $error = "Result system error: " . $e->getMessage();
}

// Grade calculation function
function calculateGrade($percentage) {
    if ($percentage >= 91) return 'A1';
    if ($percentage >= 81) return 'A2';
    if ($percentage >= 71) return 'B1';
    if ($percentage >= 61) return 'B2';
    if ($percentage >= 51) return 'C1';
    if ($percentage >= 41) return 'C2';
    if ($percentage >= 33) return 'D';
    return 'E';
}

display_result:
$conn->close();
?>

<style>
.result-card {
    border-radius: 12px;
    border: none;
    box-shadow: 0 4px 15px rgba(0,0,0,0.1);
    transition: all 0.3s ease;
    margin-bottom: 20px;
}

.result-card:hover {
    transform: translateY(-3px);
    box-shadow: 0 6px 20px rgba(0,0,0,0.15);
}

.grade-excellent { background: linear-gradient(135deg, #28a745, #20c997); }
.grade-good { background: linear-gradient(135deg, #17a2b8, #6f42c1); }
.grade-average { background: linear-gradient(135deg, #ffc107, #fd7e14); }
.grade-poor { background: linear-gradient(135deg, #dc3545, #e83e8c); }

.result-badge {
    border-radius: 20px;
    padding: 6px 15px;
    font-weight: 600;
    font-size: 0.8rem;
}

.percentage-circle {
    width: 80px;
    height: 80px;
    border-radius: 50%;
    background: conic-gradient(#28a745 var(--percentage), #e9ecef 0%);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-weight: bold;
    font-size: 1.2rem;
    margin: 0 auto;
    position: relative;
}

.percentage-circle::before {
    content: '';
    position: absolute;
    width: 70px;
    height: 70px;
    border-radius: 50%;
    background: white;
}

.percentage-circle span {
    position: relative;
    z-index: 1;
}

.exam-dates {
    font-size: 0.85rem;
    color: rgba(255,255,255,0.9);
}

.absent-badge {
    background-color: #dc3545;
    color: white;
    padding: 2px 8px;
    border-radius: 4px;
    font-size: 0.75rem;
}

@media (max-width: 768px) {
    .percentage-circle {
        width: 60px;
        height: 60px;
        font-size: 1rem;
    }
    
    .percentage-circle::before {
        width: 50px;
        height: 50px;
    }
}
</style>

<div class="page-content">
    <main>
        <div class="container-fluid px-4">
            <h1 class="mt-4">Student Result Card</h1>
            <ol class="breadcrumb mb-4">
                <li class="breadcrumb-item"><a href="dashboard.php">Dashboard</a></li>
                <li class="breadcrumb-item active">Result Card</li>
            </ol>

            <?php if (!empty($error)): ?>
                <div class="alert alert-danger">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    <?php echo $error; ?>
                </div>
            <?php endif; ?>

            <?php if ($student_id != 0 && $is_linked == 1 && empty($error)): ?>
                
                <!-- Student Info Card -->
                <div class="card result-card border-primary">
                    <div class="card-header bg-primary text-white">
                        <div class="row align-items-center">
                            <div class="col-md-8">
                                <h4 class="mb-0">
                                    <i class="fas fa-user-graduate me-2"></i>
                                    <?php echo $student_full_name; ?>
                                </h4>
                                <small>Roll No: <?php echo $roll_no; ?> | Class: <?php echo $student_class; ?></small>
                            </div>
                            <div class="col-md-4 text-md-end">
                                <span class="badge bg-light text-dark fs-6">Academic Record</span>
                            </div>
                        </div>
                    </div>
                </div>
                
                <?php if (!empty($result_data)): ?>
                
                <!-- Exam Results -->
                <?php foreach ($result_data as $result): 
                    $percentage = (float)$result['percentage'];
                    $grade_class = 'grade-poor';
                    if ($percentage >= 75) $grade_class = 'grade-excellent';
                    elseif ($percentage >= 50) $grade_class = 'grade-good';
                    elseif ($percentage >= 33) $grade_class = 'grade-average';
                    
                    $exam_type = $result['exam_type'] ?? 'Exam';
                    $start_date = !empty($result['start_date']) ? date('d M Y', strtotime($result['start_date'])) : '';
                    $end_date = !empty($result['end_date']) ? date('d M Y', strtotime($result['end_date'])) : '';
                ?>
                <div class="card result-card mt-4">
                    <div class="card-header text-white <?php echo $grade_class; ?>">
                        <div class="row align-items-center">
                            <div class="col-md-6">
                                <h5 class="mb-0">
                                    <i class="fas fa-file-alt me-2"></i>
                                    <?php echo htmlspecialchars($result['exam_name']); ?>
                                </h5>
                                <div class="exam-dates">
                                    <?php echo htmlspecialchars($exam_type); ?>
                                    <?php if (!empty($start_date)): ?>
                                        • <?php echo $start_date; ?>
                                        <?php if (!empty($end_date) && $start_date != $end_date): ?>
                                            - <?php echo $end_date; ?>
                                        <?php endif; ?>
                                    <?php endif; ?>
                                </div>
                            </div>
                            <div class="col-md-6 text-md-end">
                                <span class="result-badge bg-light text-dark">
                                    <?php echo htmlspecialchars($result['result_status']); ?>
                                </span>
                                <?php if (!empty($result['exam_status'])): ?>
                                <span class="result-badge bg-warning text-dark ms-1">
                                    <?php echo htmlspecialchars($result['exam_status']); ?>
                                </span>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    <div class="card-body">
                        <div class="row align-items-center mb-4">
                            <div class="col-md-3 text-center mb-3 mb-md-0">
                                <div class="percentage-circle" style="--percentage: <?php echo $percentage; ?>%">
                                    <span><?php echo $percentage; ?>%</span>
                                </div>
                                <div class="mt-2 fw-bold text-dark">Grade: <?php echo htmlspecialchars($result['grade']); ?></div>
                            </div>
                            <div class="col-md-9">
                                <div class="row text-center">
                                    <div class="col-4">
                                        <div class="border-end">
                                            <div class="text-muted small">Marks Obtained</div>
                                            <div class="h4 text-success fw-bold"><?php echo $result['total_marks_obtained']; ?></div>
                                        </div>
                                    </div>
                                    <div class="col-4">
                                        <div class="border-end">
                                            <div class="text-muted small">Max Marks</div>
                                            <div class="h4 text-primary fw-bold"><?php echo $result['total_max_marks']; ?></div>
                                        </div>
                                    </div>
                                    <div class="col-4">
                                        <div>
                                            <div class="text-muted small">Percentage</div>
                                            <div class="h4 fw-bold <?php echo ($percentage >= 33) ? 'text-success' : 'text-danger'; ?>">
                                                <?php echo $percentage; ?>%
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Subject-wise Marks -->
                        <?php if (!empty($result['subjects'])): ?>
                        <div class="mt-4">
                            <h6 class="border-bottom pb-2">
                                <i class="fas fa-book me-2"></i>Subject-wise Marks
                            </h6>
                            <div class="table-responsive">
                                <table class="table table-bordered table-hover table-sm">
                                    <thead class="table-light">
                                        <tr>
                                            <th>Subject</th>
                                            <th>Obtained Marks</th>
                                            <th>Max Marks</th>
                                            <th>Passing Marks</th>
                                            <th>Percentage</th>
                                            <th>Status</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($result['subjects'] as $subject): 
                                            $max_marks = $subject['max_marks'] ?? 0;
                                            $sub_percentage = ($max_marks > 0) ? round(($subject['marks_obtained'] / $max_marks) * 100, 2) : 0;
                                            $sub_status = $subject['is_absent'] ? 'Absent' : (($sub_percentage >= 33) ? 'Pass' : 'Fail');
                                        ?>
                                        <tr>
                                            <td class="fw-bold"><?php echo htmlspecialchars($subject['subject_name']); ?></td>
                                            <td>
                                                <?php if ($subject['is_absent']): ?>
                                                    <span class="absent-badge">Absent</span>
                                                <?php else: ?>
                                                    <?php echo $subject['marks_obtained']; ?>
                                                <?php endif; ?>
                                            </td>
                                            <td><?php echo $max_marks > 0 ? $max_marks : 'N/A'; ?></td>
                                            <td><?php echo $subject['passing_marks'] > 0 ? $subject['passing_marks'] : 'N/A'; ?></td>
                                            <td class="fw-bold <?php echo ($sub_status == 'Pass') ? 'text-success' : (($sub_status == 'Absent') ? 'text-warning' : 'text-danger'); ?>">
                                                <?php echo $subject['is_absent'] ? 'N/A' : $sub_percentage . '%'; ?>
                                            </td>
                                            <td>
                                                <span class="badge <?php 
                                                    echo ($sub_status == 'Pass') ? 'bg-success' : 
                                                         (($sub_status == 'Absent') ? 'bg-warning' : 'bg-danger'); 
                                                ?>">
                                                    <?php echo $sub_status; ?>
                                                </span>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
                <?php endforeach; ?>

                <?php else: ?>
                    <div class="card result-card mt-4">
                        <div class="card-body text-center py-5">
                            <i class="fas fa-file-invoice fa-3x text-muted mb-3"></i>
                            <h5 class="text-muted">No Results Available</h5>
                            <p class="text-muted">No exam results found for this student yet.</p>
                        </div>
                    </div>
                <?php endif; ?>

            <?php elseif ($student_id == 0): ?>
                 <div class="alert alert-info mt-4">
                     <i class="fas fa-info-circle me-2"></i>
                     Please go back to <a href="dashboard.php" class="alert-link">Dashboard</a> and select a student.
                 </div>
            <?php endif; ?>
        </div>
    </main>

    <?php include_once('includes/footer.php'); ?>
</div>