<?php
// FILE: student_planner_view.php - Student Class Planner View
session_start();

// Simplified login check - sirf check karo ke user logged in hai ya nahi
if (!isset($_SESSION['user_id'])) {
    header('Location: ../login.php');
    exit;
}

include_once('../config/db.php');
include_once('includes/header.php');
$conn = getDBConnection();

$parent_user_id = (int)$_SESSION['user_id'];
$student_id = (int)($_GET['student_id'] ?? 0);
$error = '';
$subjects = [];
$planner_data = [];
$dates = [];

// --- 1. Security Check ---
$is_linked_q = $conn->prepare("SELECT COUNT(*) FROM parent_student_link WHERE parent_user_id = ? AND student_id = ?");
$is_linked_q->bind_param("ii", $parent_user_id, $student_id);
$is_linked_q->execute();
$is_linked_q->bind_result($is_linked);
$is_linked_q->fetch();
$is_linked_q->close();

if ($student_id == 0 || $is_linked == 0) {
    $error = "Error: Invalid Student ID or the student is not linked to your account.";
    goto display_planner;
}

// --- 2. Fetch Student Details ---
$student_q = $conn->query("SELECT * FROM students WHERE id = $student_id LIMIT 1");
$student = $student_q->fetch_assoc();

if (!$student) {
    $error = "Error: Student details not found.";
    goto display_planner;
}

// Check karo ke class_id column exists karta hai ya nahi
if (!isset($student['class_id'])) {
    // Agar class_id nahi hai toh student_enrollments table check karo
    $enrollment_q = $conn->query("SELECT class_id, section_id FROM student_enrollments WHERE student_id = $student_id ORDER BY id DESC LIMIT 1");
    $enrollment = $enrollment_q->fetch_assoc();
    
    if ($enrollment) {
        $class_id = $enrollment['class_id'];
        $section_id = $enrollment['section_id'];
    } else {
        $error = "Error: Student class information not found.";
        goto display_planner;
    }
} else {
    $class_id = $student['class_id'];
    $section_id = $student['section_id'];
}

// --- 3. Get Class and Section Names ---
$class_info = $conn->query("SELECT name FROM academic_classes WHERE id = $class_id")->fetch_assoc();
$section_info = $conn->query("SELECT name FROM academic_sections WHERE id = $section_id")->fetch_assoc();

$class_name = $class_info['name'] ?? 'N/A';
$section_name = $section_info['name'] ?? 'N/A';

// --- 4. Get Current Week Dates ---
$start_date = date('Y-m-d', strtotime('monday this week'));
$end_date = date('Y-m-d', strtotime('saturday this week'));

// --- 5. Fetch Subjects for this Class ---
$subject_q = $conn->query("
    SELECT s.id, s.name 
    FROM subjects s 
    JOIN course_allocation ca ON s.id = ca.subject_id 
    WHERE ca.class_id = $class_id 
    ORDER BY s.name
");

while ($row = $subject_q->fetch_assoc()) {
    $subjects[] = $row;
}

// --- 6. Fetch Planner Data ---
$planner_q = $conn->query("
    SELECT planner_date, subject_id, cw_content, hw_content 
    FROM daily_planner 
    WHERE class_id = $class_id 
    AND section_id = $section_id 
    AND planner_date BETWEEN '$start_date' AND '$end_date'
");

while ($row = $planner_q->fetch_assoc()) {
    $key = $row['planner_date'] . '_' . $row['subject_id'];
    $planner_data[$key] = [
        'cw' => $row['cw_content'],
        'hw' => $row['hw_content']
    ];
}

// --- 7. Generate Dates for the Week ---
$current = strtotime($start_date);
$end = strtotime($end_date);

while ($current <= $end) {
    $dates[] = date('Y-m-d', $current);
    $current = strtotime('+1 day', $current);
}

display_planner:
$conn->close();
?>

<div class="page-content">
    <main>
        <div class="container-fluid px-4">
            <h1 class="mt-4">My Weekly Study Plan</h1>
            <ol class="breadcrumb mb-4">
                <li class="breadcrumb-item"><a href="dashboard.php">Dashboard</a></li>
                <li class="breadcrumb-item active">Study Plan</li>
            </ol>

            <?php if (!empty($error)): ?>
                <div class="alert alert-danger">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    <?php echo $error; ?>
                </div>
            <?php endif; ?>

            <?php if ($student_id != 0 && $is_linked == 1 && empty($error)): ?>
                
                <div class="card mb-4">
                    <div class="card-header">
                        <i class="fas fa-user-graduate me-1"></i>
                        Student: <strong><?php echo $student['first_name'] . ' ' . $student['last_name']; ?></strong> | 
                        Class: <strong><?php echo $class_name; ?></strong> | 
                        Section: <strong><?php echo $section_name; ?></strong>
                    </div>
                </div>
                
                <?php if (!empty($subjects)): ?>
                <div class="card mb-4">
                    <div class="card-header text-white" style="background-color: #6c757d;">
                        <i class="fas fa-calendar-alt me-1"></i>
                        Week: <?php echo date('d M Y', strtotime($start_date)); ?> - <?php echo date('d M Y', strtotime($end_date)); ?>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-bordered table-striped">
                                <thead class="table-dark">
                                    <tr>
                                        <th style="width: 120px;">Date / Day</th>
                                        <?php foreach ($subjects as $subject): ?>
                                            <th><?php echo htmlspecialchars($subject['name']); ?></th>
                                        <?php endforeach; ?>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($dates as $date): 
                                        $day_name = date('l', strtotime($date));
                                        // CHANGED: Saturday holiday removed, Sunday holiday implemented
                                        $is_weekend = (date('D', strtotime($date)) === 'Sun');
                                        $row_class = $is_weekend ? 'table-warning' : '';
                                    ?>
                                    <tr class="<?php echo $row_class; ?>">
                                        <td class="text-center">
                                            <strong><?php echo date('d M', strtotime($date)); ?></strong><br>
                                            <small class="text-primary"><?php echo $day_name; ?></small>
                                        </td>
                                        <?php foreach ($subjects as $subject): 
                                            $key = $date . '_' . $subject['id'];
                                            $cw_content = $planner_data[$key]['cw'] ?? '';
                                            $hw_content = $planner_data[$key]['hw'] ?? '';
                                        ?>
                                        <td>
                                            <?php if (!$is_weekend): ?>
                                                <div class="mb-2">
                                                    <small class="text-success font-weight-bold">C.W.</small>
                                                    <div class="border rounded p-2 bg-light">
                                                        <?php echo !empty($cw_content) ? nl2br(htmlspecialchars($cw_content)) : '<span class="text-muted">No class work</span>'; ?>
                                                    </div>
                                                </div>
                                                <div>
                                                    <small class="text-danger font-weight-bold">H.W.</small>
                                                    <div class="border rounded p-2 bg-light">
                                                        <?php echo !empty($hw_content) ? nl2br(htmlspecialchars($hw_content)) : '<span class="text-muted">No homework</span>'; ?>
                                                    </div>
                                                </div>
                                            <?php else: ?>
                                                <span class="text-muted">Weekend</span>
                                            <?php endif; ?>
                                        </td>
                                        <?php endforeach; ?>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
                <?php else: ?>
                    <div class="alert alert-info text-center">
                        <i class="fas fa-info-circle me-2"></i>
                        Is class ke liye koi subjects assign nahi kiye gaye hain.
                    </div>
                <?php endif; ?>

            <?php elseif ($student_id == 0): ?>
                 <div class="alert alert-info mt-4">
                     <i class="fas fa-info-circle me-2"></i>
                     Kripya <a href="dashboard.php" class="alert-link">Dashboard</a> par wapas jaakar bachcha select karein.
                 </div>
            <?php endif; ?>
        </div>
    </main>

    <?php include_once('includes/footer.php'); ?>
</div>